/*
 *
 * Copyright 2017-2020 by Kappich Systemberatung, Aachen
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.ars.ars.mgmt.tasks;

import de.bsvrz.ars.ars.persistence.ContainerFileHandle;
import de.bsvrz.ars.ars.persistence.writer.SerializableDataset;
import de.bsvrz.sys.funclib.debug.Debug;
import de.bsvrz.sys.funclib.losb.util.ByteIO;

import static de.bsvrz.ars.ars.persistence.ContainerFile.*;

/**
 * Klasse, die Archivdaten serialisiert um diese in Containerdateien zu schreiben.
 *
 * @author Kappich Systemberatung
 * @see ContainerFileHandle#appendSerializedData(ArchiveDataSerializer) 
 */
public class ArchiveDataSerializer {

	private static final Debug _debug = Debug.getLogger();
	
	/**
	 * Referenz auf den tatsaechlichen, von {@link #serializeData(SerializableDataset)} verwendeten Puffer, der von {@link #writeBuf}
	 * abweicht, falls dieser zu klein ist). Notwendig, da Daten vorserialisiert werden müssen, um deren Länge zu bestimmen.
	 */
	private byte[] actualWriteBuf;

	/**
	 * Länge des von {@link #serializeData(SerializableDataset)} serialisierten Datensatzes.
	 */
	private int totalWriteDataSize;

	/** Puffer zum Kopieren von Dateien oder Einlesen von Strings und Zahlen. Wird nur von ArchivTasks benutzt (und damit automatisch angelegt) */
	private final byte[] writeBuf = new byte[BUFFER_SIZE];

	/**
	 * Liefert den übergebenen Puffer falls dessen Größe ausreichend ist, andernfalls einen neu angelegten. Zweck: sowenig Arbeit für die Speicherverwaltung
	 * wie möglich.
	 *
	 * @param defaultBuffer Vorhandener Puffer
	 * @param desiredSize   Gewuenschte PufferGröße
	 * @return Byte-Array-Puffer
	 */
	private static byte[] getBuf(byte[] defaultBuffer, int desiredSize) {
		return (desiredSize <= defaultBuffer.length) ? defaultBuffer : new byte[desiredSize];
	}


	/**
	 * Schreibt einen Datensatz in den {@link #actualWriteBuf}
	 *
	 * @param dataset Datensatz
	 * @return Gesamtlänge in Bytes
	 */
	public int serializeData(SerializableDataset dataset) {
		long arcTime = dataset.archiveTime();
		long dataTime = dataset.dataTime();
		long dataIdx = dataset.dataIndex();

		if ((dataTime & 0xFFFFFFFFFFFFL) != dataTime) {
			_debug.warning("Ungültige Datenzeit: " + dataTime + ". Datensatz wird stattdessen mit Datenzeit 0 archiviert: " + dataset);
			dataTime = 0;
		}

		byte[] serData = dataset.serializedData();
		int origLength = dataset.uncompressedSize();
		boolean isAlreadyCompressed = origLength != NOT_COMPRESSED;

		byte[] dest = actualWriteBuf = getBuf(writeBuf, getTotalDataOverhead() + serData.length);

		int p = ByteIO.INT4B_LEN;                                    // [1. Platzhalter f. Länge]
		p += ByteIO.writeUnsignedLong6Byte(dest, p, arcTime);        // 2. Archivzeit
		p += ByteIO.writeUnsignedLong6Byte(dest, p, dataTime);       // 3. Datenzeit
		p += ByteIO.writeSignedLong8Byte(dest, p, dataIdx);          // 4. Datenindex
		int chkSumPos = p;                                           // [5. Platzhalter für Prüfsumme]
		p += ByteIO.INT4B_LEN;

		// 7. Daten
		if (!isAlreadyCompressed) {
			p += ByteIO.writeSignedInt4Bytes(dest, p, NOT_COMPRESSED);    // 6. Kompressions-Info (=0)
		} else {
			p += ByteIO.writeSignedInt4Bytes(dest, p, origLength);        // 6. Kompressions-Info
		}

		p += ByteIO.writeBytes(dest, p, serData);                     // 7. Daten

		totalWriteDataSize = getTotalDataOverhead() + serData.length; // tatsaechliche Länge über alles
		
		int lengthBytes = totalWriteDataSize - DATALEN_LEN;
		int checkSum = (int) (lengthBytes + arcTime + dataTime + dataIdx);

		ByteIO.writeSignedInt4Bytes(dest, 0, lengthBytes);      // 1. Länge
		ByteIO.writeSignedInt4Bytes(dest, chkSumPos, checkSum); // 5. Prüfsumme
		ByteIO.writeSeparator(dest, p);                         // 8. Separator
		return totalWriteDataSize;
	}


	/**
	 * Gibt den Schreibpuffer zurück
	 *
	 * @return den Schreibpuffer
	 */
	public byte[] getActualWriteBuf() {
		return actualWriteBuf;
	}

	/**
	 * Gibt die benutzte Länge des Schreibpuffers zurück
	 *
	 * @return benutzte Länge in Bytes
	 */
	public int getTotalWriteDataSize() {
		return totalWriteDataSize;
	}

}
