/*
 *
 * Copyright 2005-2008 by beck et al. projects GmbH, Munich
 * Copyright 2009-2020 by Kappich Systemberatung, Aachen
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.ars.ars.mgmt.tasks;

import de.bsvrz.ars.ars.mgmt.tasks.base.TaskManager;
import de.bsvrz.ars.ars.persistence.ContainerDataResult;
import de.bsvrz.ars.ars.persistence.IdDataIdentification;
import de.bsvrz.ars.ars.persistence.iter.DataIterator;
import de.bsvrz.ars.ars.persistence.iter.QueryDataSequence;
import de.bsvrz.dav.daf.main.archive.ArchiveDataSpecification;
import de.bsvrz.dav.daf.main.archive.ArchiveTimeSpecification;
import de.bsvrz.dav.daf.main.archive.TimingType;
import de.bsvrz.sys.funclib.losb.datk.MessageType;
import de.bsvrz.sys.funclib.losb.util.Util;

import java.io.IOException;

/**
 * Task zur Bearbeitung von Archiv-Informationsanfragen.
 *
 * @author beck et al. projects GmbH
 * @author Alexander Schmidt
 * @version $Revision$ / $Date$ / ($Author$)
 */
public class ArchiveInfoQueryTask extends QueryTask {

	private static final boolean GAP = true, NO_GAP = false, NO_DIR_ACCESS = false;

	private static final int NO_MEDIUM_ID = -1;

	private ArchiveDataSpecification[] adsList;

	private int entryCounter;

	/**
	 * Erstellt eine Instanz
	 *
	 * @param tMgr MultiTaskManager, der die verschiedenen Archiv-Info-Tasks verwaltet.
	 */
	public ArchiveInfoQueryTask(MultiTaskManager tMgr) {
		super(tMgr.getArchMgr(), tMgr);
	}

	private void analyze() throws Exception {
		createQueryData();
		adsList = parseArchiveDataSpec(deserializer, getArchMgr().getDataModel());
	}


	/**
	 * Beantwortet die Archivinformationsanfragen. Die einzelnen Archivinformationsanfragen aus {@link #adsList} werden bearbeitet und beantwortet.
	 *
	 * @throws IOException Allgemeiner IO-Fehler
	 */
	private void sendResponse() throws IOException {
		bosResult.reset();            // genuegt, man muss nicht jedesmal einen neuen Serialisierer anlegen

		serializer.writeByte(1);        // erfolgreich (nehmen wir erstmal an...)
		serializer.writeInt(0);            // Größe Liste (wird später besetzt)
		entryCounter = 0;

		int adsCounter = 0;

		for (ArchiveDataSpecification ads : adsList) {
			suspendTaskIfNecessary();        //Task anhalten, falls Archiv überlastet.
			if (shouldTerminate()) break;

			IdDataIdentification dataIdentification = new IdDataIdentification(ads);
			ArchiveTimeSpecification ats = ads.getTimeSpec();

			try {
				QueryDataSequence sequence = new QueryDataSequence(getPersistenceManager(), ads.getDataKinds(), ats, ads.getSortOrder(),
						dataIdentification);

				try (DataIterator iterator = sequence.iterator()) {

					ContainerDataResult result = new ContainerDataResult();

					TimingType tt = ats.getTimingType();

					long curDIdx = -1, curATime = -1, curDTime = -1;    // Werte des aktuellen Durchlaufs
					long lastDIdx = -1, lastDTime = -1, lastATime = -1; // Werte des vorherigen Durchlaufs
					long lbDIdx = -1, lbATime = -1, lbDTime = -1;       // Untere Grenze der Ausgabezeile
					// letzte und aktuelle (wg. Plattenzugriff) medienID
					boolean curDirAccess = true;

					// Hauptschleife:
					while(!iterator.isEmpty()) {
						if(shouldTerminate()) throw new IllegalStateException("Das Archivsystem wurde beendet.");

						iterator.peek(result);

						curDIdx = result.getDataIndex();
						curATime = result.getArchiveTime();
						curDTime = result.getDataTime();

						if(lbDIdx == -1) {
							// Untergrenzen initialisieren
							lbDIdx = curDIdx;
							lbATime = curATime;
							lbDTime = curDTime;
						}

						boolean blockAlreadyWritten = false;

						if(isGap(result, lastDIdx)) {
							if(!blockAlreadyWritten) {
								// Zeile für Zustand bis zur Lücke (falls nicht schon wg. Containerwechsel ausgegeben):
								addEntry(ttVal(tt, lbDIdx, lbATime, lbDTime), ttVal(tt, lastDIdx, lastATime, lastDTime), tt, NO_GAP, curDirAccess, -1, adsCounter);
							}

							// Zeile für die Lücke:
							addEntry(ttVal(tt, lastDIdx, lastATime, lastDTime), ttVal(tt, curDIdx, curATime, curDTime), tt, GAP, NO_DIR_ACCESS, NO_MEDIUM_ID, adsCounter);

							// nächste Zeile geht beim Lückenende los:
							lbDIdx = curDIdx;
							lbATime = curATime;
							lbDTime = curDTime;
						}
						lastDIdx = curDIdx;
						lastATime = curATime;
						lastDTime = curDTime;

						iterator.remove();
					}

					if(lbDIdx != -1) {
						// letzten Eintrag schreiben, aber nur falls es überhaupt Daten gibt
						addEntry(ttVal(tt, lbDIdx, lbATime, lbDTime), ttVal(tt, curDIdx, curATime, curDTime), tt, NO_GAP, curDirAccess, -1, adsCounter);
					}
				}
			}
			catch(Exception e) {
				_debug.error("Fehler beim Bearbeiten der Archivinformationsanfrage", e);
				printError("Fehler beim Bearbeiten der Archivinformationsanfrage: " + e.getMessage());
				sendResultData(bosResult.toByteArray(), MessageType.QUERY_INFO_RESULT);
				return;
			}
			adsCounter++;
		}
		sendResultData(insertNumOfEntries(entryCounter), MessageType.QUERY_INFO_RESULT);
	}

	private static boolean isGap(final ContainerDataResult result, final long lastDIdx) {
		return lastDIdx != -1 && Util.dIdxNoModBits(result.getDataIndex()) - Util.dIdxNoModBits(lastDIdx) > 1 && !result.getDataKind().isDelayed();
	}

	private static long ttVal(TimingType tt, long dIdx, long aTime, long dTime) {
		return tt.equals(TimingType.DATA_TIME) ? dTime : tt.equals(TimingType.ARCHIVE_TIME) ? aTime : dIdx;
	}

	private byte[] insertNumOfEntries(int numOfEntries) throws IOException {
		byte[] resultBytes = bosResult.toByteArray();
		bosResult.reset();
		serializer.writeInt(numOfEntries);
		System.arraycopy(bosResult.toByteArray(), 0, resultBytes, 1, 4);
		return resultBytes;
	}

	private void printError(String msg) throws IOException {
		bosResult.reset();
		serializer.writeByte(0);    // Fehler
		serializer.writeString(msg);
	}

	private void addEntry(
			final long from,
			final long to, 
			TimingType tt,
			boolean gap,
			boolean direct,
			int volID,
			int idx) throws IOException {
		serializer.writeLong(from);      // 3.
		serializer.writeLong(to);      // 4.
		serializer.writeByte(tt.equals(TimingType.DATA_TIME) ? 1 : tt.equals(TimingType.ARCHIVE_TIME) ? 2 : 3);    // 5. TT
		serializer.writeByte(gap ? 1 : 0);        // 6. Gap
		serializer.writeByte(direct ? 1 : 0);    // 7. Access
		serializer.writeInt(volID);                // 8. volID
		serializer.writeInt(idx);                // 9.
		entryCounter++;
	}

	@Override
	public void work() {
		//logger.finer(getName() + " Erhält Task: ", dataset);
		init();
		try {
			analyze();
			subscribeSender();
			// System.out.println("ArchiveInfoQueryTask.work");

			final SendingState state;
			state = waitForSendControl();

			if(state == SendingState.SEND) {
				TaskManager.run("Archivinfoanfrage", (tpi) -> sendResponse());
			}
		}
		catch(InterruptedException e) {
			_debug.finer(getName() + " Task durch Interrupt beendet.");
		}
		catch(Exception e) {
			// dataset wird vor Aufruf der Debug-Methode in einen String konvertiert, um verschachtelte Debug-Aufrufe mit Deadlock-Gefahr zu vermeiden
			_debug.error(getName() + " Fehler beim Bearbeiten der Archiv-Informationsanfrage: " + e.getMessage() + ", Anfrage: " + resultData.toString());
		}
		finally {
			unsubscribeSender(); // Als Sender der Archiv-Antwort abmelden
			adsList = null;
			bosResult.reset();
		}
	}


}
