/*
 *
 * Copyright 2005-2008 by beck et al. projects GmbH, Munich
 * Copyright 2009-2020 by Kappich Systemberatung, Aachen
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.ars.ars.mgmt.tasks;

import de.bsvrz.ars.ars.mgmt.ArchiveManager;
import de.bsvrz.ars.ars.mgmt.InQueuesMgr.DataReceiver;
import de.bsvrz.ars.ars.persistence.writer.ArchiveTask;
import de.bsvrz.dav.daf.main.ResultData;
import de.bsvrz.sys.funclib.debug.Debug;
import de.bsvrz.sys.funclib.losb.datk.ArchiveSettings;
import de.bsvrz.sys.funclib.losb.datk.ContainerSettings;
import de.bsvrz.sys.funclib.losb.kernsoftware.ConnectionManager;

import java.util.ArrayList;
import java.util.List;

/**
 * Task zur Umsetzung der Archiveinstellungen. Dies betrifft die Attributgruppen "atg.archivEinstellung" und "atg.archivContainer". Das Ende eines
 * Übernahmezyklus eines jeden dieser Parameter kann man mit {@link #createSettingLstnr(int)} abwarten.
 *
 * @author beck et al. projects GmbH
 * @author Alexander Schmidt
 * @version $Revision$ / $Date$ / ($Author$)
 */
public class ArchiveSettingsTask extends SingleTask {

	/** Unter diesem Aspekt werden die Einstellungen abonniert */
	private static final String SETTINGS_ASPECT = "asp.parameterSoll";

	/** Minimumwerte für {@link ContainerSettings#stdCloseConditions}: 1 Datensatz pro Container */
	public static final int MIN_CONTAINER_MAXDS = 1;

	/** Minimumwerte für {@link ContainerSettings#stdCloseConditions}: 4096 (entspricht einer BlockGröße) */
	public static final int MIN_CONTAINER_MAXSIZE = 4096;

	/** Minimumwerte für {@link ContainerSettings#stdCloseConditions}: 1 Minute */
	public static final int MIN_CONTAINER_MAXTIME = 60;

	// **************************************************************************************
	// anzumeldende Datenidentifikationen:

	public static final int ARCHIVEINST = 0;

	public static final int ARCHIVCONTAINER = 1;

	/** Diese Datenidentifikationen werden angemeldet */
	private static final String[][] SUBSCR_PIDS = {
			{ArchiveSettings.ATTR_GROUP, SETTINGS_ASPECT}, {ContainerSettings.ATTR_GROUP, SETTINGS_ASPECT}
	};

	private static final long[] SUBSCR_IDS = new long[SUBSCR_PIDS.length];

	@SuppressWarnings("unchecked")
	private final List<ArSSettingListener>[] settingsListener = (List<ArSSettingListener>[])new List[SUBSCR_PIDS.length];
	// **************************************************************************************

	/**
	 * Erstellt eine neue Instanz
	 * @param archMgr {@link ArchiveManager}
	 */
	public ArchiveSettingsTask(ArchiveManager archMgr) {
		super(archMgr);
		for(int i = 0; i < settingsListener.length; i++) {
			settingsListener[i] = new ArrayList<>();
		}
	}

	/**
	 * Fuehrt alle notwendigen Anmeldungen durch.
	 *
	 * @param archMgr  Archiv-Manager
	 * @param receiver Empfaengerobjekt
	 */
	public static void subscribeObjects(ArchiveManager archMgr, DataReceiver receiver) {
		for(int i = 0; i < SUBSCR_IDS.length; i++) {
			try {
				SUBSCR_IDS[i] = archMgr.getAtg(SUBSCR_PIDS[i][0]).getId();
				ConnectionManager.subscrRecNormal(archMgr.getDavCon(), receiver, archMgr.getArchiveObject(), SUBSCR_PIDS[i][0], SUBSCR_PIDS[i][1]);
				Debug.getLogger().fine("Archiveinstellungen angemeldet: " + SUBSCR_PIDS[i][0] + "/" + SUBSCR_PIDS[i][1]);
			}
			catch(Exception e) {
				Debug.getLogger().error(
						"Anmeldung fuer KV-Objekt fehlgeschlagen: " + SUBSCR_PIDS[i][0] + "/" + SUBSCR_PIDS[i][1] + Debug.NEWLINE + e.getMessage()
				);
			}
		}
	}


	public ArSSettingListener createSettingLstnr(int subscription) {
		ArSSettingListener eosl = new ArSSettingListener();
		synchronized(settingsListener[subscription]) {
			settingsListener[subscription].add(eosl);
		}
		return eosl;
	}

	@Override
	protected void work(ResultData resultData) {
		long atgID = resultData.getDataDescription().getAttributeGroup().getId();
		if(atgID == SUBSCR_IDS[ARCHIVEINST]) {
			_debug.info("Neue Archiveinstellungen erhalten" + Debug.NEWLINE, resultData.toString()); // toString ist hier wichtig, um Deadlocks durch implizite Konfigurationsanfragen zu vermeiden
			ArchiveSettings as = new ArchiveSettings(resultData.getData());

			getArchMgr().getInQueuesMgr().getArchiveDataReceiver().setTMaxATimeForwardStep(as.maxATimeLeap);
			getArchMgr().getTaskScheduler().setIntervals(as.nachfIntervall);
			ArchiveQueryTask.setBlockingFactorMux(as.queryAnzBlocks);
			ArchiveQueryTask.setBufferSizeMux(as.queryAnzBytes);
			getArchMgr().getInQueuesMgr().setQueryTaskNumbers(as.anzHohePrio, as.anzMittPrio, as.anzNiedPrio);
			getArchMgr().getSimulationMgr().setTimeOutPeriod(as.timeoutSteuerbefehl);

			notifyListener(ARCHIVEINST);
		}
		else if(atgID == SUBSCR_IDS[ARCHIVCONTAINER]) {
			_debug.info("Neue Containereinstellungen erhalten" + Debug.NEWLINE, resultData.toString()); // toString ist hier wichtig, um Deadlocks durch implizite Konfigurationsanfragen zu vermeiden
			ContainerSettings cs = new ContainerSettings(resultData.getData());
			boolean minimumsCorrected = cs.ensureMinimums(MIN_CONTAINER_MAXDS, MIN_CONTAINER_MAXSIZE, MIN_CONTAINER_MAXTIME);
			if(minimumsCorrected) {
				_debug.warning(
						"Einstellungen fuer [" + SUBSCR_PIDS[ARCHIVCONTAINER][0] + "/" + SUBSCR_PIDS[ARCHIVCONTAINER][1] + "] enthaelt Wert unter Minimum\n"
						+ "(min. MaxAnzahlArchivdatensätze=" + MIN_CONTAINER_MAXDS + ", min. MaxContainergroesse=" + MIN_CONTAINER_MAXSIZE
						+ " Bytes, min. MaxZeitspanneContainer=" + MIN_CONTAINER_MAXTIME + "sec)\n", resultData.getData()
				);
			}
			ArchiveTask.setContainerSettings(cs);
			notifyListener(ARCHIVCONTAINER);
		}
	}

	private void notifyListener(int subscr) {
		synchronized(settingsListener[subscr]) {
			for(int i = 0; i < settingsListener[subscr].size(); i++) {
				settingsListener[subscr].get(i).gotSettings();
			}
			settingsListener[subscr].clear();
		}
	}


	/**
	 * Spezielle Klasse, mit der man auf das Ende der Übernahme der nächsten Archiveinstellungen warten kann. Dies ist notwendig, da das Archivsystem ohne diese
	 * Einstellungen nicht starten kann.
	 */
	public static class ArSSettingListener {

		private boolean gotSettings;

		private ArSSettingListener() {
		}

		public synchronized void gotSettings() {
			gotSettings = true;
			notifyAll();
		}

		public synchronized void waitForSettings() throws InterruptedException {
			while (!gotSettings) wait();
		}
	}
}
