/*
 * Copyright 2020 by Kappich Systemberatung, Aachen
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.ars.ars.mgmt.tasks;

import com.google.common.collect.ImmutableList;
import de.bsvrz.ars.ars.mgmt.ArchiveManager;
import de.bsvrz.dav.daf.util.cron.CronDefinition;
import de.bsvrz.dav.daf.util.cron.CronScheduler;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;

/**
 * Basisklasse für Tasks, die regelmäßig über eine Cron-Definition ausgeführt werden.
 *
 * @author Kappich Systemberatung
 */
public class CronTask extends Task {

	private final CronScheduler _cronScheduler;
	private String _name;
	private final List<CronTaskDefinition> _cronTaskDefinitions = new ArrayList<>();
	private List<ScheduledFuture<?>> _scheduledFutures = ImmutableList.of();
	private boolean _started;

	/**
	 * Erstellt einen neuen CronTask.
	 *
	 * @param archiveManager Archivmanager (für {@link Task})
	 */
	public CronTask(final ArchiveManager archiveManager) {
		super(archiveManager);
		_cronScheduler = new CronScheduler();
	}

	@Override
	public void start() {
		_started = true;
		final List<ScheduledFuture<?>> scheduledFutures = new ArrayList<>();
		for(CronTaskDefinition cronTaskDefinition : _cronTaskDefinitions) {
			scheduledFutures.add(_cronScheduler.schedule(cronTaskDefinition.runnable(), cronTaskDefinition.cronDefinition()));
		}
		_scheduledFutures = scheduledFutures;
	}
	
	public void setCronTaskDefinitions(List<CronTaskDefinition> definitions) {
		_scheduledFutures.forEach(it -> it.cancel(false));
		_cronTaskDefinitions.clear();
		_cronTaskDefinitions.addAll(definitions);
		if(_started && !shouldTerminate()) {
			start();
		}
	}

	@Override
	public void terminateTask() {
		super.terminateTask();
		_cronScheduler.shutdownNow();
	}

	@Override
	public String getName() {
		return _name;
	}

	public void setName(final String name) {
		_name = name;
	}

	@Override
	public void join() throws InterruptedException {
		while (!_cronScheduler.awaitTermination(1, TimeUnit.MINUTES)) {
			_debug.fine("CronTask.join() hat sich noch nicht terminiert.");
		}
	}

	@Override
	public boolean isAlive() {
		return !_cronScheduler.isTerminating() && !_cronScheduler.isTerminated();
	}

	@Override
	public boolean isTerminated() {
		return _cronScheduler.isTerminated();
	}

	/**
	 * Kapselt eine {@link CronTaskDefinition} zusammen mit einem Code der ausgeführt werden soll
	 *
	 * @param cronDefinition Zeitpunkte, wann das runnable ausgeführt werden soll
	 * @param runnable       Code, der zu den angegebenen Zeitpunkten ausgeführt werden soll
	 */
	public record CronTaskDefinition(CronDefinition cronDefinition, Runnable runnable) {
	}
}
