/*
 *
 * Copyright 2005-2008 by beck et al. projects GmbH, Munich
 * Copyright 2009-2020 by Kappich Systemberatung, Aachen
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.ars.ars.mgmt.tasks;

import de.bsvrz.ars.ars.mgmt.ArchiveManager;
import de.bsvrz.ars.ars.mgmt.LongTermTaskStatePublisher;
import de.bsvrz.ars.ars.mgmt.tasks.base.*;
import de.bsvrz.ars.ars.mgmt.datatree.synchronization.SynchronizationFailedException;
import de.bsvrz.ars.ars.mgmt.tasks.base.Task;
import de.bsvrz.ars.ars.persistence.*;
import de.bsvrz.ars.ars.persistence.directories.PersistenceDirectory;
import de.bsvrz.ars.ars.persistence.index.IndexException;
import de.bsvrz.ars.ars.persistence.util.SignalingQueue;
import de.bsvrz.ars.ars.persistence.walk.ContainerAction;
import de.bsvrz.ars.ars.persistence.walk.ContainerWalk;
import de.bsvrz.ars.ars.persistence.walk.internal.StatusPrinter;
import de.bsvrz.dav.daf.main.archive.ArchiveDataSpecification;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.time.Duration;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;

/**
 * Task zur Bearbeitung von Loeschzeit-Verlängerungsauftraegen.
 *
 * @author beck et al. projects GmbH
 * @author Christian Wied
 * @version $Revision$ / $Date$ / ($Author$)
 */
public class DeletePermanentlyTask extends QueueTask<DeletePermanentlyTask.DeletePermanentlyToken> {

	private final Task _longTermTaskStatePublisherTask;
	private final AtomicInteger _numOfDeletedCont = new AtomicInteger();
	private final int _numThreads;

	/**
	 * Erzeugt einen neuen Loeschzeit-Verlängerungs-Task.
	 *
	 * @param longTermTaskStatePublisher LongTermTaskStatePublisher
	 * @param numThreads                 Anzahl Threads, die gleichzeitig das Löschen durchführen.
	 * @param archiveMgr                 Der Archiv-Manager.
	 */
	public DeletePermanentlyTask(ArchiveManager archiveMgr, final LongTermTaskStatePublisher longTermTaskStatePublisher, final int numThreads) {
		super(archiveMgr, new SignalingQueue<>());
		setName("Endgültiges Löschen");
		_longTermTaskStatePublisherTask = longTermTaskStatePublisher.registerTask("Endgültiges Löschen");
		_numThreads = numThreads;
	}

	@Override
	protected void work(DeletePermanentlyToken step) {
		try {
			TaskManager.run(_longTermTaskStatePublisherTask,
					tpi -> containerWalk(new ArchiveDataSpecification[0], new DeletePermanently()));
			_longTermTaskStatePublisherTask.setTaskState(TaskState.IDLE);
		} catch (Exception e) {
			_debug.error("Endgültiges Löschen fehlgeschlagen", e);
		}
	}

	/**
	 * Die generelle Container-Walk-Methode.
	 * <p>
	 * Je nachdem, ob die ArchiveDataSpec gefuellt ist oder nicht, wird ein ContainerWalk über das komplette Persistenz-Verzeichnis (containerWalkAllFromDir)
	 * oder alle spezifizierten Container (containerWalkAllByDataSpec) ausgeführt.<br><br>
	 * <p>
	 * Achtung!<br>Die Methode löscht je nach übergebenen Parametern Archivdaten!
	 *
	 * @param ads Archivdatenspezifikation, die bearbeitet werden soll
	 * @throws PersistenceException Lesefehler im Persistenzverzeichnis
	 * @throws IndexException       Lesefehler der Indexe (z. B. korrupt)
	 */
	private void containerWalk(ArchiveDataSpecification[] ads, ContainerHandler handler) throws Exception {
		_numOfDeletedCont.set(0);
		if (ads.length == 0) {
			doContainerWalk(handler, ContainerWalk.allContainers(getPersistenceManager().getActivePersistenceDirectory(0), getPersistenceManager()));
		} else {
			doContainerWalk(handler, ContainerWalk.fromArchiveDataSpecification(getPersistenceManager().getActivePersistenceDirectory(0), Arrays.asList(ads),
					getPersistenceManager()));
		}
	}

	private void doContainerWalk(final ContainerHandler handler, final ContainerWalk containerWalk) throws PersistenceException {
		containerWalk.setStatusPrinter(handler.getStatusPrinter());
		containerWalk.execute(handler.getActionName(), _numThreads, new ContainerAction() {
			@Override
			public void preVisitDirectory(final DataIdentificationDir dataIdentificationDir) {
				if (shouldTerminate()) containerWalk.terminate();

				// Prüfen, ob Archivsystem überlastet und Task deshalb angehalten werden soll.
				DeletePermanentlyTask.this.suspendTaskIfNecessary();
			}

			@Override
			public void run(final DataIdentificationDir containerFileDir, final LockedContainerDirectory containerDirectory, final Collection<ContainerHeaders> headers) throws Exception {
				// Auf Container zugreifen und bearbeiten:
				List<ContainerHeaders> containersToHandle = headers.stream()
						.filter(it -> handler.appliesTo(containerFileDir.getDataIdentification(), it))
						.collect(Collectors.toList());

				if (!containersToHandle.isEmpty()) {
					handler.handleContainers(containerDirectory, containerFileDir.getPersistenceDirectory(), containersToHandle);
				}
			}
		});
	}

	/**
	 * Interface, das angibt, was mit Containern gemacht werden soll, implementiert die Verhaltensunterschiede von spontanem Löschen,
	 * manuellem Löschen, regulärem regelmäßigem Löschen, Löschzeitverlängerung usw.
	 */
	private interface ContainerHandler {

		/**
		 * Bearbeitet einen Container
		 *
		 * @param directory            Synchronisierte Identifikation des Containers
		 * @param persistenceDirectory Wurzelverzeichnis
		 * @param headers              Header von Containern
		 * @throws PersistenceException Lesefehler im Persistenzverzeichnis
		 */
		void handleContainers(final LockedContainerDirectory directory, PersistenceDirectory persistenceDirectory, final Collection<ContainerHeaders> headers) throws PersistenceException, SynchronizationFailedException, IndexException;

		/**
		 * Gibt an, ob der Container aus dem übergebenen Index behandelt (gelöscht/lz-verlängert) werden soll
		 *
		 * @param dataIdentification Datenidentifikation
		 * @param headers            Header von einem Container
		 * @return true: Container soll durch {@link #handleContainers(LockedContainerDirectory, PersistenceDirectory, Collection)} bearbeitet werden, sonst false
		 */
		boolean appliesTo(IdDataIdentification dataIdentification, ContainerHeaders headers);

		/**
		 * Gibt den Anzeigenamen der Aktion zurück
		 *
		 * @return Anzeigenamen der Aktion
		 */
		String getActionName();

		StatusPrinter getStatusPrinter();
	}

	/**
	 * Spezielles Markierungsobjekt, dass das dauerhafte Löschen einmalig auslöst.
	 */
	public static class DeletePermanentlyToken {

		/**
		 * Erstellt ein neues DeletePermanentlyToken.
		 */
		public DeletePermanentlyToken() {
		}
	}

	/**
	 * Endgültiges Löschen als {@link ContainerHandler}
	 */
	public class DeletePermanently implements ContainerHandler {

		private final boolean simulatePermanentDelete = getArchMgr().simulatePermanentDelete();

		@Override
		public void handleContainers(final LockedContainerDirectory directory, PersistenceDirectory persistenceDirectory, final Collection<ContainerHeaders> headers) throws PersistenceException, SynchronizationFailedException, IndexException {
			DeletedContainerFile deletedContainerFile = null;

			if (headers.isEmpty()) {
				return;
			}

			if (simulatePermanentDelete) {
				long min = headers.stream().mapToLong(it -> it.getContainerHeaderParamAsLong(ContainerManagementInformation.CHP_ARC_TIME_MIN)).min().orElseThrow();
				long max = headers.stream().mapToLong(it -> it.getContainerHeaderParamAsLong(ContainerManagementInformation.CHP_ARC_TIME_MAX)).max().orElseThrow();
				
				_debug.info("Das endgültige Löschen würde " + headers.size() + " Container der Datenidentifikation " + getPersistenceManager().formatContainerDirectory(directory) + " mit dem Archivzeitbereich " + Instant.ofEpochMilli(min) + " bis " + Instant.ofEpochMilli(max) + " löschen. Um das Endgültige Löschen zu aktivieren bitte Archivsystem mit -permanentLoeschen=ja starten, oder das Löschen über Telnet aktivieren.");

				_numOfDeletedCont.getAndIncrement();
				return;
			}
			
			for (ContainerHeaders header : headers) {

				long containerId = header.getContainerHeaderParamAsLong(ContainerManagementInformation.CHP_CONT_ID);

				try {
					persistenceDirectory.removeContainerFromIndex(directory, containerId);

					if (header.getContainerHeaderParamAsBoolean(ContainerManagementInformation.CHP_DELETED)) {
						if (deletedContainerFile == null) {
							// deletedContainerFile nur Laden, wenn benötigt
							deletedContainerFile = persistenceDirectory.openDeletedContainerFile(directory);
						}
						deletedContainerFile.removeDeletedContainer(containerId);
					} else {
						Path contFile = persistenceDirectory
								.getPath(directory).resolve(ContainerFile.getContainerFileName(containerId));

						Files.delete(contFile);
					}
				} catch (IOException e) {
					_debug.warning("Container " + containerId + " in " + directory + " konnte nicht endgültig gelöscht werden", e);
					return;
				}

				_numOfDeletedCont.getAndIncrement();
			}
			if (deletedContainerFile != null) {
				deletedContainerFile.write();
			}
		}

		@Override
		public boolean appliesTo(final IdDataIdentification dataIdentification, final ContainerHeaders headers) {
			if (simulatePermanentDelete) {
				_longTermTaskStatePublisherTask.setTaskProgress("Endgültiges Löschen: %1$d Container wären gelöscht worden", _numOfDeletedCont.get(), -1, -1);
			} else {
				_longTermTaskStatePublisherTask.setTaskProgress("Endgültiges Löschen: %1$d Container gelöscht", _numOfDeletedCont.get(), -1, -1);
			}

			long maxArcTime = headers.getContainerHeaderParamAsLong(ContainerManagementInformation.CHP_ARC_TIME_MAX);

			if (maxArcTime == 0) {
				// nicht abgeschlossen
				return false;
			}

			long permanentDeleteTimeSeconds = getArchMgr().getArchivConfig().getPermanentDeleteTimeSeconds(getDidTree().get(dataIdentification));
			if (permanentDeleteTimeSeconds <= 0) return false;

			long systemTime = getArchMgr().getInQueuesMgr().getRuntimeControl().getSystemTime();
			return maxArcTime + 1000 * permanentDeleteTimeSeconds < systemTime;
		}

		@Override
		public String getActionName() {
			return "Endgültiges Löschen";
		}

		@Override
		public StatusPrinter getStatusPrinter() {
			return new StatusPrinter() {
				@Override
				public String getStatusMessage(final String actionName, final Duration runtime, final ApproximationType approximationType, final long allContainerDirs, final long visitedContainerDirs, final long visitedContainerFiles) {
					if (simulatePermanentDelete) {
						return super.getStatusMessage(actionName, runtime, approximationType, allContainerDirs, visitedContainerDirs, visitedContainerFiles)
								+ " " + _numOfDeletedCont.get() + " Container wären gelöscht worden.";
					}
					return super.getStatusMessage(actionName, runtime, approximationType, allContainerDirs, visitedContainerDirs, visitedContainerFiles)
							+ " " + _numOfDeletedCont.get() + " Container gelöscht.";
				}

				@Override
				public String getSuccessMessage(final String actionName, final Duration runtime, final long visitedContainerDirs, final long visitedContainerFiles) {
					if (simulatePermanentDelete) {
						return super.getSuccessMessage(actionName, runtime, visitedContainerDirs, visitedContainerFiles)
								+ " " + _numOfDeletedCont.get() + " Container wären gelöscht worden.";
					}
					return super.getSuccessMessage(actionName, runtime, visitedContainerDirs, visitedContainerFiles)
							+ " " + _numOfDeletedCont.get() + " Container gelöscht.";
				}
			};
		}
	}
}
