/*
 *
 * Copyright 2005-2008 by beck et al. projects GmbH, Munich
 * Copyright 2009-2020 by Kappich Systemberatung, Aachen
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.ars.ars.mgmt.tasks;

import de.bsvrz.ars.ars.mgmt.ArchiveManager;
import de.bsvrz.ars.ars.mgmt.datatree.synchronization.SyncKey;
import de.bsvrz.ars.ars.mgmt.datatree.synchronization.SynchronizationFailedException;
import de.bsvrz.ars.ars.mgmt.simulation.SimulationResultData;
import de.bsvrz.ars.ars.mgmt.tasks.query.QueryHandler;
import de.bsvrz.ars.ars.persistence.IdDataIdentification;
import de.bsvrz.ars.ars.persistence.PersistenceException;
import de.bsvrz.ars.ars.persistence.directories.ActivePersistenceDirectory;
import de.bsvrz.ars.ars.persistence.walk.DataIdentificationDirWalk;
import de.bsvrz.dav.daf.main.ResultData;
import de.bsvrz.sys.funclib.dataSerializer.NoSuchVersionException;
import de.bsvrz.sys.funclib.losb.datk.MessageType;

import java.io.IOException;

/**
 * Task zum Loeschen von Simulationsdaten einer bestimmten Simulationsvariante. Der Task wird verwendet um von Clients und der Simulation {@link
 * de.bsvrz.ars.ars.mgmt.simulation.SimulationAutomaton} eingehende Löschaufträge zu verarbeiten. Die Daten werden sowohl von dem Filesystem entfernt und die
 * betroffenen DataIdentNodes werden aus dem DIN-Tree genommen.
 *
 * @author beck et al. projects GmbH
 * @author Alexander Schmidt
 * @version $Revision$ / $Date$ / ($Author$)
 */
public class DeleteSimVarTask extends SingleTask {

	/**
	 * Erstellt einen neuen DeleteSimVarTask
	 * @param archiveMgr Archivmanager
	 * @see QueueTask
	 */
	public DeleteSimVarTask(ArchiveManager archiveMgr) {
		super(archiveMgr);
	}

	@Override
	protected void work(ResultData resultData) {
		if (resultData != null) {
			if (resultData instanceof SimulationResultData) {
				// Löschauftrag wurde von einer Simulation gestellt
				try {
					deleteSimVarSimulation((SimulationResultData) resultData);
				} catch (PersistenceException e) {
					_debug.warning("Fehler beim Löschen. " + e.getMessage());
				}
			} else if (resultData.hasData()) {
				// Löschauftrag wurde vom Client beauftragt
				deleteSimVarClient(resultData);
			}

		}
	}

	/**
	 * Führt Löschauftrag durch, der von einer Simulation gestellt wurde.
	 *
	 * @param srd Auftragsparameter.
	 *
	 * @throws PersistenceException Lesefehler im Persistenzverzeichnis
	 */
	private void deleteSimVarSimulation(SimulationResultData srd) throws PersistenceException {
		boolean successful = false;
		try {
			executeDelete(srd.getSimVar());
			successful = true;
		}
		finally {
			srd.dataDeleted(successful);
		}
	}

	/**
	 * Führt Löschauftrag durch, der von einem Client gestellt wurde.
	 *
	 * @param resultData Auftragsdaten
	 *
	 */
	private void deleteSimVarClient(ResultData resultData) {
		QueryHandler queryHandler = new QueryHandler(getName(), getArchMgr(), MessageType.DELETE_RESULT, resultData.getData());
		try {
			// SimVar rausholen.
			short simVarToDelete = queryHandler.getDeserializer().readShort();
			// Löschen durchführen.
			executeDelete(simVarToDelete);
			queryHandler.sendSuccessResponse();
		}
		catch(NoSuchVersionException e) {
			String msg = "Geforderte Version Serialisierers/Deserialisierers nicht verfuegbar. " + e.getMessage();
			_debug.warning(msg);
			queryHandler.sendErrorResponse(msg);
		} catch (IOException e) {
			String msg = "Fehler beim Deserialisieren. " + e.getMessage();
			_debug.warning(msg);
			queryHandler.sendErrorResponse(msg);
		} catch (PersistenceException e) {
			String msg = "Fehler beim Löschen. " + e.getMessage();
			_debug.warning(msg);
			queryHandler.sendErrorResponse(msg);
		}
	}

	/**
	 * Löscht die Daten einer Simulationsvariante. Entfernt die betroffenen DataIdentNodes aus dem Baum.<br> Vorgehen: Es müssen alle Verzeichnisse durchlaufen
	 * werden, da es möglich ist, dass Verzeichnisse existieren, die nicht durch DataIdentNodes repräsentiert werden.
	 *
	 * @param simVarToDelete Simulationsvariante.
	 *
	 * @throws PersistenceException Fehler beim Löschen der Daten.
	 */
	private void executeDelete(short simVarToDelete) throws PersistenceException {
		// Achtung: Es ist möglich, dass es hier zu einer kurzzeitigen Inkonsistenz kommt:
		// Zwei Threads greifen auf die gleiche DIN zu, der Löschtask 'gewinnt', der andere wartet
		// auf die Freigabe der DIN. Wenn der die Freigabe erhält wurde das zur DIN gehörenden
		// Verzeichnis bereits gelöscht.

		if(simVarToDelete == 0) throw new PersistenceException("Ungültige Simulationsvariante");

		ActivePersistenceDirectory directory = getPersistenceManager().getActivePersistenceDirectory(simVarToDelete);
		if (directory == null) {
			// Nicht vorhanden, also auch nichts löschen
			return;
		}
		DataIdentificationDirWalk.allDirectories(directory).execute("Simulationsvariante " + simVarToDelete + " Löschen", 1, (dataIdentificationDir, walk) -> {
			IdDataIdentification di = dataIdentificationDir.getDataIdentification();
			if(di.getSimVariant() == simVarToDelete) {
				try {
					ArchiveManager archiveManager = getArchMgr();
					try (SyncKey<IdDataIdentification> syncKey = archiveManager.getPersistenceManager().lockIndex(di)) {
						// Zuerst SV-Verzeichnis loeschen und danach DIN, da über diesen synchronisiert wird:
						getPersistenceManager().deleteSimVar(syncKey);
						getDidTree().deleteDataIdentNode(di);
					}
				} catch (SynchronizationFailedException e) {
					// Loeschen des DIN fehlgeschlagen
					_debug.warning("Loeschen eines DataIdentNodes fehlgeschlagen fuer DIN " + di, e);
				}
			}
		});
		getPersistenceManager().deletePersistenceDirectory(directory);
	}
}
