/*
 * Copyright 2020 by Kappich Systemberatung, Aachen
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.ars.ars.mgmt.tasks;

import de.bsvrz.ars.ars.mgmt.TaskManager;
import de.bsvrz.ars.ars.mgmt.TaskManagerInterface;
import de.bsvrz.ars.ars.mgmt.datatree.DataIdentTree;
import de.bsvrz.ars.ars.persistence.PersistenceManager;
import de.bsvrz.sys.funclib.debug.Debug;

/**
 * Basisklasse für Hintergrund-Tasks im Archivsystem
 *
 * @author Kappich Systemberatung
 */
public abstract class Task {
	/**
	 * Logger
	 */
	protected static final Debug _debug = Debug.getLogger();
	/**
	 * Zeigt an, ob der Task beendet werden soll. Der Task bricht dann möglichst schnell an der nächsten sicheren Stelle ab.
	 */
	protected volatile boolean _shouldTerminate;

	private final TaskManagerInterface taskManagerInterface;

	private final PersistenceManager persMgr;

	/**
	 * Erstellt einen neuen Task
	 *
	 * @param taskManagerInterface Archiv-Manager (Kann für Testfälle und ähnliches null sein, allerdings kann es dann passieren, das bestimmte
	 *				   Funktionen nicht funktionieren.)
	 */
	public Task(TaskManager taskManagerInterface) {
		this.taskManagerInterface = taskManagerInterface;
		if (this.taskManagerInterface != null) {
			persMgr = taskManagerInterface.getPersistenceManager();
		}
		else {
			persMgr = null;
		}
	}
	
	/**
	 * Erstellt einen neuen Task
	 *
	 * @param persistenceManager Persistenz-Manager
	 */
	public Task(PersistenceManager persistenceManager) {
		taskManagerInterface = persistenceManager;
		persMgr = persistenceManager;
	}

	/**
	 * Startet den Task.
	 */
	public abstract void start();

	/**
	 * Terminiert den Task, indem dafür gesorgt wird, das {@link #shouldTerminate()} true zurückliefert. Der Task
	 * sollte diese Funktion regelmäßig aufrufen um dann bei Rückgabe von {@code true} den Task zeitnah zu beenden.
	 */
	public void terminateTask() {
		_shouldTerminate = true;
	}

	/**
	 * Zeigt an, ob der Task beendet werden soll.
	 *
	 * @return {@code true} falls der Task beendet werden soll.
	 */
	public boolean shouldTerminate() {
		return _shouldTerminate;
	}

	/**
	 * Hält den Task an, falls sich das Archivsystem in einem kritischen Zustand befindet. Wird das Archivsystem beendet, wird der Task fortgesetzt, so dass er
	 * seine Aufgabe noch vollständig erfüllen kann. {@link #shouldTerminate()} liefert anschließend {@code true} zurück.
	 */
	public final void suspendTaskIfNecessary() {
		try {
			TaskManagerInterface archiveManager = getArchMgr();
			if (archiveManager != null) archiveManager.suspendTaskIfNecessary(this);
		} catch (InterruptedException e) {
			Task._debug.warning("Thread wurde beim Warten unterbrochen", e);
		}
	}

	public abstract String getName();

	/**
	 * Wartet auf das Beenden des Tasks ähnlich {@link Thread#join()}.
	 * @throws InterruptedException Unterbrochen beim warten
	 */
	public abstract void join() throws InterruptedException;

	/**
	 * Gibt {@code true} zurück, wenn der Task noch "lebt".
	 * @return {@code true}, wenn der Task noch "lebt", sonst {@code false}
	 */
	public abstract boolean isAlive();

	/**
	 * Gibt {@code true} zurück, wenn der Task terminiert wurde.
	 * @return {@code true}, wenn der Task terminiert wurde, sonst {@code false}
	 */
	public abstract boolean isTerminated();

	/**
	 * Zentrale Archiv-Verwaltung.
	 */
	public TaskManagerInterface getArchMgr() {
		return taskManagerInterface;
	}

	/**
	 * Gibt die Verwaltung der Datenidentifikationen zurück
	 */
	public DataIdentTree getDidTree() {
		return persMgr.getDataIdentTree();
	}

	/**
	 * PersistenceManager für den Zugriff auf die Daten des Archivsystems
	 */
	public PersistenceManager getPersistenceManager() {
		return persMgr;
	}
}
