/*
 *
 * Copyright 2005-2008 by beck et al. projects GmbH, Munich
 * Copyright 2009-2020 by Kappich Systemberatung, Aachen
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.ars.ars.mgmt.tasks.scheduler;

import de.bsvrz.ars.ars.mgmt.ArchiveManager;
import de.bsvrz.ars.ars.mgmt.tasks.RequestGapTask;
import de.bsvrz.sys.funclib.debug.Debug;

import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.Timer;
import java.util.TimerTask;

/**
 * Diese Klasse verwaltet einen Timer und prüft im Minutentakt, ob die Tasks für Sicherung, Löschen und Nachforndern ausgeführt werden müssen.
 * <p>
 * Über die public-Methoden können die IntervalWerte gesetzt werden.
 *
 * @author beck et al. projects GmbH
 * @author Phil Schrettenbrunner
 * @version $Revision$ / $Date$ / ($Author$)
 */
public class TaskScheduler {

	private static final ThreadLocal<DateFormat> dateFormat = ThreadLocal.withInitial(() -> new SimpleDateFormat("dd.MM.yyyy HH:mm:ss"));

	private static final int ONE_MINUTE = 60 * 1000;
	
	private static final long ONE_HOUR = 60 * ONE_MINUTE;

	private static Debug logger;

	private final Timer timer;

	private final ArchiveManager archiveManager;

	private final TimeIntervalCron deletePermanentlyTimeInterval = new TimeIntervalCron();

	private final TimeIntervalCron requestTimeInterval = new TimeIntervalCron();

	private Date deletePermanentlyLastRun;

	private Date requestLastRun;

	private final CronTask deletePermanentlyCronTask;

	private final CronTask requestCronTask;

	private final TimerTask saveUnsubscriptionTimeTask;


	/**
	 * Erzeugt einen TaskScheduler mit den angegebenen Interval-Zeiten für den Sicherungs-, Lösch- und Nachforderungstask.
	 *
	 * @param archiveManager Archiv-Manager
	 */
	public TaskScheduler(ArchiveManager archiveManager) {
		logger = Debug.getLogger();
		this.timer = new Timer();
		this.archiveManager = archiveManager;
		deletePermanentlyCronTask = new CronTask(deletePermanentlyTimeInterval);
		requestCronTask = new CronTask(requestTimeInterval);
		saveUnsubscriptionTimeTask = new UnsubscriptionTimerTask();
	}


	/** Startet den Timer, der die Tasks steuert */
	public void start() {
		// Timer zur nächsten vollen Minute starten
		long start = ONE_MINUTE - (System.currentTimeMillis() % ONE_MINUTE);
		start = start < 100 ? 0 : start;
		timer.scheduleAtFixedRate(deletePermanentlyCronTask, start, ONE_MINUTE);
		timer.scheduleAtFixedRate(requestCronTask, start, ONE_MINUTE);
		timer.scheduleAtFixedRate(saveUnsubscriptionTimeTask, start, ONE_HOUR);
	}


	/**
	 * Setzt die Intervalle
	 *
	 * @param request Nachforderungs-Intervall (Cron-String)
	 */
	public void setIntervals(String request) {
		requestTimeInterval.setFields(request);
		if(requestTimeInterval.hasParseErrors()) {
			logger.warning(
					"'" + request + "' ist kein gültiger ZeitIntervallCron für das Nachforderungsintervall. Automatischer Start für das Nachfordern DEAKTIVIERT"
			);
		}
		else {
			logger.fine("Setze Nachforderungsintervall auf: " + requestTimeInterval);
		}
	}


	/**
	 * Setzt das Interval fürs endgültige Löschen
	 * @param permanentlyDeleteCron Cron-String
	 * @param enabled               Aktiviert?
	 */
	public void setDeletePermanentlyInterval(String permanentlyDeleteCron, boolean enabled) {
		if(!enabled) {
			deletePermanentlyTimeInterval.setFields("");
		}
		else {
			deletePermanentlyTimeInterval.setFields(permanentlyDeleteCron);
		}
		if(deletePermanentlyTimeInterval.hasParseErrors()) {
			logger.warning("'" + permanentlyDeleteCron + "' ist kein gültiger ZeitIntervallCron für das endgültige Löschintervall. Automatischer Start für das endgültige Löschen DEAKTIVIERT");
		}
		else {
			logger.fine("Setze Intervall für das endgültige Löschen auf: " + deletePermanentlyTimeInterval);
		}
	}


	/**
	 * Das Interval als String. Wenn leerer String "", wird der Task nie ausgeführt.
	 *
	 * @return Das Interval (ausgewertet) als String
	 */
	public String getDeletePermanentlyIntervals() {
		return deletePermanentlyTimeInterval.toString();
	}


	/**
	 * Liefert das Datum, an dem der Task das letzte mal gestartet wurde. Oder einen leeren String, wenn der Task noch nie gestartet wurde.
	 *
	 * @return Datum als String
	 */
	public String getDeletePermanentlyLastRun() {
		if(deletePermanentlyLastRun != null) return dateFormat.get().format(deletePermanentlyLastRun);
		return "(wurde noch nie gestartet)";
	}


	/**
	 * Liefert das Datum, an dem der Task das nächste mal gestartet wird. Oder einen String, der besagt, dass der Task nie gestartet wird (weil die
	 * konfigurierten Werte ungültig sind). Berücksichtigt werden Daten, die maximal 1 Jahr und 1 Tag in der Zukunft liegen. Tritt kein Fehler auf und wird
	 * trotzdem kein Datum gefunden ist der Rückgabewert "&gt; 1 Jahr".
	 *
	 * @return Datum als String
	 */
	public String getDeletePermanentlyNextRun() {
		if(deletePermanentlyTimeInterval.hasValues()) {
			Calendar deletePermanentlyNextRun = deletePermanentlyTimeInterval.getNextRun(null); // Parameter=null bedeutet "von jetzt ab"
			if(deletePermanentlyNextRun != null) {
				return dateFormat.get().format(deletePermanentlyNextRun.getTime());
			}
			else {
				return "> 1 Jahr";
			}
		}
		return "---";
	}


	/**
	 * Das Interval als String. Wenn leerer String "", wird der Task nie ausgeführt.
	 *
	 * @return Das Interval (ausgewertet) als String
	 */
	public String getRequestIntervals() {
		return requestTimeInterval.toString();
	}


	/**
	 * Liefert das Datum, an dem der Task das letzte mal gestartet wurde. Oder einen leeren String, wenn der Task noch nie gestartet wurde.
	 *
	 * @return Datum als String
	 */
	public String getRequestLastRun() {
		if(requestLastRun != null) return dateFormat.get().format(requestLastRun);
		return "(wurde noch nie gestartet)";
	}


	/**
	 * Liefert das Datum, an dem der Task das nächste mal gestartet wird. Oder einen String, der besagt, dass der Task nie gestartet wird (weil die
	 * konfigurierten Werte ungültig sind). Berücksichtigt werden Daten, die maximal 1 Jahr und 1 Tag in der Zukunft liegen. Tritt kein Fehler auf und wird
	 * trotzdem kein Datum gefunden ist der Rückgabewert "&gt; 1 Jahr".
	 *
	 * @return Datum als String
	 */
	public String getRequestNextRun() {
		if(requestTimeInterval.hasValues()) {
			Calendar requestNextRun = requestTimeInterval.getNextRun(null); // Parameter=null bedeutet "von jetzt ab"
			if(requestNextRun != null) {
				return dateFormat.get().format(requestNextRun.getTime());
			}
			else {
				return "> 1 Jahr";
			}
		}
		return "---";
	}

	/** TaskScheduler beenden. Laufende Tasks laufen aber noch zu Ende. Es werden keine neuen Tasks mehr gestartet. */
	public void terminate() {
		if(deletePermanentlyCronTask != null) deletePermanentlyCronTask.cancel();
		if(requestCronTask != null) requestCronTask.cancel();
		if(saveUnsubscriptionTimeTask != null) saveUnsubscriptionTimeTask.cancel();
		timer.cancel();
	}

	private class UnsubscriptionTimerTask extends TimerTask {

		@Override
		public void run() {
			archiveManager.getPersistenceManager().saveUnsubscriptionTime();
		}
	}

	/**
	 * Private Klasse, die sich um den Aufruf von Sichern, Löschen und Nachfordern kümmert. Die Klasse wird minütlich aufgerufen und in der run()-Methode wird
	 * gepfüft, ob einer der Tasks gestartet werden muss (auf Grund der Intervallangabe im TimeIntervalCron)
	 *
	 * @author Philippe Schrettenbrunner
	 */
	private class CronTask extends TimerTask {

		private final TimeIntervalCron tic;

		public CronTask(TimeIntervalCron tic) {
			this.tic = tic;
		}


		@Override
		public void run() {
			Date d = new Date(scheduledExecutionTime());
			if (tic.shouldRun(d)) {
				if (tic == deletePermanentlyTimeInterval) {
					logger.info("Starte endgueltiges Loeschen");
					try {
						archiveManager.getInQueuesMgr().startDeletePermanently();
						deletePermanentlyLastRun = d;
					} catch (RuntimeException e) {
						logger.warning("Fehler beim Start des des endgültigen Löschens", e);
					}
				}
				else if(tic == requestTimeInterval) {
					logger.info("Starte Nachforderung fehlender Archivdaten");
					try {
						archiveManager.getInQueuesMgr().insertInRequestQueue(RequestGapTask.getRequestGapResultData());
						requestLastRun = d;
					}
					catch(RuntimeException e) {
						logger.warning("Fehler beim Start der Nachforderung fehlender Archivdaten", e);
					}
				}
			}
		}
	}
}
