/*
 *
 * Copyright 2009-2020 by Kappich Systemberatung, Aachen
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.ars.ars.persistence;

import de.bsvrz.ars.ars.persistence.walk.internal.StatusPrinter;
import de.bsvrz.sys.funclib.commandLineArgs.ArgumentList;
import de.bsvrz.sys.funclib.debug.Debug;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.nio.file.Path;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;

/**
 * Diese Klasse implementiert die Verwaltung der Caches für die Schreibzugriffe auf Containerdateien. Die Klasse ist als Singleton realisiert, weil es im
 * Original-Archivsystem keine zentrale Stelle zum Zugriff auf Container-Dateien gibt (der PersistenzManager war wohl dafür vorgesehen, allerdings gibt es auch
 * einige Zugriffe auf Containerdateien, die am PersistenzManager vorbei gehen). Vor dem Zugriff auf Containerdateien muss die Cache-Verwaltung mit der Methode
 * {@code init()} erzeugt und initialisiert werden. Beim Beenden des Archivsystems muss mit einem Aufruf der Methode {@code close} das Schreiben der
 * noch in den einzelnen Caches vorhandenen Daten veranlasst werden.
 *
 * @author Kappich Systemberatung
*/
public class CacheManager {

	/** Minimale Puffergröße je Datenidentifikation als Anzahl Bytes. */
	private static final int MINIMUM_BUFFER_SIZE = 0;

	/**
	 * Standard-Puffergröße je Datenidentifikation als Anzahl Bytes. Wird verwendet, wenn kein anderer Wert über das Aufrufargument cachePufferGröße=...
	 * angegeben wurde.
	 */
	private static final int DEFAULT_BUFFER_SIZE = 5 * 1024;

	/** Maximale Puffergröße je Datenidentifikation als Anzahl Bytes. */
	private static final int MAXIMUM_BUFFER_SIZE = 33 * 1024;

	/** Singleton Objekt des CacheManagers. */
	private static final CacheManager _cacheManager = new CacheManager();

	/** Hashmap zur Zuordnung einer Container-ID zum zugehörigen Cache-Objekt. */
	private final HashMap<Long, Cache> _containerId2Cache = new HashMap<>();

	/**
	 * Beim Anlegen der Cache-Objekte zu verwendende Puffergröße als Anzahl Bytes. Der Wert kann über das Aufrufargument -cachePufferGröße=... eingestellt
	 * werden.
	 *
	 * @see #DEFAULT_BUFFER_SIZE
	 * @see #init(de.bsvrz.sys.funclib.commandLineArgs.ArgumentList)
	 */
	private int _defaultBufferSize = DEFAULT_BUFFER_SIZE;

	/**
	 * Hashmap mit den Datenidentifikationen, die nicht gepuffert, sondern sofort persistiert werden. Die Hashmap wird von der Methode {@link
	 * #setCachingEnabled(long,long,long,int,boolean)} modifiziert und dient dazu, die Zwischenspeicherung von Datenidentifikationen, bei denen der
	 * Quittierungsmechanismus des Archivsystems verwendet wird, zu verhindern.
	 */
	private final HashSet<IdDataIdentification> _disabledCacheIdentifications = new HashSet<>();

	/** Debug-Logger für Debug-Ausgaben. */
	private static final Debug _debug = Debug.getLogger();

	/** Gesamtanzahl von Datei-Schreibvorgängen aller geschlossenen Cache-Objekte. */
	private long _writeCountSum;

	/** Gesamtanzahl von verarbeiteten Datenblöcken aller geschlossenen Cache-Objekte. */
	private long _processedCountSum;

	/** Gesamtanzahl von zwischengespeicherten Datenblöcken aller geschlossenen Cache-Objekte. */
	private long _bufferedCountSum;

	/** Gesamtanzahl von nicht zwischengespeicherten Datenblöcken aller geschlossenen Cache-Objekte. */
	private long _unbufferedCountSum;

	private boolean _cacheEnabled = true;

	/** Erzeugt eine neue Cacheverwaltung. */
	private CacheManager() {
	}

	/** @return Liefert das CacheManager-Objekt zurück. */
	public static CacheManager getInstance() {
		return _cacheManager;
	}

	/**
	 * Initialisiert den CacheManager. Das Aufrufargument -cachePufferGröße=..., mit dem die Größe des Puffers pro Cache eingestellt werden kann, wird gelesen.
	 *
	 * @param argumentList Aufrufargumente der Applikation
	 */
	public void init(final ArgumentList argumentList) {
		_debug.fine("CacheManager.init " + argumentList);
		_writeCountSum = 0;
		_processedCountSum = 0;
		_bufferedCountSum = 0;
		_unbufferedCountSum = 0;
		_containerId2Cache.clear();
		_disabledCacheIdentifications.clear();
		_defaultBufferSize = argumentList.fetchArgument("-cachePufferGroesse=" + DEFAULT_BUFFER_SIZE).intValue();
		if(_defaultBufferSize < MINIMUM_BUFFER_SIZE) _defaultBufferSize = MINIMUM_BUFFER_SIZE;
		if(_defaultBufferSize > MAXIMUM_BUFFER_SIZE) _defaultBufferSize = MAXIMUM_BUFFER_SIZE;
		String info = "Cache wurde initialisiert. Puffergröße wurde auf " + _defaultBufferSize + " Bytes je Container eingestellt.";
		if (!isCacheEnabled()) info += " Cache ist momentan deaktiviert.";
		_debug.info(info);
	}

	/**
	 * Schreibt alle in den einzelnen Caches vorhandenen Daten und schließt die Cache-Verwaltung. Vor einer erneuten Verwendung muss die Cache-Verwaltung mit einem
	 * erneuten Aufruf der Methode {@code init} initialisiert werden.
	 */
	public void close() {
		_debug.fine("CacheManager.close");
		_debug.info("Cache wird geschlossen. Alle Puffer werden auf Platte geschrieben.");

		final Collection<Cache> caches;
		_disabledCacheIdentifications.clear();
		synchronized(_containerId2Cache) {
			caches = new ArrayList<>(_containerId2Cache.values());
			_containerId2Cache.clear();
		}
		int goodFlushCount = 0;
		int badFlushCount = 0;
		long start = System.currentTimeMillis();
		long lastPrintTime = start;
		int count = caches.size();
		for(Cache cache : caches) {
			// Ausgabe über Fortschritt jede Minute
			long now = System.currentTimeMillis();
			if(now - lastPrintTime > 60000) {
				// Ausgabe über Fortschritt
				lastPrintTime = now;
				String runtimeProgress = StatusPrinter.estimateRuntime(Duration.ofMillis(now - start), badFlushCount + goodFlushCount, count);
				if(badFlushCount == 0) {
					_debug.info(goodFlushCount + " von " + count + " Puffer wurden bisher geschrieben...\n" + runtimeProgress);
				}
				else {
					_debug.info(goodFlushCount + " von " + count + " Puffer wurden bisher geschrieben. " + badFlushCount + " Puffer konnten nicht geschrieben werden...\n" + runtimeProgress);
				}
			}


			try {
				cache.flush();
				aggregateCounts(cache);
				goodFlushCount++;
			}
			catch(PersistenceException e) {
				badFlushCount++;
				_debug.warning("Fehler beim Schreiben von gecachten Daten eines Containers: " + cache, e);
			}
		}
		logCloseMessage(badFlushCount, goodFlushCount);
	}

	private void logCloseMessage(int badFlushCount, int goodFlushCount) {
		String message = "CacheManager wurde geschlossen.\n";
		if(badFlushCount == 0) {
			message += "Alle " + goodFlushCount + " Puffer wurden auf Platte geschrieben.\n";
		}
		else {
			message += goodFlushCount + " Puffer wurden auf Platte geschrieben. " + badFlushCount + " Puffer konnten nicht geschrieben werden\n";
		}
		message += _processedCountSum + " Datenblöcke wurden verarbeitet.\n";
		message += _bufferedCountSum + " Datenblöcke wurden zwischengespeichert.\n";
		message += _unbufferedCountSum + " Datenblöcke wurden nicht zwischengespeichert.\n";
		message += _writeCountSum + " Schreibzugriffe.";
		_debug.info(message);
	}

	/** Schreibt alle in den einzelnen Caches vorhandenen Daten. */
	public void flushAll() {
		_debug.fine("CacheManager.flushAll");
		_debug.info("Alle Puffer werden auf Platte geschrieben.");

		final Collection<Cache> caches;
		synchronized(_containerId2Cache) {
			caches = new ArrayList<>(_containerId2Cache.values());
		}
		int goodFlushCount = 0;
		int badFlushCount = 0;
		for(Cache cache : caches) {
			try {
				cache.flush();
				goodFlushCount++;
			}
			catch(PersistenceException e) {
				badFlushCount++;
				_debug.warning("Fehler beim Schreiben von gecachten Daten eines Containers: " + cache, e);
			}
		}
		String message = "";
		if(badFlushCount == 0) {
			message += "Alle " + goodFlushCount + " Puffer wurden auf Platte geschrieben.\n";
		}
		else {
			message += goodFlushCount + " Puffer wurden auf Platte geschrieben. " + badFlushCount + " Puffer konnten nicht geschrieben werden\n";
		}
		_debug.info(message);
	}

	/**
	 * Gibt den vom Cache verbrauchten Speicher zurück (ungefähr)
	 * @return Speicherverbrauch in bytes
	 */
	public CacheMemoryUsage getCachedMemory() {
		long bufferSum = 0;
		long usedSum = 0;
		long overheadSum = 100; // Ungefähre Größe dieser Klasse, verhindert Division durch 0 Fehler
		synchronized(_containerId2Cache) {
			for(Cache cache : _containerId2Cache.values()) {
				bufferSum += cache._out.bufSize();
				usedSum += cache._out.size();
				overheadSum += 77; // 77 == Ungefähre Speichergröße eines Cache-Objekts
			}
		}
		return new CacheMemoryUsage(bufferSum + overheadSum, usedSum);
	}

	/**
	 * Liefert den Cache für den angegebenen Container zurück.
	 *
	 * @param containerFile Container, dessen Cache bestimmt werden soll
	 *
	 * @return Cache des Containers oder {@code null}, wenn der Container keinen Cache hat.
	 */
	public Cache getCache(final ContainerFile containerFile) {
		return getCache(containerFile.getContainerId());
	}

	/**
	 * Liefert den Cache für den angegebenen Container zurück.
	 *
	 * @param containerId Container, dessen Cache bestimmt werden soll
	 * 
	 * @return Cache des Containers oder {@code null}, wenn der Container keinen Cache hat.
	 */
	public Cache getCache(final long containerId) {
		_debug.fine("CacheManager.getCache " + containerId);
		synchronized(_containerId2Cache) {
			return _containerId2Cache.get(containerId);
		}
	}

	/**
	 * Erzeugt einen neuen Cache für den angegebenen Container.
	 *
	 * @param cont               Container für den ein neuer Cache erzeugt werden soll.
	 * @param createFile         {@code true}, falls eine neue Datei erzeugt werden soll; {@code false}, falls zu einer vorhandenen Datei hinzugefügt werden
	 *                           soll.
	 * @param dataIdentification Datenidentifikation des Containers
	 * @return Neuer Cache oder {@code null}, falls die Datenidentifikation des Containers nicht gecacht werden soll.
	 */
	public Cache createCache(final ContainerFile cont, final boolean createFile, final IdDataIdentification dataIdentification) {
		Path contFile = cont.getContFile();
		if (contFile == null) {
			throw new AssertionError("Ungültige Containerdatei angegeben.");
		}
		if(!_cacheEnabled) return null;
		synchronized(_disabledCacheIdentifications) {
			if (_disabledCacheIdentifications.contains(dataIdentification)) {
				_debug.fine("CacheManager.createCache " + dataIdentification + " wird nicht gecacht");
				return null;
			}
		}
		_debug.fine("CacheManager.createCache " + dataIdentification + " wird gecacht");
		final Cache cache = new Cache(contFile.toFile(), createFile, _defaultBufferSize);
		synchronized(_containerId2Cache) {
			_containerId2Cache.put(cont.getContainerId(), cache);
		}
		return cache;
	}

	/**
	 * Entfernt den Cache des angegebenen Containers aus der Containerverwaltung. Eventuell noch im Cache befindliche Daten werden nicht automatisch geschrieben,
	 * sondern müssen vorher durch einen expliziten Aufruf der Methode {@link de.bsvrz.ars.ars.persistence.CacheManager.Cache#flush()} geschrieben werden.
	 *
	 * @param containerFile Container dessen Cache aus der Verwaltung entfernt werden soll.
	 */
	public void forgetCache(final ContainerFile containerFile) {
		forgetCache(containerFile.getContainerId());
	}

	/**
	 * Entfernt den Cache des angegebenen Containers aus der Containerverwaltung. Eventuell noch im Cache befindliche Daten werden nicht automatisch geschrieben,
	 * sondern müssen vorher durch einen expliziten Aufruf der Methode {@link de.bsvrz.ars.ars.persistence.CacheManager.Cache#flush()} geschrieben werden.
	 *
	 * @param containerId ID des Containers dessen Cache aus der Verwaltung entfernt werden soll.
	 */
	public void forgetCache(final long containerId) {
		_debug.fine("CacheManager.forgetCache " + containerId);
		synchronized(_containerId2Cache) {
			final CacheManager.Cache cache = _containerId2Cache.remove(containerId);
			if(cache != null) aggregateCounts(cache);
		}
	}

	/**
	 * Überträgt, die verschiedenen Ereigniszähler eines Cache-Objekts in die entsprechenden Summenzähler der Cache-Verwaltung.
	 *
	 * @param cache Cache-Objekt, dessen Zählerstände übernommen werden sollen.
	 */
	private void aggregateCounts(final Cache cache) {
		_writeCountSum += cache._writeCount;
		_processedCountSum += cache._processedCount;
		_bufferedCountSum += cache._bufferedCount;
		_unbufferedCountSum += cache._unbufferedCount;
	}

	/**
	 * Liefert ein Array mit folgenden statistischen Informationen:
	 * <p>
	 * Gesamtanzahl von Datei-Schreibvorgängen aller geschlossenen Cache-Objekte.
	 * <p>
	 * Gesamtanzahl von verarbeiteten Datenblöcken aller geschlossenen Cache-Objekte.
	 * <p>
	 * Gesamtanzahl von zwischengespeicherten Datenblöcken aller geschlossenen Cache-Objekte.
	 * <p>
	 * Gesamtanzahl von nicht zwischengespeicherten Datenblöcken aller geschlossenen Cache-Objekte.
	 *
	 * @return Array mit statistischen Informationen
	 */
	public long[] getCounts() {
		synchronized(_containerId2Cache) {
			return new long[]{_writeCountSum, _processedCountSum, _bufferedCountSum, _unbufferedCountSum};
		}
	}

	/**
	 * Mit dieser Methode kann die Zwischenspeicherung für eine angegebene Datenidentifikation ein- bzw. ausgeschaltet werden.
	 *
	 * @param objectId          Objekt-ID der Datenidentifikation.
	 * @param atgId             ID der Attribugruppe der Datenidentifikation.
	 * @param aspectId          ID des Aspekts der Datenidentifikation.
	 * @param simulationVariant Simulationsvariante der Datenidentifikation.
	 * @param enable            {@code true}, wenn die Zwischenspeicherung eingeschaltetwerden soll; {@code false}, wenn die Zwischenspeicherung
	 *						  ausgeschaltet werden soll.
	 */
	public void setCachingEnabled(final long objectId, final long atgId, final long aspectId, final int simulationVariant, final boolean enable) {
		final IdDataIdentification identificationIds = new IdDataIdentification(objectId, atgId, aspectId, simulationVariant);
		_debug.fine("CacheManager.setCachingEnabled " + identificationIds + (enable ? " ENABLE" : " DISABLE"));
		synchronized(_disabledCacheIdentifications) {
			if(enable) {
				_disabledCacheIdentifications.remove(identificationIds);
			}
			else {
				_disabledCacheIdentifications.add(identificationIds);
			}
		}
	}


	public boolean isCacheEnabled() {
		return _cacheEnabled;
	}

	public void setCacheEnabled(final boolean cacheEnabled) {
		_cacheEnabled = cacheEnabled;
	}

	long getWriteCountSum() {
		return _writeCountSum;
	}

	long getProcessedCountSum() {
		return _processedCountSum;
	}

	long getBufferedCountSum() {
		return _bufferedCountSum;
	}

	long getUnbufferedCountSum() {
		return _unbufferedCountSum;
	}

	/** Realisiert den Zwischenspeicher für eine Containerdatei. */
	public static class Cache {

		private static final Debug _debug = Debug.getLogger();
		
		/** Datei in die zwischengespeicherte Daten letztendlich geschrieben werden sollen. */
		private final File _file;

		/**
		 * {@code true}, falls beim nächsten Schreibvorgang eine neue Datei erzeugt werden soll; {@code false}, false beim nächsten Schreibvorgang an die
		 * bereits vorhandenen Datei angehangen werden soll.
		 */
		private boolean _createFile;

		/** Kapazität des Zwischenspeichers in Byte */
		private final int _bufferSize;

		/** Zwischenspeicher für noch zu schreibende Bytes. */
		private final MyByteArrayOutputStream _out;

		/** Größe der ContainerDatei wie sie nach dem Schreiben der noch zwischengespeicherten Daten wäre. */
		private int _containerSize;

		/** Anzahl der bisher durchgeführten Datei-Schreibvorgänge in diesem Container. */
		private int _writeCount;

		/** Anzahl der bisher von diesem Cache bearbeiteten Datenblöcke. */
		private int _processedCount;

		/** Anzahl der bisher von diesem Cache zwischengespeicherten Datenblöcke. */
		private int _bufferedCount;

		/** Anzahl der bisher von diesem Cache nicht zwischengespeicherten Datenblöcke. */
		private int _unbufferedCount;

		/** @return Datei in die zwischengespeicherte Daten letztendlich geschrieben werden sollen. */
		File getFile() {
			return _file;
		}

		/** @return Kapazität des Zwischenspeichers in Byte */
		public int getBufferSize() {
			return _bufferSize;
		}

		/** @return Anzahl der bisher durchgeführten Datei-Schreibvorgänge in diesem Container. */
		int getWriteCount() {
			return _writeCount;
		}

		/** @return Anzahl der bisher von diesem Cache bearbeiteten Datenblöcke. */
		int getProcessedCount() {
			return _processedCount;
		}

		/** @return Anzahl der bisher von diesem Cache zwischengespeicherten Datenblöcke. */
		int getBufferedCount() {
			return _bufferedCount;
		}

		/** @return Anzahl der bisher von diesem Cache nicht zwischengespeicherten Datenblöcke. */
		int getUnbufferedCount() {
			return _unbufferedCount;
		}

		/**
		 * Erzeugt einen neuen Cache für die angegebene Datei.
		 *
		 * @param file       Datei in die zwischengespeicherte Daten letztendlich geschrieben werden sollen.
		 * @param createFile {@code true}, falls eine neue Datei erzeugt werden soll; {@code false}, false an die bereits vorhandenen Datei angehangen
		 *				   werden soll.
		 * @param bufferSize Kapazität des Zwischenspeichers in Bytes.
		 */
		private Cache(final File file, final boolean createFile, int bufferSize) {
			_debug.fine("CacheManager$Cache.Cache " + file + (createFile ? " create " : " append ") + bufferSize);
			_file = file;
			_createFile = createFile;
			_bufferSize = bufferSize;
			if(_createFile) {
				_containerSize = 0;
			}
			else {
				_containerSize = (int)file.length();
			}
			_out = new MyByteArrayOutputStream();
		}

		/**
		 * Bestimmt die Größe der ContainerDatei wie sie nach dem Schreiben der noch zwischengespeicherten Daten wäre.
		 *
		 * @return Größe der ContainerDatei in Bytes.
		 */
		public long getContainerSize() {
			_debug.fine("CacheManager$Cache.getContainerSize");
			return _containerSize;
		}

		/**
		 * Cachen der übergebenen Daten. Falls die Gesamtgröße der zwischengespeicherten Daten die Größe des Caches überschreiten würde, werden die bereits
		 * zwischengespeicherten Daten vorher in die Containerdatei geschrieben.
		 *
		 * @param data   Array mit den zu speichernden Bytes
		 * @param length Anzahl der zu speichernden Bytes im Array
		 *
		 * @throws PersistenceException Wenn beim Schreiben der Daten ein Problem aufgetreten ist.
		 */
		public void cache(byte[] data, int length) throws PersistenceException {
			_processedCount++;
			final int cachedSize = _out.size();
			if(length + cachedSize <= _bufferSize) {
				// Block passt noch in den Cache
				_bufferedCount++;
				_out.write(data, 0, length);
			}
			else {
				// Wenn Gesamtlänge nicht in den Cache passt, wird geschrieben
				if(length >= (_bufferSize / 2)) {
					// Wenn der neue Block größer oder gleich der halben Cachegröße ist, dann wird der Cache zusammen mit dem neuen Block geschrieben
					writeCacheAndDataToFile(data, length);
				}
				else {
					// Wenn der neue Block kleiner als die halbe Cachegröße ist, dann wird der Cache geschrieben und der neue Block im Cache gespeichert
					writeCacheAndDataToFile(null, 0);
					_bufferedCount++;
					_out.write(data, 0, length);
				}
			}
			_containerSize += length;
		}

		/**
		 * Schreibt die zwischengespeicherten Daten in die ContainerDatei.
		 *
		 * @throws PersistenceException Wenn beim Schreiben der Daten ein Problem aufgetreten ist.
		 */
		public void flush() throws PersistenceException {
			_debug.fine("CacheManager$Cache.flush");
			if(_out.size() > 0) {
				writeCacheAndDataToFile(null, 0);
			}
		}

		public String toString() {
			return "Cache für Container-Datei " + _file.toString();
		}

		/**
		 * Schreibt eventuell zwischengespeicherte Daten und optional zusätzlich einen weiteren Datensatz in die ContainerDatei
		 *
		 * @param data   Array mit weiteren zu schreibenden Daten oder {@code null}, falls nur die zwischengespeicherten Daten geschrieben werden sollen.
		 * @param length Anzahl der zu schreibenden Bytes im Array
		 * @throws PersistenceException Wenn beim Schreiben der Daten ein Problem aufgetreten ist.
		 */
		private void writeCacheAndDataToFile(final byte[] data, final int length) throws PersistenceException {
			ContainerFile.writeContainerFileSafely(_file.toPath(), _createFile, outputStream -> {
				_writeCount++;
				_createFile = false;
				if (_out.size() > 0) {
					// Cache schreiben
					_out.writeTo(outputStream);
					_out.reset();
				}
				if (data != null) {
					_unbufferedCount++;
					outputStream.write(data, 0, length);
				}
			});
		}
	}

	/**
	 * Speicherverbrauch des Caches
	 */
	public record CacheMemoryUsage(long totalBytesUsed, long cachedBytes) {
	}

	private static class MyByteArrayOutputStream extends ByteArrayOutputStream {
		public int bufSize() {
			return buf.length;
		}
	}
}
