/*
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */
package de.bsvrz.ars.ars.persistence;

import de.bsvrz.ars.ars.persistence.directories.PersistenceDirectory;
import de.bsvrz.ars.ars.persistence.index.DataIndexIndex;
import de.bsvrz.ars.ars.persistence.index.DataTimeIndex;
import de.bsvrz.ars.ars.persistence.index.IndexException;
import de.bsvrz.ars.ars.persistence.index.IndexValues;
import de.bsvrz.ars.ars.persistence.index.result.AggregateIndexResult;
import de.bsvrz.ars.ars.persistence.index.result.IndexResult;
import de.bsvrz.ars.ars.persistence.index.result.LocatedIndexResult;
import de.bsvrz.ars.ars.persistence.sequence.AllDataSpecification;
import de.bsvrz.ars.ars.persistence.sequence.DataIndexSequenceSpecification;
import de.bsvrz.ars.ars.persistence.sequence.SequenceSpecification;

import java.util.List;
import java.util.Map;

/**
 * Klasse, die mehrere DataIndexIndex-Objekte zusammenfasst und als einen einzigen Index behandelt. Damit werden
 * Archivanfragen und ähnliche Operationen über mehrere Wochenverzeichnisse hinweg unterstützt.
 */
public class CompoundDataIndexIndex {
	private final List<DataIndexIndex> indexes;
	private final Map<DataIndexIndex, PersistenceDirectory> directories;

	/**
	 * Erstellt eine neuen CompoundDataIndexIndex.
	 *
	 * @param indexes     Zu verkettende Indexe in datenindexmäßig aufsteigender Reihenfolge
	 * @param directories Zuordnung der Indexe zu den zugehörigen Verzeichnissen. Alle Elemente von indexes müssen enthalten sein.
	 */
	public CompoundDataIndexIndex(List<DataIndexIndex> indexes, Map<DataIndexIndex, PersistenceDirectory> directories) {
		this.indexes = indexes;
		this.directories = directories;
	}

	public CompoundDataIndexIndex(IndexAggregator<DataIndexIndex> aggregator) {
		this(aggregator.getIndexes(), aggregator.getMap());
	}

	/**
	 * Gibt die Indexwerte eines Datenindexbereichs zurück.
	 *
	 * @param minimumTime Minimum-Datenindex
	 * @param maximumTime Maximum-Datenindex
	 * @return die Container eines Datenindexbereichs
	 * @throws IndexException Lesefehler
	 */
	public LocatedIndexResult<IndexValues> getContainerIDByDataIndex(long minimumTime, long maximumTime) throws IndexException {
		AggregateIndexResult<IndexValues> result = new AggregateIndexResult<>();
		for (DataIndexIndex index : indexes) {
			IndexResult<IndexValues> time = index.getContainerIDByDataIndex(minimumTime, maximumTime);
			result.add(time, directories.get(index));
		}
		return result;
	}

	/**
	 * Gibt zurück, ob diese Version des Indexes den gesamten übergebenen Datenbereich umfasst. Dabei muss
	 * ggf. auch ein Anfangszustand oder ein Endzustand enthalten sein. Falls dieser möglicherweise fehlt, wird false
	 * zurückgegeben, und der Aufrufer muss evtl. einen größeren Bereich der Persistenz analysieren.
	 * <p>
	 * Diese Methode wird für die Optimierung bei Archivanfragen und ähnlichen Operationen verwendet, wo
	 * im Normalfall Anhand der Verzeichnisstruktur nut bestimmte Wochenverzeichnisse benötigt werden.
	 * Erst wenn hier festgestellt wird, dass nicht alle benötigten Daten vorliegen (hier also false zurückgegeben
	 * wird), dann müssen ggf. mehr oder alle Wochenverzeichnisse analysiert werden.
	 * </p>
	 *
	 * @param sequenceSpecification Ursprüngliche Zeitspezifikation. Diese Methode unterstützt nur {@link AllDataSpecification} und
	 *                              {@link DataIndexSequenceSpecification}. Bei anderen Spezifikationen ist keine Aussage
	 *                              möglich.
	 * @return true: Dieser zusammengefasste Index enthält sicher alle benötigten Daten, false: Es
	 * fehlen ggf. weitere Daten
	 * @throws IndexException Lesefehler bei Abfrage des Indexes
	 */
	public boolean isComplete(SequenceSpecification sequenceSpecification) throws IndexException {
		if (sequenceSpecification instanceof AllDataSpecification) {
			return true;
		}
		if (sequenceSpecification instanceof DataIndexSequenceSpecification specification) {
			if (indexes.get(0).getFirst(IndexValues.DataIndexMin) > specification.minimumIndex()) {
				// Optimierung fehlgeschlagen, der Anfangszustand fehlt.
				return false;
			}
			if (indexes.get(indexes.size() - 1).getLast(IndexValues.DataIndexMax) < specification.maximumIndex()) {
				// Optimierung fehlgeschlagen, der Endzustand fehlt.
				return false;
			}
			return true;
		}
		throw new UnsupportedOperationException();
	}
}
