/*
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */
package de.bsvrz.ars.ars.persistence;

import de.bsvrz.ars.ars.persistence.directories.PersistenceDirectory;
import de.bsvrz.ars.ars.persistence.index.DataTimeIndex;
import de.bsvrz.ars.ars.persistence.index.IndexException;
import de.bsvrz.ars.ars.persistence.index.IndexValues;
import de.bsvrz.ars.ars.persistence.index.result.AggregateIndexResult;
import de.bsvrz.ars.ars.persistence.index.result.IndexResult;
import de.bsvrz.ars.ars.persistence.index.result.LocatedIndexResult;

import java.util.Map;

/**
 * Klasse, die mehrere DataTimeIndex-Objekte zusammenfasst und als einen einzigen Index behandelt. Damit werden
 * Archivanfragen und ähnliche Operationen über mehrere Wochenverzeichnisse hinweg unterstützt.
 */
public class CompoundDataTimeIndex {
	private final Map<DataTimeIndex, PersistenceDirectory> subIndexes;

	/**
	 * Erstellt eine neuen CompoundDataTimeIndex.
	 *
	 * @param subIndexes Indexe mit zugehörigen Verzeichnissen, die verkettet werden sollen. Die verwendete Map muss
	 *                   zeitlich aufsteigend sortiert sein (z. B. LinkedHashMap)
	 */
	public CompoundDataTimeIndex(Map<DataTimeIndex, PersistenceDirectory> subIndexes) {
		this.subIndexes = subIndexes;
	}

	public CompoundDataTimeIndex(IndexAggregator<DataTimeIndex> aggregator) {
		this(aggregator.getMap());
	}

	/**
	 * Gibt die Indexwerte eines Datenzeitbereiches zurück.
	 *
	 * @param minimumTime Minimum-Datenzeit in Epoch-Millis
	 * @param maximumTime Maximum-Datenzeit in Epoch-Millis
	 * @return die Container eines Datenzeitbereiches
	 * @throws IndexException Lesefehler
	 */
	public LocatedIndexResult<IndexValues> getContainerIDByDataTime(long minimumTime, long maximumTime) throws IndexException {
		AggregateIndexResult<IndexValues> result = new AggregateIndexResult<>();
		for (Map.Entry<DataTimeIndex, PersistenceDirectory> e : subIndexes.entrySet()) {
			IndexResult<IndexValues> time = e.getKey().getContainerIDByDataTime(minimumTime, maximumTime);
			result.add(time, e.getValue());
		}
		return result;
	}
}
