/*
 *
 * Copyright 2005-2008 by beck et al. projects GmbH, Munich
 * Copyright 2009-2020 by Kappich Systemberatung, Aachen
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.ars.ars.persistence;

import de.bsvrz.ars.ars.mgmt.tasks.AbstractTask;
import de.bsvrz.ars.ars.mgmt.tasks.ArchiveDataSerializer;
import de.bsvrz.dav.daf.main.archive.ArchiveDataKind;
import de.bsvrz.dav.daf.util.BufferedRandomAccessFile;
import de.bsvrz.dav.daf.util.CloseableRandomAccessFile;
import de.bsvrz.dav.daf.util.FileAccess;
import de.bsvrz.sys.funclib.kappich.annotations.NotNull;
import de.bsvrz.sys.funclib.kappich.annotations.Nullable;
import de.bsvrz.sys.funclib.losb.util.ByteIO;
import de.bsvrz.sys.funclib.losb.util.Util;

import java.io.*;
import java.nio.file.*;
import java.util.regex.Pattern;

import static java.nio.charset.StandardCharsets.ISO_8859_1;
import static java.nio.file.StandardOpenOption.APPEND;
import static java.nio.file.StandardOpenOption.CREATE;

/**
 * Diese Klasse enthält die Funktionalitaet zum Erzeugen, Lesen, Schreiben und Loeschen von Container-Dateien. Jeder Task, der mit Container-Dateien arbeitet,
 * enthält ein Objekt dieser Klasse. Die Sichtbarkeit von Konstruktor und Methoden ist beschraenkt, da diese Klasse nur durch den {@link
 * de.bsvrz.ars.ars.persistence.PersistenceManager} im gleichen Package benutzt wird. Dieser hält für jede anfragende {@link AbstractTask}
 * ein Objekt dieser Klasse und leitet Anfragen bzgl. Container-Dateien an dieses {@code ContainerFile}-Objekt weiter.
 *
 * @author beck et al. projects GmbH
 * @author Thomas Schaefer
 * @author Alexander Schmidt
 * @version $Revision$ / $Date$ / ($Author$)
 */
public final class ContainerFile {

	/** 4 Byte langes Versionskennzeichen. */
	static final byte[] VERSION_STRING = {'v', '1', '0', '0'};

	/** Länge des Längenbytes vor jedem Datensatz (4 Bytes, Integer). */
	public static final int DATALEN_LEN = ByteIO.INT4B_LEN;

	/** Länge des Headers eines jeden Datensatzes. Archivzeitstempel (6B), Datenzeitstempel (6B), Datenindex (8B), Prüfsumme (4B). */
	static final int DATAHDR_LEN = (2 * ByteIO.LONG6B_LEN) + ByteIO.LONG8B_LEN + ByteIO.INT4B_LEN;

	/** Länge des Compress-Info-Feldes. Compress-Info enthält 0 bei unkomprimierten Datensatz, sonst dekomprimierte Länge des komprimierten Datensatzes. */
	static final int COMPRESS_LEN = ByteIO.INT4B_LEN;

	/** Wert des Compress-Info-Feldes, wenn DS nicht komprimiert ist */
	public static final int NOT_COMPRESSED = 0;

	/** Datensatzbytes, falls "keine Quelle" signalisiert wurde */
	public static final byte[] NO_SOURCE = {'N', 'O', '_', 'S', 'R', 'C'};

	/** Datensatzbytes, falls "keine Daten" signalisiert wurde */
	public static final byte[] NO_DATA = {'N', 'O', 'D', 'A', 'T', 'A'};

	/** Datensatzbytes, falls "keine Rechte" signalisiert wurde */
	public static final byte[] NO_RIGHTS = {'N', 'O', 'R', 'I', 'G', 'H'};

	/** Datensatzbytes um eine potentielle Datenlücke zu kennzeichnen */
	public static final byte[] POT_GAP = {'P', 'O', 'T', 'G', 'A', 'P'};
	
	/**
	 * Datensätze bis zu dieser Länge bleiben immer unkomprimiert. <b>Muss mindestens so gross sein wie {@link #NO_SOURCE}, {@link #NO_DATA}, {@link #POT_GAP}
	 * lang sind!</b> <p>Auf {@code Integer.MAX_VALUE} setzen, um die Komprimierung zu deaktivieren.
	 */
	public static int MAX_UNCOMPRESSED = 64;

	/** Regulaerer Ausdruck zur Erkennung von Containerdateinamen. */
	public static final Pattern CONT_FILENAME_PAT = Pattern.compile("dc(\\d{13})\\.dat");

	/** Filter zur Erkennung von Containerdateien anhand des Dateinamens. */
	public static final FilenameFilter CONT_FILENAME_FILTER = (dir, name) -> CONT_FILENAME_PAT.matcher(name).matches();

	/**
	 * Größe der Daten-Puffer in Byte. Wenn eine Anfrage 10000 Unteranfragen enthält, werden auch 10000 ContainerFile-Objekte angelegt. Drum nicht zu gross
	 * wählen. Bei Bedarf werden Größere Puffer extra angelegt.
	 */
	public static final int BUFFER_SIZE = 4 * 1024;

	//
	// ENDE STATISCHE FELDER
	////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

	/** Puffer zum Einlesen von Längenbytes etc. */
	private final byte[] byte8Buf = new byte[8];

	/**
	 * Container-Header Parameter als Key/Value-Paare. Die Parameter sind in {@link KeyValParam} statisch deklariert. Als Key eines Parameters wird {@link
	 * KeyValParam#getKey()} benutzt.
	 */
	private final ContainerHdr containerHdr = new ContainerHdr();

	/** Aktuelle Container-ID. */
	private long containerId;

	/** Aktuelle Container-Datei. */
	@Nullable
	private Path contFile;

	/** Container read-only adressiert. */
	private boolean readOnly;

	/** Container-Header eingelesen. */
	private boolean headerRead; // Initial kein Header eingelesen.

	private final CacheManager _cacheManager = CacheManager.getInstance();

	@Nullable
	private CacheManager.Cache _cache;

	private static final int HEADER_LENGTH = VERSION_STRING.length + DATALEN_LEN + ContainerHdr.HDR_TXT_LEN + ByteIO.SEPARATOR.length;

	/**
	 * Gibt {@code true} zurück, wenn es sich bei der angegebenen Datei (laut Dateinamenschema) um eine Containerdatei handelt.
	 *
	 * @param it Datei
	 * @return {@code true}, wenn es sich bei der angegebenen Datei (laut Dateinamenschema) um eine Containerdatei handelt, sonst {@code false}
	 */
	public static boolean isContainerFile(Path it) {
		return CONT_FILENAME_PAT.matcher(it.getFileName().toString()).matches();
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
	////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

	private boolean accessed() {
		return contFile != null;
	}
	
	/*
	 * @see java.lang.Object#toString()
	 */
	public String toString() {
		if (accessed()) {
			if (readOnly) {
				return "[accessed read only: file='" + contFile.toAbsolutePath() + "']";
			} else {
				return "[accessed: cId=" + containerId + ",file='" + contFile.toAbsolutePath() + "']";
			}
		}
		else {
			return "[unaccessed]";
		}
	}

	//////////////////////////////////////////////////////////////////////////////////////////////////////////
	// Container: Hilfsmethoden Dateiname
	//////////////////////////////////////////////////////////////////////////////////////////////////////////

	/**
	 * Liefert die Container-ID zur angegebenen Container-Datei, indem die im Container-Dateinamen enthaltene Container-ID ausgelesen wird. Alternativ kann die
	 * Container-ID aus den Container-Header-Parametern gelesen werden; siehe dazu {@link #getContHdrParamAsLong(KeyValParam)}.
	 *
	 * @param file Container-Datei
	 * @return Container-ID
	 */
	public static long getContID(Path file) {
		return getContID(file.getFileName().toString());
	}

	/**
	 * Liefert die Container-ID zum angegebenen Container-Datei-Namen, indem die im Container-Dateinamen enthaltene Container-ID ausgelesen wird. Alternativ
	 * kann die Container-ID aus den Container-Header-Parametern gelesen werden; siehe dazu {@link #getContHdrParamAsLong(KeyValParam)}.
	 *
	 * @param fileName Container-Datei
	 * @return Container-ID
	 */
	public static long getContID(String fileName) {
		if(fileName.length() == 19) {
			if(fileName.startsWith("dc") && fileName.endsWith(".dat")){
				try {
					return Long.parseLong(fileName.substring(2, 15));
				}
				catch(NumberFormatException e) {
					throw new IllegalArgumentException("Ungültiger Container-Dateiname: " + fileName, e);
				}
			}
		}
		throw new IllegalArgumentException("Ungültiger Container-Dateiname: " + fileName + " Länge: " + fileName.length()+ ", Erwartet: 19");
	}

	/**
	 * Liefert den Dateinamen des Daten-Containers mit der angegebenen Container-ID.
	 *
	 * @param containerId Container-ID
	 * @return Dateiname
	 */
	public static String getContainerFileName(long containerId) {
		// Container-ID ist 5-Byte-Ganzzahl, d.h. max 13-stellig dezimal
		final char[] chars = {'d', 'c', '0', '0', '0', '0', '0', '0', '0', '0', '0', '0', '0', '0', '0', '.', 'd', 'a', 't'};
		final String containerIdString = Long.toString(containerId);
		final int containerIdStringLength = containerIdString.length();
		containerIdString.getChars(0, containerIdStringLength, chars, 15 - containerIdStringLength);
		return new String(chars);
	}

	//////////////////////////////////////////////////////////////////////////////////////////////////////////
	// Container: Access/Create/Close
	//////////////////////////////////////////////////////////////////////////////////////////////////////////

	/**
	 * Definiert auf welchem Container die folgenden Methoden-Aufrufe wirksam sein sollen. Muss immer aufgerufen werden bevor mit einem Container gearbeitet
	 * werden kann. Der Containerdateiname wird aus der Container-ID gebildet. Soll der Zugriff auf diesen Container beendet werden (um z.B. einen anderen
	 * Container zu adressieren), muss dazu die Methode {@link #leaveContainer()} aufgerufen werden.
	 *
	 * @param contId       Container-ID (muss zwischen 0 und 2^40-1 liegen)
	 * @param containerDir Pfad des Verzeichnisses, das die Container-Datei enthält
	 * @param readonly     Nur schreibgeschützt öffnen? In dem Fall wird z. B. kein Schreibcache initialisiert
	 * @throws PersistenceException Lesefehler im Persistenzverzeichnis
	 */
	void accessContainer(long contId, Path containerDir, final boolean readonly) throws PersistenceException {
		if (accessed()) {
//			lastAccessed.printStackTrace();
			throw new PersistenceException(
					"accessContainer(" + contId + "," + containerDir + ") nicht möglich: bereits Container im Zugriff: " + this
			);
		}

		this.containerId = contId;
		this.contFile = containerDir.resolve(getContainerFileName(contId));
		this.readOnly = readonly;
		this.headerRead = false; // Header noch nicht eingelesen.
		if(readonly) {
			this._cache = _cacheManager.getCache(this);
			if(_cache != null) {
				_cache.flush();
				// Ein lesender Task braucht den Cache im weiteren Verlauf nicht mehr
				this._cache = null;
			}
		}
		else {
			this._cache = _cacheManager.getCache(this);
		}
	}

	/**
	 * Definiert auf welchem Container die folgenden Methoden-Aufrufe wirksam sein sollen. Es können nur lesende Operationen auf die Container-Datei ausgeführt
	 * werden. Vorgesehen für den Zugriff auf Container, die bereits auf ein Medium vom Typ B ausgelagert sind. Soll der Zugriff auf diesen Container beendet
	 * werden (um z.B. einen anderen Container zu adressieren), muss dazu die Methode {@link #leaveContainer()} aufgerufen werden. Die Container-Datei kann einen
	 * beliebigen Namen haben.
	 *
	 * @param contFile Container-Datei (muss existieren)
	 * @throws PersistenceException Lesefehler im Persistenzverzeichnis
	 */
	void accessContainerReadOnly(Path contFile) throws PersistenceException {
		if (accessed()) {
			throw new PersistenceException("accessContainerReadOnly(" + contFile.toAbsolutePath() + ") nicht möglich: bereits Container im Zugriff\n" + this);
		}
		this.containerId = getContID(contFile);
		this.contFile = contFile;
		this.readOnly = true;  // Nur lesen
		this.headerRead = false; // Header noch nicht eingelesen.
		this._cache = _cacheManager.getCache(this);
		if(_cache != null) {
			_cache.flush();
			// Ein lesender Task braucht den Cache im weiteren Verlauf nicht mehr
			this._cache = null;
		}
	}

	private void forgetCache() {
		_cacheManager.forgetCache(this);
		_cache = null;
	}


	/**
	 * Beendet das Arbeiten mit dem zuvor über eine der {@code accessContainer()}-Methoden definierten Container. Diese Methode muss aufgerufen werden, wenn
	 * danach auf einen anderen Container zugegriffen werden soll. Die Methode prüft nicht, ob ein Container vorher mit {@link #accessContainer(long, Path, boolean)} definiert wurde. D.h., man kann diese Methode mehrfach aufrufen. Dadurch wird die Ausnahmebehandlung
	 * erleichtert.
	 *
	 */
	void leaveContainer() {
		_cache = null;
		containerId = -1;
		contFile = null;
		containerHdr.clear();    // Header-Info loeschen
		headerRead = false;     // Kein Header mehr eingelesen.
	}

	/**
	 * Liefert Kennzeichen, ob bereits eine Container-Datei für den mit {@link #accessContainer(long, Path, boolean)}
	 * spezifizierten Container besteht.
	 *
	 * @return Kennzeichen
	 * @throws PersistenceException Lesefehler im Persistenzverzeichnis
	 */
	boolean existsContainer() throws PersistenceException {
		return _cache != null || Files.exists(checkContainerAccessed());
	}

	/**
	 * Erzeugt eine neue Container-Datei für den mit accessContainer() spezifizierten Container. Schreibt den Container-Header und befuellt diesen mit den
	 * angegebenen Parametern.
	 *
	 * @param location Ort des Containers
	 * @throws PersistenceException Lesefehler im Persistenzverzeichnis
	 */
	void createContainer(final ContainerDirectory location) throws PersistenceException {
		Path contFile = checkContainerAccessedRW();
		if (Files.exists(contFile))
			throw new PersistenceException("Erzeugen der Container-Datei fehlgeschlagen: Datei existiert bereits\n" + this);
		if (_cache != null) {
			forgetCache();
		}
		_cache = _cacheManager.createCache(this, true, location.dataIdentification());
		createDefaultContainerHeader(location); // Neuen Container-Header erzeugen
		writeInitialContainerHeader();          // Neuen Container-Header in neue Datei schreiben
	}

	/**
	 * Schliesst die Container-Datei für den mit einer der {@code accessContainer()}-Methoden spezifizierten Container. Der Container-Header wird entsprechend
	 * aktualisiert.
	 *
	 * @param openContainerData     Daten des offenen Containers, die in den Header eingetragen werden sollen
	 * @throws PersistenceException Lesefehler im Persistenzverzeichnis
	 */
	public void closeContainer(
			final StandardOpenContainerData openContainerData)
			throws PersistenceException {
		checkContainerAccessedRW();
		if (_cache != null) {
			_cache.flush();
			forgetCache();
		}
		if (!existsContainer()) {
			System.out.println("openContainerData = " + openContainerData);
		}
		ensureHeaderRead();
		// Container-Header-Parameter im Speicher setzen:
		setContHdrParam(ContainerHdr.CHP_ANZ_DS, openContainerData.getNumContainerEntries());
		setContHdrParam(ContainerHdr.CHP_ARC_TIME_MIN, openContainerData.getMinArcTime());
		setContHdrParam(ContainerHdr.CHP_ARC_TIME_MAX, openContainerData.getMaxArcTime());
		setContHdrParam(ContainerHdr.CHP_DATA_TIME_MIN, openContainerData.getMinDataTime());
		setContHdrParam(ContainerHdr.CHP_DATA_TIME_MAX, openContainerData.getMaxDataTime());
		setContHdrParam(ContainerHdr.CHP_DATA_IDX_MIN, openContainerData.getMinDataIdx());
		setContHdrParam(ContainerHdr.CHP_DATA_IDX_MAX, openContainerData.getMaxDataIdx());
		// Alle geänderten Parameter zusammen in Container-Header speichern:
		writeContainerHeader();
	}

	/**
	 * Prüft ob der mit einer der {@code accessContainer()}-Methoden spezifizierte Container bereits abgeschlossen ist.
	 *
	 * @return Kennzeichen, ob Container abgeschlossen ist
	 * @throws PersistenceException Lesefehler im Persistenzverzeichnis
	 */
	boolean isContainerClosed() throws PersistenceException {
		checkContainerAccessed();
		ensureHeaderRead();
		return getContHdrParamAsInt(ContainerHdr.CHP_ANZ_DS) >= 0; // Abgeschlossener Container fuehrt die Anzahl seiner Datensätze im Header
	}

	/**
	 * Stellt sicher dass der Header eingelesen wurde. Liest ihn ein, falls noch nicht geschehen.
	 *
	 * @throws PersistenceException Lesefehler im Persistenzverzeichnis
	 */
	public void ensureHeaderRead() throws PersistenceException {
		if(!headerRead) readContainerHeader();
	}

	//////////////////////////////////////////////////////////////////////////////////////////////////////////
	// Container: Header
	//////////////////////////////////////////////////////////////////////////////////////////////////////////

	/**
	 * Erzeugt Default-Header-Parameter für den Anfang einer neuen Container-Datei. Bevor mit {@link #writeContainerHeader()} der Header geschrieben werden
	 * kann, muss entweder mit dieser Methode ein neuer Header erzeugt oder mit {@link #readContainerHeader()} ein Header eingelesen werden.
	 *
	 * @param location Containerverzeichnis
	 */
	private void createDefaultContainerHeader(final ContainerDirectory location) throws PersistenceException {
		containerHdr.clear();

		// zu setzende Werte:
		containerHdr.setVal(ContainerHdr.CHP_CONT_ID, containerId);
		containerHdr.setVal(ContainerHdr.CHP_OBJ_ID, location.getObjectId());
		containerHdr.setVal(ContainerHdr.CHP_ATG_ID, location.getAtgId());
		containerHdr.setVal(ContainerHdr.CHP_ASP_ID, location.getAspectId());
		containerHdr.setVal(ContainerHdr.CHP_SIM_VAR, location.getSimVariant());
		containerHdr.setVal(ContainerHdr.CHP_DATA_KIND, location.archiveDataKind());

		// Default-Werte:
		containerHdr.setDefaultVal(ContainerHdr.CHP_ANZ_DS);
		containerHdr.setDefaultVal(ContainerHdr.CHP_DATA_IDX_MIN);
		containerHdr.setDefaultVal(ContainerHdr.CHP_DATA_IDX_MAX);
		containerHdr.setDefaultVal(ContainerHdr.CHP_DATA_TIME_MIN);
		containerHdr.setDefaultVal(ContainerHdr.CHP_DATA_TIME_MAX);
		containerHdr.setDefaultVal(ContainerHdr.CHP_ARC_TIME_MIN);
		containerHdr.setDefaultVal(ContainerHdr.CHP_ARC_TIME_MAX);

		// Obsolete Parameter
		containerHdr.setDefaultVal(ContainerHdr.CHP_LOESCHEN);
		containerHdr.setDefaultVal(ContainerHdr.CHP_MEDIUM_ID);
		containerHdr.setDefaultVal(ContainerHdr.CHP_TO_SAVE);
		containerHdr.setDefaultVal(ContainerHdr.CHP_DELETED);
		containerHdr.setDefaultVal(ContainerHdr.CHP_LOESCHUTZ);
		containerHdr.setDefaultVal(ContainerHdr.CHP_RESTORED);
		headerRead = true;
	}

	/**
	 * Schreibt die aktuellen Header-Parameter an den Anfang der Container-Datei. Bei einer neuen leeren Datei werden die Parameter an den Anfang der Datei
	 * angehangen, bei einer zu ändernden Container-Datei werden die vorhandenen Parameter überschrieben. Bevor mit dieser Methode der Header geschrieben
	 * werden kann, muss entweder mit {@link #createDefaultContainerHeader(ContainerDirectory)} neuer Header erzeugt oder mit {@link #readContainerHeader()} ein Header
	 * eingelesen worden sein.
	 *
	 * @throws PersistenceException Lesefehler im Persistenzverzeichnis
	 */
	public void writeContainerHeader() throws PersistenceException {
		Path contFile = checkContainerAccessedHeaderRead();
		String hdrTxt = containerHdr.writeContainerHdr();
		ByteIO.writeSignedInt4Bytes(byte8Buf, 0, hdrTxt.length() + ByteIO.SEPARATOR.length); // Header-Länge (4 Bytes)
		CloseableRandomAccessFile raf = null;
		try {
			ByteArrayOutputStream out = new ByteArrayOutputStream(getHeaderLen());
			out.write(VERSION_STRING);                      // Header-Version
			out.write(byte8Buf, 0, DATALEN_LEN);            // Header-Länge schreiben (aus readBuf)
			out.write(hdrTxt.getBytes(ISO_8859_1));         // Header-Parameter schreiben
			out.write(ByteIO.SEPARATOR);                    // Header-Seperator schreiben
			raf = new CloseableRandomAccessFile(contFile.toFile());  // Nur ReadWrite, kein flush
			final CloseableRandomAccessFile finalRaf = raf;
			// Inhalt des Puffers ohne neue Kopie des Byte-Arrays schreiben
			out.writeTo(
					new OutputStream() {
						@Override
						public void write(int b) throws IOException {
							// Wird nicht aufgerufen; nur der Vollständigkeit halber implementiert.
							finalRaf.write(b);
						}

						@Override
						public void write(@NotNull byte[] b, int off, int len) throws IOException {
							finalRaf.write(b, off, len);
						}
					}
			);
		}
		catch(IOException e) {
			throw new PersistenceException("Container-Header-Parameter konnte nicht geaendert werden: " + e.getMessage() + "\n" + this, e);
		}
		finally {
			closeRandomAccessFile(raf);
		}
	}

	/**
	 * Schreibt die aktuellen Header-Parameter an den Anfang der Container-Datei. Bei einer neuen leeren Datei werden die Parameter an den Anfang der Datei
	 * angehangen, bei einer zu ändernden Container-Datei werden die vorhandenen Parameter überschrieben. Bevor mit dieser Methode der Header geschrieben
	 * werden kann, muss entweder mit {@link #createDefaultContainerHeader(ContainerDirectory)} neuer Header erzeugt oder mit {@link #readContainerHeader()} ein Header
	 * eingelesen worden sein.
	 *
	 * @throws PersistenceException Lesefehler im Persistenzverzeichnis
	 */
	private void writeInitialContainerHeader() throws PersistenceException {
		Path contFile = checkContainerAccessedHeaderRead();
		String hdrTxt = containerHdr.writeContainerHdr();
		ByteIO.writeSignedInt4Bytes(byte8Buf, 0, hdrTxt.length() + ByteIO.SEPARATOR.length); // Header-Länge (4 Bytes)
		if(_cache != null) {
			_cache.cache(VERSION_STRING, VERSION_STRING.length);
			_cache.cache(byte8Buf, DATALEN_LEN);
			final byte[] hdrBytes = hdrTxt.getBytes(ISO_8859_1);
			_cache.cache(hdrBytes, hdrBytes.length);
			_cache.cache(ByteIO.SEPARATOR, ByteIO.SEPARATOR.length);
			// Cache wird später nach der Zwischenspeicherung des ersten Datensatzes geschrieben und damit im Filesystem sichtbar, weil manche Klassen das
			// Filesystem als Index verwenden (z.B. MultiContainerDataIterator)
		} else {
			try (BufferedOutputStream out = new BufferedOutputStream(Files.newOutputStream(contFile), getHeaderLen())) {
				out.write(VERSION_STRING);                  // Header-Version
				out.write(byte8Buf, 0, DATALEN_LEN);        // Header-Länge schreiben (aus readBuf)
				out.write(hdrTxt.getBytes(ISO_8859_1));     // Header-Parameter schreiben
				out.write(ByteIO.SEPARATOR);                // Header-Seperator schreiben
				out.flush();
			}
			catch(IOException e) {
				throw new PersistenceException("Container-Header-Parameter konnte nicht geaendert werden: " + e.getMessage() + "\n" + this, e);
			}
		}
	}

	void readContainerHeader(FileAccess raf) throws PersistenceException {
		try {
			raf.readFully(byte8Buf, 0, VERSION_STRING.length + ByteIO.INT4B_LEN);
			int headerLength = ByteIO.readSignedInt4Bytes(byte8Buf, VERSION_STRING.length); // Länge Header und Seperator

			byte[] buf = new byte[headerLength];
			raf.readFully(buf, 0, headerLength);
			containerHdr.readContainerHdr(buf, headerLength);
			headerRead = true; // Header eingelesen
		}
		catch(IOException e) {
			throw new PersistenceException("Container-Header konnte nicht gelesen werden: " + e.getMessage() + "\n" + this, e);
		}
	}

	/**
	 * Liest den Header der Container-Datei ein und springt im Eingabestrom an die Stelle hinter dem Header. Die Container-Header Parameter stehen dann als
	 * Key/Value-Paare zur Verfuegung und können mit {@link #getContHdrParamAsLong(KeyValParam)} usw. abgefragt werden. Bevor mit {@link
	 * #writeContainerHeader()} der Header geschrieben werden kann, muss entweder mit dieser Methode der Header eingelesen oder mit {@link
	 * #createDefaultContainerHeader(ContainerDirectory)} ein neuer Header erzeugt werden.
	 *
	 * @throws PersistenceException Lesefehler im Persistenzverzeichnis
	 */
	void readContainerHeader() throws PersistenceException {
		if (_cache != null) _cache.flush();
		try {
			Path contFile = checkContainerAccessed();
			if (!Files.exists(contFile)) {
				throw new PersistenceException("Container-Header konnte nicht gelesen werden: Datei existiert nicht: " + contFile + "\n" + this);
			}
			try (CloseableRandomAccessFile raf = new CloseableRandomAccessFile(contFile.toFile())) {
				readContainerHeader(raf);
			}
		}
		catch(PersistenceException e) {
			throw e;
		}
		catch(Exception e) {
			throw new PersistenceException("Container-Header konnte nicht gelesen werden: " + e.getMessage() + "\n" + this, e);
		}
	}


	/**
	 * Nachdem mit {@link #readContainerHeader()} der Header einer Container-Datei eingelesen worden ist, kann mit dieser Methode der Wert eines
	 * Header-Parameters ermittelt werden. Wirft eine {@link PersistenceException}, falls der Parameter nicht gefunden wurde. Liefert den Wert des Parameters
	 * unabhaengig von dessen Typ immer als String zurück.
	 *
	 * @param param Container-Header-Parameter
	 * @return Parameter-Wert als String
	 * @throws PersistenceException Lesefehler im Persistenzverzeichnis
	 */
	String getContHdrParamAsString(KeyValParam param) throws PersistenceException {
		checkContainerAccessedHeaderRead();
		String val = containerHdr.getValAsString(param);
		if(val != null) return val;
		throw new PersistenceException("ContainerHeaderParam nicht vorhanden: " + param + "\n" + this);
	}

	/**
	 * Nachdem mit {@link #readContainerHeader()} der Header einer Container-Datei eingelesen worden ist, kann mit dieser Methode der Wert eines
	 * Header-Parameters ermittelt werden. Liefert den Wert des Parameters als Datensatzart zurück. Wirft eine {@link PersistenceException}, falls der
	 * Parameter nicht gefunden wurde oder nicht in den Typ {@link ArchiveDataKind} umgeformt werden kann.
	 *
	 * @param param Container-Header-Parameter
	 * @return Datensatzart
	 * @throws PersistenceException Lesefehler im Persistenzverzeichnis
	 */
	ArchiveDataKind getContHdrParamAsArchiveDataKind(KeyValParam param) throws PersistenceException {
		checkContainerAccessedHeaderRead();
		ArchiveDataKind adk = containerHdr.getValAsArchiveDataKind(param);
		if(adk != null) return adk;
		throw new PersistenceException("ContainerHeaderParam nicht besetzt: " + param + "\n" + this);
	}

	/**
	 * Nachdem mit {@link #readContainerHeader()} der Header einer Container-Datei eingelesen worden ist, kann mit dieser Methode der Wert eines
	 * Header-Parameters ermittelt werden. Liefert den Wert des Parameters als numerischen Wert vom Typ {@code long} zurück. Wirfte eine {@link
	 * PersistenceException}, falls der Parameter nicht gefunden wurde oder nicht in den Typ {@code long} umgeformt werden kann.
	 *
	 * @param param Container-Header-Parameter
	 * @return Ganzzahl als {@code long}
	 * @throws PersistenceException Lesefehler im Persistenzverzeichnis
	 */
	long getContHdrParamAsLong(KeyValParam param) throws PersistenceException {
		checkContainerAccessedHeaderRead();
		Long l = containerHdr.getValAsLong(param);
		if(l != null) return l;
		throw new PersistenceException("ContainerHeaderParam nicht besetzt: " + param + "\n" + this);
	}

	int getContHdrParamAsInt(KeyValParam param) throws PersistenceException {
		checkContainerAccessedHeaderRead();
		Integer i = containerHdr.getValAsInt(param);
		if(i != null) return i;
		throw new PersistenceException("ContainerHeaderParam nicht besetzt: " + param + "\n" + this);
	}

	boolean getContHdrParamAsBool(KeyValParam param) throws PersistenceException {
		checkContainerAccessedHeaderRead();
		Boolean b = containerHdr.getValAsBool(param);
		if(b != null) return b;
		throw new PersistenceException("ContainerHeaderParam nicht besetzt: " + param + "\n" + this);
	}

	/**
	 * Setzt einen Parameter im Container-Header. Der Header muss vorher entweder gelesen oder mit Default-Werten erzeugt worden sein. Wirft eine {@link
	 * PersistenceException}, wenn die Änderung nicht durchgefuehrt werden kann.
	 *
	 * @param param Container-Header-Parameter
	 * @param val   Numerischer Wert des Container-Header-Parameters
	 * @throws PersistenceException Lesefehler im Persistenzverzeichnis
	 */
	void setContHdrParam(KeyValParam param, long val) throws PersistenceException {
		checkContainerAccessedHeaderRead();
		containerHdr.setVal(param, val);
	}

	/**
	 * Setzt einen Parameters im Container-Header. Der Header muss vorher entweder gelesen oder mit Default-Werten erzeugt worden sein. Wirft eine {@link
	 * PersistenceException}, wenn die Änderung nicht durchgefuehrt werden kann.
	 *
	 * @param param Container-Header-Parameter
	 * @param val   Wahrheitswert des Container-Header-Parameters
	 * @throws PersistenceException Lesefehler im Persistenzverzeichnis
	 */
	void setContHdrParam(KeyValParam param, boolean val) throws PersistenceException {
		checkContainerAccessedHeaderRead();
		containerHdr.setVal(param, val);
	}

	/**
	 * Setzt einen Parameters im Container-Header. Der Header muss vorher entweder gelesen oder mit Default-Werten erzeugt worden sein. Wirft eine {@link
	 * PersistenceException}, wenn die Änderung nicht durchgefuehrt werden kann.
	 *
	 * @param param Container-Header-Parameter
	 * @param val   Textwert des Container-Header-Parameters
	 * @throws PersistenceException Lesefehler im Persistenzverzeichnis
	 */
	void setContHdrParam(KeyValParam param, String val) throws PersistenceException {
		checkContainerAccessedHeaderRead();
		containerHdr.setVal(param, val);
	}

	/**
	 * Der Container-Header der Container-Datei, die mit der {@code accessContainer()}-Methode spezifiziert worden ist, wird mit dem Container-Header aus der
	 * angegebenen Container-Datei überschrieben.
	 *
	 * @param srcFile Container-Datei
	 * @throws PersistenceException Lesefehler im Persistenzverzeichnis
	 */
	void restoreHeader(File srcFile) throws PersistenceException {
		File contFile = checkContainerAccessedRW().toFile();
		if(_cache != null) {
			_cache.flush();
			forgetCache();
		}
		try {
			long oldHdrLenNoLenField;
			try(BufferedRandomAccessFile origContStr = new BufferedRandomAccessFile(contFile)) {
				origContStr.readFully(byte8Buf, 0, VERSION_STRING.length + ByteIO.INT4B_LEN);
				// Länge Header und Längenbyte
				oldHdrLenNoLenField = ByteIO.readSignedInt4Bytes(byte8Buf, VERSION_STRING.length);
			}

			int newHdrLenNoLenField;
			byte[] newHdr;
			try(BufferedRandomAccessFile srcContStr = new BufferedRandomAccessFile(srcFile)) {
				srcContStr.readFully(byte8Buf, 0, VERSION_STRING.length + ByteIO.INT4B_LEN);
				// Länge Header und Längenbyte
				newHdrLenNoLenField = ByteIO.readSignedInt4Bytes(byte8Buf, VERSION_STRING.length);

				newHdr = new byte[newHdrLenNoLenField];
				srcContStr.readFully(newHdr, 0, newHdrLenNoLenField);
			}

			if(newHdrLenNoLenField == oldHdrLenNoLenField) {
				// Nur ReadWrite, kein flush
				try(BufferedRandomAccessFile raf = new BufferedRandomAccessFile(contFile, "rw")) {
					raf.seek(0);
					raf.write(byte8Buf, 0, VERSION_STRING.length + ByteIO.INT4B_LEN);
					raf.write(newHdr, 0, newHdrLenNoLenField);  // Header drüberschreiben
				}
			}
			else {
				File tmpFile = Util.deleteCreateNewFile(contFile.getParentFile(), contFile.getName() + ".tmp");
				try(DataInputStream origContStr = new DataInputStream(new FileInputStream(contFile))) {
					try(DataOutputStream newContStr = new DataOutputStream(new FileOutputStream(tmpFile))) {
						origContStr.readFully(byte8Buf, 0, VERSION_STRING.length);
						newContStr.write(byte8Buf, 0, VERSION_STRING.length);
						newContStr.write(newHdr, 0, newHdrLenNoLenField);
						origContStr.skipBytes((int) oldHdrLenNoLenField);
						Util.copyStreams(origContStr, newContStr);
					}
				}
				Files.move(tmpFile.toPath(), contFile.toPath(), StandardCopyOption.REPLACE_EXISTING);
			}
			readContainerHeader();    // Header muss neu eingelesen werden
		}
		catch(IOException e) {
			throw new PersistenceException(
					"Container-Header konnte nicht wiederhergestellt werden. Quelle: '" + srcFile.getAbsolutePath() + "' Grund: " + e + "\n" + this, e
			);
		}
	}

	//////////////////////////////////////////////////////////////////////////////////////////////////////////
	// Container: Datensätze
	//////////////////////////////////////////////////////////////////////////////////////////////////////////

	void appendSerializedData(ArchiveDataSerializer serializer, final IdDataIdentification dataIdentification) throws PersistenceException {
		appendSerializedData(serializer.getActualWriteBuf(), serializer.getTotalWriteDataSize(), dataIdentification);
	}

	/**
	 * Haengt den bereits übergebenen und serialisierten Datensatz an die aktuelle Container-Datei an.
	 *
	 * @param actualWriteBuf Schreibpuffer
	 * @param totalWriteDataSize Benutzte Länge im Schreibpuffer
	 * @param dataIdentification Datenidentifikation
	 * @throws PersistenceException Lesefehler im Persistenzverzeichnis
	 */
	void appendSerializedData(final byte[] actualWriteBuf, final int totalWriteDataSize, final IdDataIdentification dataIdentification) throws PersistenceException {
		Path contFile = checkContainerAccessedRW();
		if(_cache == null) {
			_cache = _cacheManager.createCache(this, false, dataIdentification);
		}
		if(_cache != null) {
			boolean isFirstDataInContainer = _cache.getContainerSize() == getHeaderLen();
			_cache.cache(actualWriteBuf, totalWriteDataSize);
			// Nach dem cachen des ersten Datensatzes muss der Cache geflusht werden, damit die Datei vorhanden ist und gefunden werden kann.
			// Ein flushen nach dem cachen des Headers wird unterdrückt, damit keine Container-Dateien ohne Daten entstehen
			if(isFirstDataInContainer) {
				_cache.flush();
			}
		} else {
			writeContainerFileSafely(contFile, false, out -> out.write(actualWriteBuf, 0, totalWriteDataSize));
		}
	}

	/**
	 * Führt die angegebene Schreib-Aktion auf die Containerdatei aus und behandelt dabei Fehler
	 *
	 * @param containerFile Pfad der Containerdatei
	 * @param create        Daten erstellen? Ansonsten wird an die Datei angehängt.
	 * @param writeAction   Schreibaktion (Definiert, was geschrieben werden soll)
	 * @throws PersistenceException Fehler beim Schreiben
	 */
	public static void writeContainerFileSafely(Path containerFile, boolean create, WriteAction writeAction) throws PersistenceException {
		boolean openedFile = false;
		while(true) {
			StandardOpenOption openOption;
			if(create)
				openOption = CREATE;
			else {
				openOption = APPEND;
			}
			try (OutputStream out = Files.newOutputStream(containerFile, openOption)) {
				openedFile = true;
				writeAction.execute(out);
				return;
			} catch (NoSuchFileException e) {
				if(openedFile || !Files.isRegularFile(containerFile)) {
					throw new PersistenceException("Archivdatensatz konnte nicht in Container-Datei geschrieben werden: " + e.getMessage() + "\nDatei existiert nicht: " + containerFile, e);
				}
				// Falls die Datei doch existiert, dann gibt es vermutlich das bekannte Problem, dass Windows eine zum Lesen geöffnete Datei nicht gleichzeitig schreiben kann.
				// Also einfach warten und nochmal versuchen
				Thread.yield();
			} catch (IOException e) {
				throw new PersistenceException("Archivdatensatz konnte nicht in Container-Datei geschrieben werden: " + e.getMessage() + "\n" + containerFile, e);
			}
		}
	}

	/**
	 * Ermittelt die Gesamtlänge des mit {@code accessContainer()} im Zugriff befindlichen Containers.
	 *
	 * @return Länge der Container-Datei
	 * @throws PersistenceException Lesefehler im Persistenzverzeichnis
	 */
	long getContainerSize() throws PersistenceException {
		Path contFile = checkContainerAccessed();
		if(_cache != null) {
			return _cache.getContainerSize();
		}
		try {
			return Files.size(contFile);
		} catch (IOException e) {
			throw new PersistenceException(e);
		}
	}

	/**
	 * Ermittelt die Gesamtlänge des mit {@code accessContainer()} im Zugriff befindlichen Containers.
	 *
	 * @return Länge der Container-Datei
	 */
	public static int getHeaderLen() {
		return HEADER_LENGTH;
	}

	/**
	 * Ermittelt den vollständigen Overhead für einen Datensatz. Inklusive Längenbyte, Datensatz-Header, Compress-Feld und abschliessendem Separator.
	 *
	 * @return Overhead in Bytes
	 */
	public static int getTotalDataOverhead() {
		return DATALEN_LEN + DATAHDR_LEN + COMPRESS_LEN + ByteIO.SEPARATOR.length;
	}


	//////////////////////////////////////////////////////////////////////////////////////////////////////////
	// Private Methoden
	//////////////////////////////////////////////////////////////////////////////////////////////////////////

	/**
	 * Prüft, ob ein Container zum Lesen im Zugriff ist. Wirft eine {@link PersistenceException} falls nicht.
	 *
	 * @return aktuelle Container-Datei
	 */
	@NotNull
	private Path checkContainerAccessed() throws PersistenceException {
		if (!accessed()) throw new PersistenceException("Container zuerst mit accessContainer() spezifizieren");
		return contFile;
	}

	/**
	 * Prüft, ob ein Container zum Lesen und Schreiben im Zugriff ist. Wirft eine {@link PersistenceException} falls nicht.
	 *
	 * @return aktuelle Container-Datei
	 */
	private Path checkContainerAccessedRW() throws PersistenceException {
		Path result = checkContainerAccessed();
		if (readOnly) throw new PersistenceException("Container nur zum Lesen adressiert\n" + this);
		return result;
	}

	/**
	 * Prüft, ob ein Container zum Lesen im Zugriff ist und dessen Container-Header bereits eingelesen ist. Wirft eine {@link PersistenceException} falls
	 * nicht.
	 *
	 * @return aktuelle Container-Datei
	 */
	private Path checkContainerAccessedHeaderRead() throws PersistenceException {
		Path result = checkContainerAccessed();
		if (!headerRead) throw new PersistenceException("Container-Header weder eingelesen noch neu erzeugt\n" + this);
		return result;
	}

	/**
	 * Schliesst ein RandomAccessFile und faengt die Exception ab. (Nur zur Abkuerzung verwendet).
	 *
	 * @param raf Zu schliessendes RandomAccessFile
	 * @throws PersistenceException Lesefehler im Persistenzverzeichnis
	 */
	private void closeRandomAccessFile(@Nullable FileAccess raf) throws PersistenceException {
		if(raf != null) {
			try {
				raf.close();
			}
			catch(IOException e) {
				throw new PersistenceException("Container-Datei konnte nicht geschlossen werden: " + e.getMessage() + "\n" + this, e);
			}
		}
	}

	public long getContainerId() {
		return containerId;
	}

	@Nullable
	public Path getContFile() {
		return contFile;
	}
}
