/*
 *
 * Copyright 2005-2008 by beck et al. projects GmbH, Munich
 * Copyright 2009-2019 by Kappich Systemberatung, Aachen
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.ars.ars.persistence;

import de.bsvrz.dav.daf.main.archive.ArchiveDataKind;
import de.bsvrz.sys.funclib.kappich.annotations.Nullable;
import de.bsvrz.sys.funclib.losb.util.Util;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Properties;

/**
 * Klasse, die einen Container-Header repraesentiert. Ist eine Ansammlung von {@link de.bsvrz.ars.ars.persistence.KeyValParam}-Objekten, die die speziellen
 * Eigenschaften eines Eintrags kapseln.
 *
 * @author beck et al. projects GmbH
 * @author Alexander Schmidt
 * @version $Revision$ / $Date$ / ($Author$)
 */
public class ContainerHdr extends KeyValProps {

	/** Sortierte Liste aller ContainerHeader-Parameter. */
	protected static final List<KeyValParam> allParams = new ArrayList<>();

	/** Länge der Simulationsvariante (0-999) in Dezimaldarstellung. */
	private static final int SV_STR_LEN = 3;

	/** Länge einer Datensatzart bei textueller Repraesentation mit 'OA', 'ON', 'NA' oder 'NN'. */
	private static final int DATAKIND_STR_LEN = 2;

	// --------------------------------------------------------------------------------------------------------
	// Konstante Werte von Parametern in Textdarstellung:

	/** Medium-ID bei noch nicht auf externes Medium gesichertem Container. */
	public static final String MEDIUM_UNSAVED = "unsaved   "; // 10 Zeichen lang

	/** Medium-ID bei gesichertem Container mit dauerhaft nicht mehr verfuegbarem Medium. */
	public static final String MEDIUM_DESTROYED = "destroyed "; // 10 Zeichen lang

	private static final String DATAKIND_UNDEF = "  "; // 2 Zeichen lang

	/**
	 * Wert von anzDS bei nicht abgeschlossenem Container.
	 */
	public static final int CONT_UNCLOSED = -1;

	/**
	 * Textuelle Repraesentation der Datensatzarten. Die ersten vier Indexe entsprechen {@link Util#getDataKindIndex(ArchiveDataKind)}.
	 */
	private static final String[] DATAKINDS = {"OA", "ON", "NA", "NN", DATAKIND_UNDEF};


	/**
	 * Container-Header-Parameter.
	 */
	public static final KeyValParam CHP_CONT_ID = KeyValParam.createNumParam("contID", 0, BYTES5_STR_LEN, 0, BYTE5_MAXVAL, 5);
	public static final KeyValParam CHP_ANZ_DS = KeyValParam.createNumParam("anzDS", CONT_UNCLOSED, BYTES4_STR_LEN, CONT_UNCLOSED, BYTE4_MAXVAL, 4);
	public static final KeyValParam CHP_OBJ_ID = KeyValParam.createNumParam("objID", 0, BYTES8_STR_LEN, 0, BYTE8_MAXVAL, 8);
	public static final KeyValParam CHP_ATG_ID = KeyValParam.createNumParam("atgID", 0, BYTES8_STR_LEN, 0, BYTE8_MAXVAL, 8);
	public static final KeyValParam CHP_ASP_ID = KeyValParam.createNumParam("aspID", 0, BYTES8_STR_LEN, 0, BYTE8_MAXVAL, 8);
	public static final KeyValParam CHP_SIM_VAR = KeyValParam.createNumParam("sv", 0, SV_STR_LEN, 0, 999, 2);
	public static final KeyValParam CHP_DATA_KIND = KeyValParam.createStringParam("art", DATAKIND_UNDEF, DATAKIND_STR_LEN, DATAKINDS);
	public static final KeyValParam CHP_DATA_IDX_MIN = KeyValParam.createNumParam("DImin", 0, BYTES8_STR_LEN, 0, BYTE8_MAXVAL, 8);
	public static final KeyValParam CHP_DATA_IDX_MAX = KeyValParam.createNumParam("DImax", 0, BYTES8_STR_LEN, 0, BYTE8_MAXVAL, 8);
	public static final KeyValParam CHP_DATA_TIME_MIN = KeyValParam.createNumParam("DZmin", 0, BYTES6_STR_LEN, 0, BYTE6_MAXVAL, 6);
	public static final KeyValParam CHP_DATA_TIME_MAX = KeyValParam.createNumParam("DZmax", 0, BYTES6_STR_LEN, 0, BYTE6_MAXVAL, 6);
	public static final KeyValParam CHP_ARC_TIME_MIN = KeyValParam.createNumParam("AZmin", 0, BYTES6_STR_LEN, 0, BYTE6_MAXVAL, 6);
	public static final KeyValParam CHP_ARC_TIME_MAX = KeyValParam.createNumParam("AZmax", 0, BYTES6_STR_LEN, 0, BYTE6_MAXVAL, 6);
	public static final KeyValParam CHP_TO_SAVE = KeyValParam.createStringParam("sichern", FALSE, BOOL_STR_LEN, new String[]{FALSE, TRUE});
	public static final KeyValParam CHP_MEDIUM_ID = KeyValParam.createStringParam("mediumID", MEDIUM_UNSAVED, BYTES4_STR_LEN);
	/**
	 * Das ist ein absoluter Zeitpunkt in Millisekunden. Wenn der Container abgeschlossen wird, wird
	 * dieser Zeitpunkt auf die Archivzeit des juengsten Datensatzes + den parametrierten Vorhaltezeitraum gesetzt.
	 * Dieser Parameter wird in den TAnfArS "Loeschzeitpunkt" oder "Vorhaltezeitraum" genannt.
	 */
	public static final KeyValParam CHP_LOESCHEN = KeyValParam.createNumParam("loeschen", 0, BYTES6_STR_LEN, 0, BYTE6_MAXVAL, 6);
	/**
	 * Dies ist ein absoluter Zeitpunkt in Millisekunden. Bei jeder Anfrage oder Archivinfo-Anfrage wird
	 * CHP_NO_DEL_TIL = max(CHP_DELETE_AT + atg.archivEinstellung::LöschschutzverlängerungMax,
	 * max(jetzt + atg.archivEinstellung::Löschschutzverlängerung, CHP_NO_DEL_TIL))
	 * gesetzt. Bei einer manuellen Loeschschutzverlängerung um T_lsv (atg.archivAnfrageschnittstelle:Typ13) wird
	 * CHP_NO_DEL_TIL = max(CHP_DELETE_AT + atg.archivEinstellung::LöschschutzverlängerungMax,
	 * max(jetzt + T_lsv, CHP_NO_DEL_TIL))
	 * gesetzt.
	 */
	public static final KeyValParam CHP_LOESCHUTZ = KeyValParam.createNumParam("loeschutz", 0, BYTES6_STR_LEN, 0, BYTE6_MAXVAL, 6);
	public static final KeyValParam CHP_DELETED = KeyValParam.createStringParam("geloescht", FALSE, BOOL_STR_LEN, new String[]{FALSE, TRUE});
	public static final KeyValParam CHP_RESTORED = KeyValParam.createStringParam("wdhgest", FALSE, BOOL_STR_LEN, new String[]{FALSE, TRUE});

	/** Länge des Key/Value-Blocks in Byte. */
	public static final int HDR_TXT_LEN;

	static {
		allParams.add(CHP_CONT_ID);
		allParams.add(CHP_ANZ_DS);
		allParams.add(CHP_OBJ_ID);
		allParams.add(CHP_ATG_ID);
		allParams.add(CHP_ASP_ID);
		allParams.add(CHP_SIM_VAR);
		allParams.add(CHP_DATA_KIND);
		allParams.add(CHP_DATA_IDX_MIN);
		allParams.add(CHP_DATA_IDX_MAX);
		allParams.add(CHP_DATA_TIME_MIN);
		allParams.add(CHP_DATA_TIME_MAX);
		allParams.add(CHP_ARC_TIME_MIN);
		allParams.add(CHP_ARC_TIME_MAX);
		allParams.add(CHP_TO_SAVE);
		allParams.add(CHP_MEDIUM_ID);
		allParams.add(CHP_LOESCHEN);
		allParams.add(CHP_DELETED);
		allParams.add(CHP_LOESCHUTZ);
		allParams.add(CHP_RESTORED);

		int headerLen = 0;
		for(KeyValParam param : getAllParams()) {
			headerLen += param.getKey().length() + "=".length() + param.getValLen() + CH_DELIM.length();
		}
		HDR_TXT_LEN = headerLen;
	}

	@Nullable
	public ArchiveDataKind getValAsArchiveDataKind(KeyValParam chp) throws PersistenceException {
		if(chp.isNumeric()) {
			throw new PersistenceException("Parameter ist numerisch und nicht als ArchiveDataKind interpretierbar: " + chp + " " + this);
		}
		else {
			String val = properties.getProperty(chp.getKey());
			if(val == null) {
				return null;
			}
			else {
				for(int i = 0; i < DATAKINDS.length; i++) {
					if(DATAKINDS[i].equals(val)) return Util.getDataKindFromIndex(i);
				}
			}
			throw new PersistenceException("Parameter ist nicht als ArchiveDataKind interpretierbar: " + chp + " " + this);
		}
	}

	public void setVal(KeyValParam chp, ArchiveDataKind val) throws PersistenceException {
		String txtVal = DATAKINDS[Util.getDataKindIndex(val)];
		chp.checkVal(txtVal);
		properties.setProperty(chp.getKey(), chp.formatVal(txtVal));
	}

	/**
	 * Liefert eine sortierte Liste aller Container-Header-Parameter.
	 *
	 * @return Alle Parameter
	 */
	public static List<KeyValParam> getAllParams() {
		return allParams;
	}


	public String writeContainerHdr() {
		StringBuilder sb = new StringBuilder(HDR_TXT_LEN);
		for(KeyValParam param : getAllParams()) {
			sb.append(param.getKey()).append("=").append(properties.getProperty(param.getKey())).append(CH_DELIM);
		}
		return sb.toString();
	}

	public void readContainerHdr(byte[] buf, int length) throws IOException, PersistenceException {
		properties.clear();
		Properties tmpProp = new Properties();
		tmpProp.load(new ByteArrayInputStream(buf, 0, length));
		for(KeyValParam param : getAllParams()) {
			setVal(param, (String)tmpProp.get(param.getKey()));
		}
	}
}
