/*
 *
 * Copyright 2017-2020 by Kappich Systemberatung, Aachen
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.ars.ars.persistence;

import de.bsvrz.ars.ars.persistence.index.result.IndexResult;
import de.bsvrz.dav.daf.main.archive.ArchiveDataKind;

import java.util.EnumMap;

/**
 * Einfache standalone Daten-Klasse, die Containerheader speichern kann. Sinnvoll beispielsweise um eine temporäre Kopie von Containerheadern erstellen zu 
 * können ohne die Containerdatei aufzuhalten oder um im Speicher temporär Werte verändern zu können
 *
 * @author Kappich Systemberatung
 */
public class ContainerHeaders implements ContainerManagementData {
	private final EnumMap<ContainerManagementInformation, Object> _data = new EnumMap<>(ContainerManagementInformation.class);
	private final ContainerDirectory _location;

	/**
	 * Kopiert die Container-Header von einer anderen Klasse, die das abrufen von Containerheadern erlaubt ({@link ContainerManagementData}).
	 * @param data Zu kopierende Daten
	 * @throws PersistenceException Fehler beim Lesen
	 */
	public ContainerHeaders(ContainerManagementData data) throws PersistenceException {
		_location = data.getLocation();
		for(ContainerManagementInformation information : ContainerManagementInformation.values()) {
			if(information.isNumeric()) {
				_data.put(information, data.getContainerHeaderParamAsLong(information));
			}
			else {
				_data.put(information, data.getContainerHeaderParamAsString(information));
			}
		}
	}

	/**
	 * Erzeugt die ContainerHeader aus einem einzeiligen Ergebnis des Verwaltungsdatenindex.
	 *
	 * @param indexResult        Index-Abfrageergebnis
	 * @param dataIdentification Datenidentifikation
	 * @param adk                Archivdatenart
	 * @param resultIndex        Zeile, die im IndexResult verwendet werden soll
	 */
	public ContainerHeaders(IndexResult<ContainerManagementInformation> indexResult, final IdDataIdentification dataIdentification, final ArchiveDataKind adk, final int resultIndex) {
		this(indexResult, resultIndex, dataIdentification.resolve(adk));
	}

	/**
	 * Erzeugt die ContainerHeader aus einem einzeiligen Ergebnis des Verwaltungsdatenindex.
	 *
	 * @param indexResult Index-Abfrageergebnis
	 * @param resultIndex Zeile, die im IndexResult verwendet werden soll
	 * @param location    Ort des Containers
	 */
	public ContainerHeaders(IndexResult<ContainerManagementInformation> indexResult, final int resultIndex, ContainerDirectory location) {
		_location = location;
		for(ContainerManagementInformation information : ContainerManagementInformation.values()) {
			if(information.isNumeric()) {
				_data.put(information, indexResult.get(resultIndex, information));
			}
			else {
				_data.put(information, indexResult.getString(resultIndex, information));
			}
		}
	}

	@Override
	public String getContainerHeaderParamAsString(final ContainerManagementInformation param) {
		return (String) _data.get(param);
	}

	@Override
	public boolean getContainerHeaderParamAsBoolean(final ContainerManagementInformation param) {
		return getContainerHeaderParamAsString(param).equals("1");
	}

	@Override
	public long getContainerHeaderParamAsLong(final ContainerManagementInformation param) {
		return (Long) _data.get(param);
	}

	@Override
	public int getContainerHeaderParamAsInt(final ContainerManagementInformation param) {
		return (int)getContainerHeaderParamAsLong(param);
	}

	@Override
	public ContainerDirectory getLocation() {
		return _location;
	}

	@Override
	public boolean equals(final Object o) {
		if(this == o) return true;
		if(o == null || getClass() != o.getClass()) return false;

		final ContainerHeaders that = (ContainerHeaders) o;

		if(!_data.equals(that._data)) return false;
		return _location.equals(that._location);
	}

	@Override
	public int hashCode() {
		int result = _data.hashCode();
		result = 31 * result + _location.hashCode();
		return result;
	}
}
