/*
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */
package de.bsvrz.ars.ars.persistence;

import de.bsvrz.ars.ars.mgmt.datatree.synchronization.SyncKey;
import de.bsvrz.ars.ars.mgmt.datatree.synchronization.SynchronizationFailedException;
import de.bsvrz.sys.funclib.kappich.annotations.NotNull;

/**
 * Interface für Objekte, die den Zugriff auf Datenidentifikationen erlauben.
 */
public interface DataIdentificationManager {

	/**
	 * Gibt zurück, ob bei Fehlern beim Lesen von Containerdateien die Datei umbenannt (und damit deaktiviert)
	 * werden soll.
	 *
	 * @return Defekte Container umbenennen/löschen?
	 */
	default boolean shouldDeleteBrokenContainers() {
		return false;
	}

	/**
	 * Muss immer vor Zugriffen auf den kritischen Bereich (die Indexe oder Container) der Datenidentifikation aufgerufen werden.
	 * <p>
	 * Wenn ein anderer Task sich im kritischen Bereich befindet, blockiert diese Methode bis der kritische Bereich wieder frei ist.
	 * <p>
	 * Da der Index-Bereich möglicherweise von vielen Tasks benutzt wird (und auch Zugriffe beim Archivieren von Daten erfolgen sollte), der Bereich so
	 * schnell wie möglich wieder verlassen werden. Es ist zwingend erforderlich, das zurückgegebene Lock-Objekt nach Benutzung zu schließen, sonst
	 * können andere Threads blockiert werden.
	 * <p>
	 * Beispiel-Code:
	 * <pre>{@code
	 * 	 try(SyncKey<IdDataIdentification> lock = archMgr.lockIndex(din.getDataIdentification())) {
	 * 		  // Tue was mit den Indexen
	 *   }
	 * }</pre>
	 * <p>
	 * Das zurückgegebene {@link SyncKey}-Objekt verhält sich wie ein {@link java.util.concurrent.locks.ReentrantLock}, es ist also möglich, ein
	 * zweites Lock zu erhalten, wenn bereits ein Lock geholt wurde.
	 *
	 * @param dataIdentification Datenidentifikation
	 * @return Lock
	 * @throws SynchronizationFailedException Synchronisierung ist fehlgeschlagen (Unterbrochen beim Warten)
	 */
	SyncKey<IdDataIdentification> lockIndex(IdDataIdentification dataIdentification) throws SynchronizationFailedException;

	/**
	 * Muss immer vor Zugriffen auf den kritischen Bereich (die Indexe oder Container) eines Containerverzeichnisses aufgerufen werden.
	 * <p>
	 * Wenn ein anderer Task sich im kritischen Bereich befindet, blockiert diese Methode bis der kritische Bereich wieder frei ist.
	 * <p>
	 * Da der Index-Bereich möglicherweise von vielen Tasks benutzt wird (und auch Zugriffe beim Archivieren von Daten erfolgen), sollte der Bereich so
	 * schnell wie möglich wieder verlassen werden. Es ist zwingend erforderlich, das zurückgegebene Lock-Objekt nach Benutzung zu schließen, sonst
	 * können andere Threads blockiert werden.
	 * <p>
	 * Beispiel-Code:
	 * <pre>{@code
	 * 	 try(SyncKey<IdDataIdentification> lock = archMgr.lockIndex(din.getDataIdentification())) {
	 * 		  // Tue was mit den Indexen
	 *   }
	 * }</pre>
	 * <p>
	 * Das zurückgegebene {@link SyncKey}-Objekt verhält sich wie ein {@link java.util.concurrent.locks.ReentrantLock}, es ist also möglich, ein
	 * zweites Lock zu erhalten, wenn bereits ein Lock geholt wurde.
	 * <p>
	 * Diese Methode verwendet dieselbe Synchronisierung wie die andere {@link #lockIndex(IdDataIdentification)}-Überladung,
	 * bietet jedoch einfachere Nutzung, wenn ein {@link LockedContainerDirectory} benötigt wird, das zusätzlich
	 * die Datenart mit enthält.
	 *
	 * @param containerFileDir Containerverzeichnis
	 * @return LockedContainerDirectory
	 * @throws SynchronizationFailedException Synchronisierung ist fehlgeschlagen (Unterbrochen beim Warten)
	 */
	default LockedContainerDirectory lockIndex(ContainerDirectory containerFileDir) throws SynchronizationFailedException {
		return new LockedContainerDirectory(lockIndex(containerFileDir.dataIdentification()), containerFileDir.archiveDataKind());
	}

	/**
	 * Gibt die maximale Größe für Index-Caches zurück
	 *
	 * @return Maximale Größe in Bytes
	 */
	default int getIndexCacheMaxSize() {
		return 16384;
	}

	/**
	 * Formatiert eine Objekt-ID. Diese Methode konvertiert die ID in einen String, kann jedoch überladen werden,
	 * wenn die Klasse eine bessere Ausgabe bereitstellen kann.
	 *
	 * @param objId Objekt-ID
	 * @return String-Ausgabe
	 */
	default String formatObj(long objId) {
		return String.valueOf(objId);
	}


	/**
	 * Die Methode formatDataIdentification gibt eine Datenidentifikation als lesbaren String aus.
	 * Dabei wird in der Regel {@link #formatObj(long)} für die einzelnen Bestandteile aufgerufen.
	 *
	 * @param location Containerverzeichnis-ID
	 * @return String
	 */
	@NotNull
	default String formatContainerDirectory(ContainerDirectory location) {
		return formatObj(location.getObjectId()) + ":" + formatObj(location.getAtgId()) + ":" + formatObj(location.getAspectId()) + " (" +
				location.archiveDataKind() + ")";
	}
}
