/*
 *
 * Copyright 2017-2020 by Kappich Systemberatung, Aachen
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.ars.ars.persistence;

import static de.bsvrz.ars.ars.persistence.index.ContainerManagementIndex.INDEX_CONTENT_DESCRIPTOR;


import de.bsvrz.ars.ars.mgmt.datatree.synchronization.SynchronizationFailedException;
import de.bsvrz.ars.ars.persistence.directories.PersistenceDirectory;
import de.bsvrz.ars.ars.persistence.iter.DataIterator;
import de.bsvrz.ars.ars.persistence.iter.DataSequence;
import de.bsvrz.dav.daf.main.DataState;
import de.bsvrz.dav.daf.main.archive.ArchiveDataKind;
import de.bsvrz.dav.daf.util.BufferedRandomAccessFile;
import de.bsvrz.sys.funclib.debug.Debug;
import de.bsvrz.sys.funclib.kappich.annotations.NotNull;
import de.bsvrz.sys.funclib.kappich.annotations.Nullable;
import de.bsvrz.sys.funclib.losb.util.Util;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Set;
import java.util.TreeMap;

/**
 * Klasse, die den Zugriff auf die Datei bietet, die die Header und Datensatzindizes von gelöschten Containern enthält.
 * <p>
 * Soll eine vorhandene Datei eingelesen werden, muss nach dem Konstruktor {@link #read()} aufgerufen werden. Die Datei wird dabei so weit gelesen wie möglich.
 * Treten durch fehlerhafte Daten dabei Exceptions auf, enthält das DeletedContainerFile trotzdem alle Daten, die noch gelesen werden konnten, d.h. es hat einen konsistenten Zustand.
 *
 * @author Kappich Systemberatung
 */
public final class DeletedContainerFile {

	private static final Debug _debug = Debug.getLogger();

	/**
	 * Dateiname für Reguläte dateien für gelöschte Container
	 */
	public static final String DELETED_CONTAINER_FILENAME = "_deleted.dat";

	/**
	 * Dateiname unter der die neue Datei geschrieben wird. Sie wird erst nach "_deleted.dat" umbenannt, wenn sie erfolgreich komplett geschrieben wurde.
	 */
	public static final String DELETED_CONTAINER_FILENAME_TMP = "_deleted.tmp";

	/**
	 * Dateiname für temporäres Backup für die vorige Version der Datei
	 */
	public static final String DELETED_CONTAINER_FILENAME_BACKUP = "_deleted.backup";

	private final Map<Long, DeletedContainerData> _map = new TreeMap<>();
	private final Path _fileTmp;
	private final Path _fileBackup;
	private final Path _file;
	private final ContainerDirectory _containerFileDir;

	/**
	 * Erstellt ein neues DeletedContainerFile ohne Inhalt. Soll die existierende Datei (falls vorhanden) gelesen werden <b>muss</b> {@link #read()} aufgerufen werden.
	 *
	 * @param adkPath          Pfad der Archivdatenart (Verzeichnis, in dem Containerdateien angelegt werden)
	 * @param containerFileDir Datenidentifikation (Wird später beim Iterator benutzt, um die Datenidentifikation an den Daten zurückzugeben)
	 */
	public DeletedContainerFile(final Path adkPath, final ContainerDirectory containerFileDir) {
		_file = adkPath.resolve(DELETED_CONTAINER_FILENAME);
		_fileTmp = adkPath.resolve(DELETED_CONTAINER_FILENAME_TMP);
		_fileBackup = adkPath.resolve(DELETED_CONTAINER_FILENAME_BACKUP);
		_containerFileDir = containerFileDir;
	}

	/**
	 * Erstellt ein neues DeletedContainerFile ohne Inhalt. Soll die existierende Datei (falls vorhanden) gelesen werden <b>muss</b> {@link #read()} aufgerufen werden.
	 *
	 * @param containerFileDir     Datenidentifikation (Wird später beim Iterator benutzt, um die Datenidentifikation an den Daten zurückzugeben)
	 * @param persistenceDirectory Persistenzverzeichnis
	 */
	public DeletedContainerFile(final ContainerDirectory containerFileDir, PersistenceDirectory persistenceDirectory) {
		this(persistenceDirectory.getPath(containerFileDir), containerFileDir);
	}

	/**
	 * Gibt {@code true} zurück, wenn eien bestehende Datei existiert
	 *
	 * @return {@code true}, wenn eien bestehende Datei existiert, sonst {@code false}
	 */
	public boolean exists() {
		return Files.exists(_file);
	}

	/**
	 * Liest die bestehende Datei ein.
	 *
	 * @throws PersistenceException Fehler in der Datei oder IO-Fehler
	 */
	public void read() throws PersistenceException {
		try (BufferedRandomAccessFile fileAccess = new BufferedRandomAccessFile(_file.toFile(), "r")) {
			int length = fileAccess.readInt();
			byte[] tmpHeaderData = new byte[INDEX_CONTENT_DESCRIPTOR.getEntryLengthBytes()];
			for (int i = 0; i < length; i++) {
				fileAccess.readFully(tmpHeaderData);
				DeletedContainerData data = new DeletedContainerData(tmpHeaderData, _containerFileDir);
				_map.put(data.getContainerHeaderParamAsLong(ContainerManagementInformation.CHP_CONT_ID), data);

				int numDeletedBlocks = fileAccess.readInt();
				for (int f = 0; f < numDeletedBlocks; f++) {
					long from = fileAccess.readLong();
					long to = fileAccess.readLong();
					int hashExpected = Long.hashCode(from) ^ Long.hashCode(to);
					int hashActual = fileAccess.readInt();
					if (hashExpected != hashActual) {
						throw new PersistenceException("Fehler in der Prüfsumme der Datei: " + hashExpected + "!=" + hashActual);
					}
					byte separator = fileAccess.readByte();
					if (separator != 0) {
						throw new PersistenceException("Fehler in der Datei-Struktur");
					}
					data.addIndexRange(from, to);
				}
			}
		} catch (IOException e) {
			throw new PersistenceException(e);
		}
	}

	/**
	 * Schreibt die aktuelle Datei. Dabei wird der Inhalt in eine temporäre Datei geschreiben, die existierende Datei falls vorhanden weggeschoben,
	 * die temporäre Datei in den richtige Dateinamen umbenannt und ggf. dann bei Erfolg die vorher weg geschobene Datei gelöscht.
	 * <p>
	 * Auf diese Weise ist die Datei, falls sie existiert, immer in einem konsistenten Zustand.
	 *
	 * @throws PersistenceException Lesefehler im Persistenzverzeichnis
	 */
	public void write() throws PersistenceException {
		try (BufferedRandomAccessFile fileAccess = new BufferedRandomAccessFile(_fileTmp.toFile(), "rw")) {
			fileAccess.writeInt(_map.size());
			byte[] tmpHeaderData = new byte[INDEX_CONTENT_DESCRIPTOR.getEntryLengthBytes()];
			for (Map.Entry<Long, DeletedContainerData> entry : _map.entrySet()) {
				DeletedContainerData data = entry.getValue();
				data.serialize(tmpHeaderData);
				fileAccess.write(tmpHeaderData);

				List<IndexRange> blocks = data.getDeletedBlocks();
				fileAccess.writeInt(blocks.size());
				for (IndexRange block : blocks) {
					fileAccess.writeLong(block.from);
					fileAccess.writeLong(block.to);
					int hashExpected = Long.hashCode(block.from) ^ Long.hashCode(block.to);
					fileAccess.writeInt(hashExpected);
					fileAccess.writeByte(0); // Separator
				}
			}
			fileAccess.setLength(fileAccess.getFilePointer());
		} catch (IOException e) {
			throw new PersistenceException("Kann Datei mit gelöschten Containern nicht schreiben", e);
		}
		boolean existed = false;
		try {
			if (Files.exists(_file)) {
				Files.move(_file, _fileBackup, StandardCopyOption.REPLACE_EXISTING);
				existed = true;
			}
		} catch (IOException e) {
			throw new PersistenceException("Kann bestehende Datei nicht umbenennen: " + _file + " -> " + _fileBackup, e);
		}
		try {
			Files.move(_fileTmp, _file, StandardCopyOption.REPLACE_EXISTING);
		} catch (IOException e) {
			// Versuchen, Backup wiederherzustellen
			try {
				Files.move(_fileBackup, _file, StandardCopyOption.REPLACE_EXISTING);
			} catch (IOException e1) {
				e.addSuppressed(e1);
			}
			throw new PersistenceException("Fehler beim Umbenennen der neuen Datei zur Speicherung der gelöschten Container", e);
		}
		if (existed) {
			try {
				Files.delete(_fileBackup);
			} catch (IOException e) {
				_debug.warning("Backup-Datei konnte nicht gelöscht werden: " + _fileBackup, e);
			}
		}
	}

	/**
	 * Fügt einen (gelöschten) Container hinzu
	 *
	 * @param existingContainer Container, der in die Datei eingetragen werden soll
	 * @throws PersistenceException Falls der Container nicht gelesen werden konnte
	 * @throws SynchronizationFailedException Fehler bei Synchronisierung
	 */
	public void addDeletedContainer(BasicContainerFileHandle existingContainer) throws PersistenceException, SynchronizationFailedException {
		addDeletedContainer(existingContainer, existingContainer);
	}

	/**
	 * Fügt einen (gelöschten) Container hinzu
	 *
	 * @param header Container-Header
	 * @param data   Container-Daten
	 * @throws PersistenceException Falls der Container nicht gelesen werden konnte
	 * @throws SynchronizationFailedException Fehler bei Synchronisierung
	 */
	public void addDeletedContainer(final ContainerManagementData header, final DataSequence data) throws PersistenceException, SynchronizationFailedException {
		_map.put(header.getContainerHeaderParamAsLong(ContainerManagementInformation.CHP_CONT_ID), new DeletedContainerData(header, data));
	}

	/**
	 * Entfernt einen gelöschten Container aus der Datei (z. B. sinnvoll beim Wiederherstellen)
	 *
	 * @param containerID Container-ID
	 */
	public void removeDeletedContainer(final long containerID) {
		_map.remove(containerID);
	}

	/**
	 * Iteriert über die Daten eines gelöschten Containers. Der Iterator enthält nur Werte für die folgenden Attribute: containerID, dataIndex, dataKind.
	 *
	 * @param containerId Container-ID
	 * @return DataIterator
	 * @throws PersistenceException Fehler beim elsen aus Datei oder angegebener Container ist nicht enthalten.
	 */
	public DataIterator dataIterator(final long containerId) throws PersistenceException {
		DeletedContainerData deletedContainerData = _map.get(containerId);
		if (deletedContainerData == null) {
			throw new PersistenceException("Gelöschter Container nicht in " + DELETED_CONTAINER_FILENAME + " gefunden: " + containerId);
		}
		return deletedContainerData.iterator();
	}

	/**
	 * Gibt die Container-Header eines gelöschten Containers zurück
	 *
	 * @param containerId ID
	 * @return Headers oder null (falls Container mit der ID nicht enthalten)
	 */
	public ContainerManagementData headers(final long containerId) {
		return _map.get(containerId);
	}

	/**
	 * Gibt die Liste mit monotonen Datenindexblöcken für den angegebenen gelöschten Container zurück.
	 *
	 * @param containerId Container-ID
	 * @return Liste oder null (falls Container mit der ID nicht enthalten)
	 */
	@Nullable
	public List<IndexRange> deletedBlocks(final long containerId) {
		DeletedContainerData data = _map.get(containerId);
		if (data == null) return null;
		return Collections.unmodifiableList(data.getDeletedBlocks());
	}

	/**
	 * Gibt eine Collection über alle Container in der Datei zurück
	 *
	 * @return Container-IDs
	 */
	public Set<Long> containers() {
		return _map.keySet();
	}

	/**
	 * Gibt die Datei zurück
	 *
	 * @return die Datei
	 */
	public Path getFile() {
		return _file;
	}

	private static final class DeletedContainerData implements ContainerManagementData, DataSequence {
		private final List<IndexRange> _deletedBlocks = new ArrayList<>();
		private final byte[] _data;
		private final ContainerDirectory _containerDir;

		public DeletedContainerData(final byte[] tmpHeaderData, final ContainerDirectory containerFileDir) {
			_data = tmpHeaderData.clone();
			_containerDir = containerFileDir;
		}

		public DeletedContainerData(final ContainerManagementData header, final DataSequence dataSequence) throws PersistenceException, SynchronizationFailedException {
			_containerDir = header.getLocation();
			_data = new byte[INDEX_CONTENT_DESCRIPTOR.getEntryLengthBytes()];
			for (ContainerManagementInformation information : ContainerManagementInformation.values()) {
				if (information.isNumeric()) {
					long paramAsLong = header.getContainerHeaderParamAsLong(information);
					INDEX_CONTENT_DESCRIPTOR.getColumn(information).writeBytes(paramAsLong, _data);
				} else {
					String paramAsString = header.getContainerHeaderParamAsString(information);
					INDEX_CONTENT_DESCRIPTOR.getColumn(information).writeBytes(paramAsString, _data);
				}
			}
			try (DataIterator dataIter = dataSequence.iterator()) {
				ContainerDataResult result = new ContainerDataResult();
				while (!dataIter.isEmpty()) {
					dataIter.peek(result);
					long from = result.getDataIndex();
					long last = from;
					while (!dataIter.isEmpty()) {
						dataIter.peek(result);
						long to = result.getDataIndex();
						if (isGap(last, to)) {
							break;
						}
						if (!result.isPotDataGap()) {
							last = to;
						}
						dataIter.remove();
					}
					addIndexRange(from, last);
				}
			}
		}

		/**
		 * Prüft ob zwischen dem Index start und end eine Lücke ist.
		 *
		 * @param start StartIndex (inklusive)
		 * @param end   EndIndex (inklusive)
		 * @return true falls die laufende Nummern der Datenindexe sich um mehr als den Wert 1 unterscheiden
		 */
		private static boolean isGap(long start, long end) {
			return start != -1 && Util.dIdxNoModBits(end) - Util.dIdxNoModBits(start) > 1;
		}

		@Override
		public DataIterator iterator() {
			return new DeletedFullIterator(_deletedBlocks, this, _containerDir.archiveDataKind());
		}

		public void addIndexRange(final long from, final long to) {
			_deletedBlocks.add(new IndexRange(from, to));
		}

		public void serialize(final byte[] tmpHeaderData) {
			System.arraycopy(_data, 0, tmpHeaderData, 0, _data.length);
		}

		public List<IndexRange> getDeletedBlocks() {
			return _deletedBlocks;
		}

		@Override
		public String getContainerHeaderParamAsString(final ContainerManagementInformation param) {
			return INDEX_CONTENT_DESCRIPTOR.getColumn(param).readString(_data);
		}

		@Override
		public boolean getContainerHeaderParamAsBoolean(final ContainerManagementInformation param) {
			return "1".equals(getContainerHeaderParamAsString(param));
		}

		@Override
		public long getContainerHeaderParamAsLong(final ContainerManagementInformation param) {
			return INDEX_CONTENT_DESCRIPTOR.getColumn(param).readLong(_data);
		}

		@Override
		public int getContainerHeaderParamAsInt(final ContainerManagementInformation param) {
			return (int) getContainerHeaderParamAsLong(param);
		}

		@Override
		public ContainerDirectory getLocation() {
			return _containerDir;
		}
	}

	/**
	 * Ein Datenindex-Bereich
	 */
	public record IndexRange(long from, long to) {
		/**
		 * Erstellt eine neue Instanz
		 *
		 * @param from Von-Datenindex (inklusiv)
		 * @param to   Bis-Datenindex (inklusiv)
		 */
		public IndexRange {
		}

		/**
		 * Gibt den Von-Datenindex zurück
		 *
		 * @return den Von-Datenindex
		 */
		@Override
		public long from() {
			return from;
		}

		/**
		 * Gibt den Bis-Datenindex zurück
		 *
		 * @return den Bis-Datenindex
		 */
		@Override
		public long to() {
			return to;
		}

		@Override
		public boolean equals(final Object o) {
			if (this == o) return true;
			if (o == null || getClass() != o.getClass()) return false;

			final IndexRange that = (IndexRange) o;

			if (from != that.from) return false;
			return to == that.to;
		}

		@Override
		public int hashCode() {
			int result = (int) (from ^ (from >>> 32));
			result = 31 * result + (int) (to ^ (to >>> 32));
			return result;
		}
	}

	/**
	 * Iterator über die gespeicherten Datenindizes eines gelöschten Containers
	 */
	private static class DeletedFullIterator implements DataIterator {
		private final List<Long> _indexes = new ArrayList<>();
		int _idx;
		private final ContainerManagementData _handle;
		private final ArchiveDataKind _adk;

		/**
		 * Erstellt einen neuen DeletedFullIterator
		 *
		 * @param deletedBlocks Indexbereiche mit gelöschten Daten
		 * @param adk           Archivdatenart
		 * @param handle        Containerheader
		 */
		public DeletedFullIterator(final List<IndexRange> deletedBlocks, final ContainerManagementData handle, final ArchiveDataKind adk) {
			_handle = handle;
			_adk = adk;
			for (IndexRange deletedBlock : deletedBlocks) {
				long from = deletedBlock.from;
				long to = deletedBlock.to;
				for (long l = from; l <= to; l += 4) {
					_indexes.add(l);
				}
			}
		}

		@Override
		public void peek(final ContainerDataResult result) throws PersistenceException {
			if (_idx >= _indexes.size()) throw new NoSuchElementException();
			result.setContainerID(_handle.getContainerHeaderParamAsLong(ContainerManagementInformation.CHP_CONT_ID));
			result.setDataIndex(_indexes.get(_idx));
			result.setDataTime(0);
			result.setArchiveTime(0);
			result.setDataKind(_adk);
			result.setData(null);
			result.setCompressed(false);
			result.setDataSize(0);
			result.setDataUncompressedSize(0);
			result.setDataState(DataState.DATA);
		}

		@Override
		public ContainerDataResult peekNext() {
			return null;
		}

		@Override
		public long peekDataIndex() {
			if (_idx >= _indexes.size()) throw new NoSuchElementException();
			return _indexes.get(_idx);
		}

		@Override
		public long peekDataTime() {
			return 0;
		}

		@Override
		public long peekArchiveTime() {
			return 0;
		}

		@Override
		public void remove() {
			if (_idx >= _indexes.size()) throw new NoSuchElementException();
			_idx++;
		}

		@Override
		public boolean isEmpty() {
			return _idx >= _indexes.size();
		}

		@Override
		public void close() {
		}

		@NotNull
		@Override
		public ContainerManagementData getContainerManagementData() {
			return _handle;
		}
	}
}
