/*
 *
 * Copyright 2017-2020 by Kappich Systemberatung, Aachen
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.ars.ars.persistence;

import de.bsvrz.ars.ars.persistence.iter.DataIterator;
import de.bsvrz.dav.daf.main.DataState;
import de.bsvrz.dav.daf.main.archive.ArchiveDataKind;
import de.bsvrz.sys.funclib.kappich.annotations.NotNull;

import java.util.NoSuchElementException;

/**
 * Iterator über die Datensätze eines gelöschten Containers. Es werden genau 2 künstliche Einträge zurückgegeben mit den Minimum-Werten und dann den Maximum-Werten aus den Headern.
 * Daten können nicht abgefragt werden.
 *
 * @author Kappich Systemberatung
 */
public final class DeletedMinMaxIterator implements DataIterator, ContainerManagementData {

	private final ContainerHeaders _headers;
	private int _position;
	private final ArchiveDataKind _adk;
	private final IdDataIdentification _dataIdentification;

	/**
	 * Erteugt einen neuen DeletedMinMaxIterator
	 * @param headers Header des Containers
	 * @param adk Archivdatenart
	 * @param dataIdentification Datenidentifikation
	 */
	public DeletedMinMaxIterator(final ContainerHeaders headers, final ArchiveDataKind adk, final IdDataIdentification dataIdentification) {
		_headers = headers;
		_adk = adk;
		_dataIdentification = dataIdentification;
	}

	@Override
	public void peek(final ContainerDataResult result) {
		switch (_position) {
			case 0 -> {
				result.setContainerID(getContainerHeaderParamAsLong(ContainerManagementInformation.CHP_CONT_ID));
				result.setDataIndex(getContainerHeaderParamAsLong(ContainerManagementInformation.CHP_DATA_IDX_MIN));
				result.setDataTime(getContainerHeaderParamAsLong(ContainerManagementInformation.CHP_DATA_TIME_MIN));
				result.setArchiveTime(getContainerHeaderParamAsLong(ContainerManagementInformation.CHP_ARC_TIME_MIN));
				result.setDataKind(_adk);
				result.setData(null);
				result.setCompressed(false);
				result.setDataSize(0);
				result.setDataUncompressedSize(0);
				result.setDataState(DataState.DATA);
			}
			case 1 -> {
				result.setContainerID(getContainerHeaderParamAsLong(ContainerManagementInformation.CHP_CONT_ID));
				result.setDataIndex(getContainerHeaderParamAsLong(ContainerManagementInformation.CHP_DATA_IDX_MAX));
				result.setDataTime(getContainerHeaderParamAsLong(ContainerManagementInformation.CHP_DATA_TIME_MAX));
				result.setArchiveTime(getContainerHeaderParamAsLong(ContainerManagementInformation.CHP_ARC_TIME_MAX));
				result.setDataKind(_adk);
				result.setData(null);
				result.setCompressed(false);
				result.setDataSize(0);
				result.setDataUncompressedSize(0);
				result.setDataState(DataState.DATA);
			}
			default -> throw new NoSuchElementException();
		}
	}

	@Override
	public ContainerDataResult peekNext() {
		return null;
	}

	@Override
	public long peekDataIndex() {
		return switch (_position) {
			case 0 -> getContainerHeaderParamAsLong(ContainerManagementInformation.CHP_DATA_IDX_MIN);
			case 1 -> getContainerHeaderParamAsLong(ContainerManagementInformation.CHP_DATA_IDX_MAX);
			default -> throw new NoSuchElementException();
		};
	}

	@Override
	public long peekDataTime() {
		return switch (_position) {
			case 0 -> getContainerHeaderParamAsLong(ContainerManagementInformation.CHP_DATA_TIME_MIN);
			case 1 -> getContainerHeaderParamAsLong(ContainerManagementInformation.CHP_DATA_TIME_MAX);
			default -> throw new NoSuchElementException();
		};
	}

	@Override
	public long peekArchiveTime() {
		return switch (_position) {
			case 0 -> getContainerHeaderParamAsLong(ContainerManagementInformation.CHP_ARC_TIME_MIN);
			case 1 -> getContainerHeaderParamAsLong(ContainerManagementInformation.CHP_ARC_TIME_MAX);
			default -> throw new NoSuchElementException();
		};
	}

	@Override
	public void remove() {
		if(_position > 1) throw new NoSuchElementException();
		_position++;
	}

	@Override
	public boolean isEmpty() {
		return _position > 1;
	}

	@Override
	public void close() {
	}

	@NotNull
	@Override
	public ContainerManagementData getContainerManagementData() {
		return this;
	}

	@Override
	public String getContainerHeaderParamAsString(final ContainerManagementInformation param) {
		return _headers.getContainerHeaderParamAsString(param);
	}

	@Override
	public boolean getContainerHeaderParamAsBoolean(final ContainerManagementInformation param) {
		return _headers.getContainerHeaderParamAsBoolean(param);
	}

	@Override
	public long getContainerHeaderParamAsLong(final ContainerManagementInformation param) {
		return _headers.getContainerHeaderParamAsLong(param);
	}

	@Override
	public int getContainerHeaderParamAsInt(final ContainerManagementInformation param) {
		return _headers.getContainerHeaderParamAsInt(param);
	}

	@Override
	public IdContainerFileDir getLocation() {
		return new IdContainerFileDir(_dataIdentification, _adk);
	}

}
