/*
 * Copyright 2019 by Kappich Systemberatung, Aachen
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.ars.ars.persistence;

import de.bsvrz.dav.daf.main.archive.ArchiveDataKind;
import de.bsvrz.sys.funclib.dataIdentificationSettings.DataIdentification;

import java.util.Objects;

/**
 * Klasse, die eine Datenidentifikation mit Archivdatenart, also den Speicherort von Containerdaten repräsentiert.
 * <p>
 * Diese Klasse trifft keine Aussage über einen konkreten Speicherort auf der Festplatte oder
 * eventuelle Wochenverzeichnisse.
 * <p>
 * Diese Klasse ist <em>value based</em>,
 * <em>immutable</em> und es werden keine Aussagen über Objektidentität getroffen.
 *
 * @author Kappich Systemberatung
 */
public final class IdContainerFileDir implements ContainerDirectory {

	/**
	 * Datenidentifikation
	 */
	private final IdDataIdentification idDataIdentification;

	/**
	 * Archivdatenart
	 */
	private final ArchiveDataKind archiveDataKind;

	/**
	 * Erzeugt ein neues Objekt mit den angegebenen Werten
	 *
	 * @param objectId        Id des Systemobjekts dieser Datenidentifikation
	 * @param atgId           Id der Attributgruppe dieser Datenidentifikation
	 * @param aspectId        Id des Aspekts dieser Datenidentifikation
	 * @param simVariant      Simulationsvariante dieser Datenidentifikation
	 * @param archiveDataKind Archivdatenart
	 */
	public IdContainerFileDir(final long objectId, final long atgId, final long aspectId, final int simVariant, final ArchiveDataKind archiveDataKind) {
		this.idDataIdentification = new IdDataIdentification(objectId, atgId, aspectId, simVariant);
		this.archiveDataKind = Objects.requireNonNull(archiveDataKind);
	}

	/**
	 * Erstellt eine neue Instanz aus einer {@link DataIdentification}
	 *
	 * @param dataIdentification {@link DataIdentification}
	 * @param archiveDataKind    Archivdatenart
	 */
	public IdContainerFileDir(final IdDataIdentification dataIdentification, final ArchiveDataKind archiveDataKind) {
		this.idDataIdentification = Objects.requireNonNull(dataIdentification);
		this.archiveDataKind = Objects.requireNonNull(archiveDataKind);
	}

	/**
	 * Konvertiert eine beliebige Implementierung von {@link ContainerDirectory} in ein IdContainerFileDir,
	 * z. B. damit eine einheitliche Implementierung der Klasse als Map-Key benutzt werden kann.
	 *
	 * @param containerDirectory Referenz auf die gelockte Datenidentifikation und Datenart für den Zugriff auf Containerdaten
	 *                           Verzeichnis
	 * @return IdContainerFileDir
	 */
	public static IdContainerFileDir of(ContainerDirectory containerDirectory) {
		if (containerDirectory instanceof IdContainerFileDir dir) {
			return dir;
		}
		return new IdContainerFileDir(containerDirectory.dataIdentification(), containerDirectory.archiveDataKind());
	}

	@Override
	public boolean equals(Object o) {
		if (this == o) return true;
		if (o == null || getClass() != o.getClass()) return false;

		IdContainerFileDir that = (IdContainerFileDir) o;

		if (!idDataIdentification.equals(that.idDataIdentification)) return false;
		return archiveDataKind.equals(that.archiveDataKind);
	}

	@Override
	public int hashCode() {
		int result = idDataIdentification.hashCode();
		result = 31 * result + archiveDataKind.hashCode();
		return result;
	}

	/**
	 * Gibt die Archivdatenart zurück
	 *
	 * @return die Archivdatenart
	 */
	@Override
	public ArchiveDataKind archiveDataKind() {
		return archiveDataKind;
	}

	/**
	 * Gibt die Datenidentifikation zurück
	 *
	 * @return Datenidentifikation
	 */
	@Override
	public IdDataIdentification dataIdentification() {
		return idDataIdentification;
	}

	public String toString() {
		return "obj: " + getObjectId() + " atg: " + getAtgId() + " asp: " + getAspectId() + " sv: " + getSimVariant() + " adk: " + archiveDataKind;
	}
}
