/*
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */
package de.bsvrz.ars.ars.persistence;

import de.bsvrz.ars.ars.mgmt.datatree.IndexTree;
import de.bsvrz.ars.ars.persistence.directories.PersistenceDirectory;
import de.bsvrz.ars.ars.persistence.directories.cache.DataRange;
import de.bsvrz.ars.ars.persistence.directories.cache.ValidDataRange;
import de.bsvrz.ars.ars.persistence.index.DataIndexIndex;
import de.bsvrz.ars.ars.persistence.index.IndexException;
import de.bsvrz.ars.ars.persistence.index.backend.management.BaseIndex;

import java.util.*;
import java.util.function.BiFunction;
import java.util.function.Function;
import java.util.function.ToLongFunction;

public class IndexAggregator<T extends BaseIndex<?>> {
	private final List<IndexAndDirectory<T>> result = new ArrayList<>();
	private final ToLongFunction<ValidDataRange> minSupplier;
	private final ToLongFunction<ValidDataRange> maxSupplier;
	private final IndexSupplier<T> indexSupplier;

	public IndexAggregator(ToLongFunction<ValidDataRange> minSupplier, ToLongFunction<ValidDataRange> maxSupplier, IndexSupplier<T> indexSupplier) {
		this.minSupplier = minSupplier;
		this.maxSupplier = maxSupplier;
		this.indexSupplier = indexSupplier;
	}

	public void aggregate(Collection<? extends PersistenceDirectory> directories, LockedContainerDirectory dataIdentification, long lowerBound, long upperBound) throws IndexException {

		final List<PersistenceDirectory> directoriesInRange = new ArrayList<>();
		PersistenceDirectory latestBefore = null;
		long latestBeforeValue = Long.MIN_VALUE;

		for (PersistenceDirectory it : directories) {
			DataRange dataRange = it.getDataRange(dataIdentification);
			if (dataRange instanceof ValidDataRange validDataRange) {
				long directoryMinimum = minSupplier.applyAsLong(validDataRange);
				long directoryMaximum = maxSupplier.applyAsLong(validDataRange);

				if (directoryMaximum < lowerBound && (latestBefore == null || directoryMaximum > latestBeforeValue)) {
					latestBeforeValue = directoryMaximum;
					latestBefore = it;
				}

				if (directoryMinimum <= upperBound && directoryMaximum >= lowerBound) {
					directoriesInRange.add(it);
				}
			}
		}

		result.clear();
		if (latestBefore != null) {
			// Vorhergehendes Verzeichnis einfügen, vielleicht gibt es dort einen Anfangszustand...
			addResult(dataIdentification, latestBefore);
		}
		for (PersistenceDirectory directory : directoriesInRange) {
			addResult(dataIdentification, directory);
		}
	}

	private void addResult(LockedContainerDirectory dataIdentification, PersistenceDirectory directory) throws IndexException {
		Optional<? extends T> optionalIndex = indexSupplier.apply(directory.getIndexTree(), dataIdentification);
		optionalIndex.ifPresent(i -> result.add(new IndexAndDirectory<>(i, directory)));
	}

	public List<T> getIndexes() {
		ArrayList<T> tmp = new ArrayList<>(result.size());
		for (IndexAndDirectory<T> item : result) {
			tmp.add(item.index);
		}
		return tmp;
	}

	public Map<T, PersistenceDirectory> getMap() {
		HashMap<T, PersistenceDirectory> tmp = new HashMap<>(result.size());
		for (IndexAndDirectory<T> item : result) {
			tmp.put(item.index, item.directory);
		}
		return tmp;
	}

	record IndexAndDirectory<T extends BaseIndex<?>>(T index, PersistenceDirectory directory) {
	}

	interface IndexSupplier<T> {
		Optional<? extends T> apply(IndexTree indexTree, LockedContainerDirectory containerDirectory) throws IndexException;
	}
}
