/*
 *
 * Copyright 2005-2008 by beck et al. projects GmbH, Munich
 * Copyright 2009-2020 by Kappich Systemberatung, Aachen
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.ars.ars.persistence;

import com.google.common.collect.Lists;
import com.google.common.collect.Multimap;
import com.google.common.collect.MultimapBuilder;
import com.google.common.collect.SetMultimap;
import de.bsvrz.ars.ars.mgmt.ArchiveManager;
import de.bsvrz.ars.ars.mgmt.TaskManager;
import de.bsvrz.ars.ars.mgmt.TaskManagerInterface;
import de.bsvrz.ars.ars.mgmt.RuntimeControl;
import de.bsvrz.ars.ars.mgmt.datatree.DataIdentNode;
import de.bsvrz.ars.ars.mgmt.datatree.DataIdentTree;
import de.bsvrz.ars.ars.mgmt.datatree.IndexTree;
import de.bsvrz.ars.ars.mgmt.datatree.synchronization.*;
import de.bsvrz.ars.ars.mgmt.tasks.Task;
import de.bsvrz.ars.ars.persistence.directories.ActivePersistenceDirectory;
import de.bsvrz.ars.ars.persistence.directories.PersistenceDirectory;
import de.bsvrz.ars.ars.persistence.directories.cache.ValidDataRange;
import de.bsvrz.ars.ars.persistence.directories.mgmt.PersistenceDirectoryManager;
import de.bsvrz.ars.ars.persistence.directories.mgmt.SingletonPersistenceDirectoryManager;
import de.bsvrz.ars.ars.persistence.directories.mgmt.TimeBasedPersistenceDirectoryManager;
import de.bsvrz.ars.ars.persistence.directories.mgmt.lock.DirectoryIsLockedException;
import de.bsvrz.ars.ars.persistence.directories.mgmt.range.TimeDomain;
import de.bsvrz.ars.ars.persistence.index.*;
import de.bsvrz.ars.ars.persistence.index.result.IndexResult;
import de.bsvrz.ars.ars.persistence.index.result.LocatedIndexResult;
import de.bsvrz.ars.ars.persistence.iter.DataGapManager;
import de.bsvrz.ars.ars.persistence.iter.DataIterator;
import de.bsvrz.ars.ars.persistence.sequence.AllDataSpecification;
import de.bsvrz.ars.ars.persistence.sequence.ArchiveTimeSequenceSpecification;
import de.bsvrz.ars.ars.persistence.sequence.DataIndexSequenceSpecification;
import de.bsvrz.ars.ars.persistence.sequence.SequenceSpecification;
import de.bsvrz.ars.ars.persistence.writer.ArchiveTask;
import de.bsvrz.dav.daf.main.ClientDavInterface;
import de.bsvrz.dav.daf.main.archive.ArchiveDataKind;
import de.bsvrz.dav.daf.main.archive.ArchiveDataKindCombination;
import de.bsvrz.dav.daf.main.archive.ArchiveTimeSpecification;
import de.bsvrz.dav.daf.main.archive.TimingType;
import de.bsvrz.dav.daf.main.config.DataModel;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.sys.funclib.debug.Debug;
import de.bsvrz.sys.funclib.kappich.annotations.NotNull;
import de.bsvrz.sys.funclib.kappich.annotations.Nullable;
import de.bsvrz.sys.funclib.losb.util.Util;

import java.io.*;
import java.nio.file.*;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.*;
import java.util.concurrent.atomic.AtomicLong;
import java.util.zip.GZIPInputStream;
import java.util.zip.GZIPOutputStream;

/**
 * Zentrale Persistenz-Verwaltung für die Archivierung von Daten auf einem Speichermedium vom Typ A (Festplatte) und die entsprechende Meta-Daten-Verwaltung.
 *
 * @author beck et al. projects GmbH
 * @author Thomas Schaefer
 * @author Alexander Schmidt
 * @version $Revision$ / $Date$ / ($Author$)
 */
public final class PersistenceManager implements ContainerCreator, DataGapManager, TaskManager {

	private static final Debug _debug = Debug.getLogger();

	/**
	 * Name des Verzeichnisses für das Datenkonsistente Backup
	 */
	public static final String BACKUP_DIR_NAME = "backup";

	/**
	 * Name der Datei mit den Datenzeitinformationen für den Neustart
	 */
	public static final String RESTART_TIME_FILE_NAME = "_restartTime.property";

	/**
	 * Sollen defekte Container gelöscht bzw. umbenannt werden (statisch, globale Einstellung)
	 */
	private static boolean _deleteBrokenContainers;

	/**
	 * Die Verwaltung der Unterverzeichnisse/Wochenverzeichnisse zurück.
	 */
	private final PersistenceDirectoryManager _persistenceDirectoryManager;

	/**
	 * Verzeichnis für die Spezifikation
	 */
	private final Path backupDirectory;

	/**
	 * Klasse für Parameterierungsinformationen zu Datenidentifikationen
	 */
	private final DataIdentTree _dataIdentTree = new DataIdentTree();

	/**
	 * Synchronisierung auf Datenidentifikationen (für Indexzugriff)
	 */
	private final SynchronizationManager<IdDataIdentification> _indexLockManager;

	/**
	 * Setzt, ob defekte Containerdateien umbenannt werden sollen
	 *
	 * @param deleteBrokenContainers Sollen defekte Containerdateien umbenannt werden?
	 */
	public static void setDeleteBrokenContainers(final boolean deleteBrokenContainers) {
		_deleteBrokenContainers = deleteBrokenContainers;
	}

	@Override
	public boolean shouldDeleteBrokenContainers() {
		return _deleteBrokenContainers;
	}

	/**
	 * Enthält die zuletzt verwendete ContainerID
	 */
	private AtomicLong nextContainerID = new AtomicLong(1);

	private final TaskManagerInterface archivMgr;

	private final StartupProperties startupProps;

	private boolean _preventWriteStartupInfo = true;

	private final Path _unsubscriptionFile;

	/**
	 * Erzeugt den Persistenz-Manager.
	 *
	 * @param archMgr  Archiv-Manager oder andere Implementierung von {@link TaskManagerInterface} (insb. für Tests)
	 * @param archPath Archivierungs-Verzeichnis
	 * @param domain   Klasse, die die Unterteilung der Persistenzverzeichnisse definiert
	 */
	public PersistenceManager(TaskManagerInterface archMgr, Path archPath, TimeDomain<?> domain) {
		archivMgr = Objects.requireNonNull(archMgr);
		if (domain == null) {
			_persistenceDirectoryManager = new SingletonPersistenceDirectoryManager(this, archPath);
		} else {
			_persistenceDirectoryManager = new TimeBasedPersistenceDirectoryManager<>(this, archPath, domain);
		}
		backupDirectory = archPath.resolve(BACKUP_DIR_NAME);
		startupProps = new StartupProperties(archPath);
		_unsubscriptionFile = archPath.resolve(RESTART_TIME_FILE_NAME);
		if (Files.exists(_unsubscriptionFile)) {
			try {
				loadUnsubscriptionTime();
			} catch (IOException e) {
				_debug.warning("Kann " + _unsubscriptionFile + " nicht laden", e);
			}
		}
		_indexLockManager = new DebuggingSynchronizationManager<>(new SynchronizationManagerImpl<>((element) -> {
			for (PersistenceDirectory directory : getPersistenceDirectories(element.getSimVariant())) {
				directory.getIndexTree().ensureNoCached();
			}
		}, (element) -> {
			for (PersistenceDirectory directory : getPersistenceDirectories(element.getSimVariant())) {
				directory.getIndexTree().closeIndexes();
			}
		}));
	}
	
	/**
	 * Gibt den Dateipfad zurück, in dem Lückendateien gespeichert werden sollen
	 * @param dataIdentification Datenidentifikation
	 * @return Dateipfad
	 */
	@NotNull
	public Path getGapFilePath(IdDataIdentification dataIdentification) {
		return _persistenceDirectoryManager.getGapFilePath(dataIdentification);
	}

	/**
	 * Initialisiert die vorhandenen Wochenverzeichnisse von der Festplatte
	 * @throws IOException Lesefehler
	 * @throws InterruptedException Unterbrochen
	 * @throws PersistenceException Lesefehler 
	 * @throws DirectoryIsLockedException Das aktuelle Persistenzverzeichnis ist noch gesperrt (_isActive.flag-Datei existiert)
	 */
	public void initialize() throws IOException, InterruptedException, PersistenceException, DirectoryIsLockedException {
		_persistenceDirectoryManager.initialize();
	}

	//////////////////////////////////////////////////////////////////////////////////////////////////////////
	// Container: Verzeichnis
	//////////////////////////////////////////////////////////////////////////////////////////////////////////

	/**
	 * Diese Methode loescht das komplette Verzeichnis eines {@link DataIdentNode} mit allen Unterverzeichnissen der Datensatzarten und aller darin befindlichen
	 * Datencontainern; damit werden alle archivierten Datensätze einer Simulationsvariante geloescht. Der {@link DataIdentNode} muss in der Verwaltung aus dem
	 * {@link DataIdentTree} geloescht werden. Falls die Simulationsvariante gleich null ist, oder das Verzeichnis nicht geloescht werden konnte, wird eine {@link
	 * PersistenceException} geworfen.
	 *
	 * @param syncKey Schlüssel für Zugriff auf Indexknoten
	 * @throws PersistenceException Schreibfehler im Persistenzverzeichnis
	 */
	public void deleteSimVar(final SyncKey<IdDataIdentification> syncKey) throws PersistenceException {

		IdDataIdentification dataIdentification = syncKey.getElement();
		if (dataIdentification.getSimVariant() == 0)
			throw new PersistenceException("Simulations-Variante '0' (Echtdaten) darf nicht geloescht werden!");

		int simVariant = dataIdentification.getSimVariant();
		ActivePersistenceDirectory directory = getPersistenceDirectoryManager().getSimulationPersistenceDirectory(simVariant);

		if (directory == null) {
			// Existiert nicht, braucht also auch nicht gelöscht zu werden...
			return;
		}
		
		try {
			final CacheManager cacheManager = CacheManager.getInstance();
			cacheManager.forgetCache(directory.getOpenContID(dataIdentification.resolve(ArchiveDataKind.ONLINE)));
			cacheManager.forgetCache(directory.getOpenContID(dataIdentification.resolve(ArchiveDataKind.ONLINE_DELAYED)));
			cacheManager.forgetCache(directory.getOpenContID(dataIdentification.resolve(ArchiveDataKind.REQUESTED)));
			cacheManager.forgetCache(directory.getOpenContID(dataIdentification.resolve(ArchiveDataKind.REQUESTED_DELAYED)));

			for (ArchiveDataKind archiveDataKind : ArchiveDataKindCombination.all()) {
				// Indexe schließen
				directory.getIndexTree().closeIndexes(new LockedContainerDirectory(syncKey, archiveDataKind));
				directory.deleteOpenContainerData(syncKey, archiveDataKind);
			}
			Path path = directory.getPath(dataIdentification);
			try {
				deletePath(path);
			} catch (IOException e) {
				throw new PersistenceException("Simulations-Varianten-Verzeichnis konnte nicht geloescht werden: " + path, e);
			}
		} catch (Exception e) {
			_debug.warning("Fehler beim Vergessen des Caches", e);
		}
	}

	/**
	 * Löscht ein Verzeichnis inklusive enthaltener Dateien.
	 *
	 * @param path Verzeichnis
	 * @throws IOException Mögliche Exception
	 */
	public static void deletePath(final Path path) throws IOException {
		if (!Files.exists(path)) return;
		Files.walkFileTree(path, new SimpleFileVisitor<>() {
			@Override
			public FileVisitResult visitFile(Path file, BasicFileAttributes attrs)
			throws IOException {
				Files.deleteIfExists(file);
				return FileVisitResult.CONTINUE;
			}

			@Override
			public FileVisitResult visitFileFailed(Path file, IOException exc) {
				return FileVisitResult.CONTINUE;
			}

			@Override
			public FileVisitResult postVisitDirectory(Path dir, IOException exc) throws IOException {
				Files.deleteIfExists(dir);
				return FileVisitResult.CONTINUE;
			}
		});
	}


	/**
	 * Die Methode prepareShutdown wird beim Herunterfahren des Archivsystems ausgeführt.
	 * @throws InterruptedException Wenn der Thread beim Warten auf das Schließen der Indexdateien unterbrochen wurde
	 */
	public void prepareShutdown() throws InterruptedException {

		CacheManager.getInstance().close();

		// Beenden darf nicht gestartet werden, wenn vom gleichen Thread Locks gehalten werden
		// (DeadLock mit IndexCloserWorker möglich)
		assertNoLocks();

		ActivePersistenceDirectory persistenceDirectory = getPersistenceDirectoryManager().getActivePersistenceDirectory();
		if (persistenceDirectory != null) {
			persistenceDirectory.closeIndexes(archivMgr.getNumCloseIndexThreads());
		}

		getPersistenceDirectoryManager().shutDown();

		if (_preventWriteStartupInfo) {
			_debug
					.warning("Herunterfahren: Startup-Info wird unterdrückt, weil beim Wiederherstellungslauf ein Fehler aufgetreten ist");
			return;
		}
		try {
			startupProps.setVal(StartupProperties.STUP_MAX_CONT_ID, getLastContainerID());
			startupProps.setVal(StartupProperties.STUP_LAST_ATIME, ArchiveTask.getLastArchiveTime());
			startupProps.writeStartUpProperties();
		} catch (PersistenceException e) {
			_debug.error("Herunterfahren: Fehler beim Erzeugen der Startup-Info: " + e.getMessage());
		}
	}

	/**
	 * Versucht, die StartUp-Properties-Datei einzulesen. Wenn das nicht gelingt, wird das gesamte Persistenzverzeichnis ({@link RestorePersDirTsk}) durchlaufen
	 * und versucht, einen gueltigen Startpunkt wiederherzustellen. Das Persistenzverzeichnis wird in {@code InQueuesMgr.NUM_OF_ARCH_QUEUES_ONLINE} Teile
	 * aufgeteilt, die jeweils von einem Thread bearbeitet werden (zwecks Performance). Am Schluss wird die StartUp-Properties-Datei geloescht. Beim Herunterfahren
	 * wird in {@link #prepareShutdown()} die Datei neu geschrieben. Daran kann das ArS beim nächsten Start erkennen, ob es ordnungsgemäß
	 * heruntergefahren wurde oder ob ein Wiederherstellungslauf erforderlich ist.
	 *
	 * @return Wahr, falls der Durchlauf des Persistenzverzeichnisses erfolgreich war und das Archivsystem starten kann, falsch sonst.
	 * @param rebuildMode Wiederherstellungsmodus
	 */
	public boolean startupProcedure(RebuildMode rebuildMode) {
		long mxAT;
		try {
			startupProps.readStartUpProperties();
			nextContainerID.set(startupProps.getValAsLong(StartupProperties.STUP_MAX_CONT_ID));
			mxAT = startupProps.getValAsLong(StartupProperties.STUP_LAST_ATIME);
		} catch (PersistenceException e) {
			_debug.info(e.getMessage() + Debug.NEWLINE + "Starte Wiederherstellungslauf im Modus '" + rebuildMode + "'");
			RestorePersDirTsk restorePersDirTsk = new RestorePersDirTsk(this, "Wiederherstellungslauf", rebuildMode);
			while (!restorePersDirTsk.isTerminated()) {
				try {
					if (archivMgr.wasTerminated()) {        // Archivsystem per Telnet beendet?
						restorePersDirTsk.terminateTask();
						return false;
					}
					//noinspection BusyWait
					Thread.sleep(100);
				} catch (InterruptedException ie) {
					_debug.info("Wiederherstellungslauf abgebrochen.", ie);
					return false;
				}
			}
			if (!restorePersDirTsk.getWorker().success()) {
				_debug.error("Wiederherstellungslauf fehlgeschlagen.");
				return false;
			}
			nextContainerID = new AtomicLong(restorePersDirTsk.getWorker().getMaxContID());
			mxAT = restorePersDirTsk.getWorker().getMaxArchiveTime();
		}
		_preventWriteStartupInfo = false;

		RuntimeControl runtimeControl = archivMgr.getRuntimeControl();
		if(runtimeControl == null) {
			throw new AssertionError("Fehler in Initialisierungsreihenfolge");
		}
		long now = runtimeControl.getSystemTime();
		ArchiveTask.setLastArchiveTime(mxAT > 0 ? mxAT : now);    // gemäß TAnfArS 5.1.2.4.3.3

		_debug.info(
				"StartupInfo: max. ContainerID=" + nextContainerID + ", max. Archivzeit="
						+ Util.timestrMillisFormatted(mxAT) + (mxAT <= 0 ? " (nicht ermittelbar, wurde auf " + Util.timestrMillisFormatted(now) + " gesetzt)" : "")
		);

		try {
			startupProps.deleteStartupPropsFile();
		} catch (IOException e) {
			// Gibt zwar ein Problem beim Beenden des Archivsystems, sollte aber den Start nicht verhindern
			_debug.warning("StartUp-Properties konnten nicht geloescht werden: " + startupProps, e);
		}

		return true;
	}

	/**
	 * Gibt Statistiken von diesem PersistenceManager-Objekt zurück.
	 *
	 * @return Statistiken (Typ Statistics)
	 */
	public Statistics getStatistics() {
		return new Statistics();
	}

	//////////////////////////////////////////////////////////////////////////////////////////////////////////
	// Private Methoden
	//////////////////////////////////////////////////////////////////////////////////////////////////////////

	/**
	 * Erhoeht nextContainerID um 1 und gibt den Wert zurück.
	 *
	 * @return als nächste zu verwendende Container-ID
	 */
	@Override
	public long nextContainerID() {
		return nextContainerID.incrementAndGet();
	}

	@Override
	public int getCloseThreadCount() {
		return archivMgr.getNumCloseIndexThreads();
	}

	/**
	 * @return Letzte vergebe ContainerID
	 */
	public long getLastContainerID() {
		return nextContainerID.longValue();
	}

	/**
	 * Sichert den letzten Zeitpunkt, wo gültige Daten empfangen wurden bzw. beim Beenden wo Daten abgemeldet wurden. Dieser Zeitpunkt wird nach einem Neustart als
	 * Grundlage für die Bildung von potenziellen Datenlücken verwendet.
	 */
	public void saveUnsubscriptionTime() {
		Multimap<Long, IdDataIdentification> didForUnsubscriptionTime = MultimapBuilder.treeKeys().arrayListValues().build();
		RuntimeControl runtimeControl = archivMgr.getRuntimeControl();
		if (runtimeControl == null) {
			// Archivsystem war noch nicht initialisiert
			return;
		}

		final Long now = runtimeControl.getSystemTime();

		for (DataIdentNode node : _dataIdentTree) {
			if (node.hasValidData()) {
				didForUnsubscriptionTime.put(now, node.getDataIdentification());
			} else {
				long unsubscriptionTime = node.getUnsubscriptionTime();
				if (unsubscriptionTime != -1) {
					didForUnsubscriptionTime.put(unsubscriptionTime, node.getDataIdentification());
				}
			}
		}

		saveUnsubscriptionTime(didForUnsubscriptionTime, _unsubscriptionFile);
	}

	/**
	 * Sichert den letzten Zeitpunkt, wo gültige Daten empfangen wurden bzw. beim Beenden wo Daten abgemeldet wurden. Dieser Zeitpunkt wird nach einem Neustart als
	 * Grundlage für die Bildung von potenziellen Datenlücken verwendet.
	 *
	 * @param didForUnsubscriptionTime Zu schreibende Zeitstempel
	 * @param unsubscriptionFile       Datei, die geschrieben wird.
	 */
	public void saveUnsubscriptionTime(final Multimap<Long, IdDataIdentification> didForUnsubscriptionTime, final Path unsubscriptionFile) {
		try {
			try (DataOutputStream stream = new DataOutputStream(new BufferedOutputStream(new GZIPOutputStream(Files.newOutputStream(unsubscriptionFile))))) {
				Set<Map.Entry<Long, Collection<IdDataIdentification>>> entries = didForUnsubscriptionTime.asMap().entrySet();
				stream.writeInt(entries.size());
				for (Map.Entry<Long, Collection<IdDataIdentification>> entry : entries) {
					stream.writeLong(entry.getKey());
					List<IdDataIdentification> values = (List<IdDataIdentification>) entry.getValue();
					values.sort(null);
					stream.writeInt(values.size());
					for (IdDataIdentification value : values) {
						stream.writeLong(value.getObjectId());
						stream.writeLong(value.getAtgId());
						stream.writeLong(value.getAspectId());
						stream.writeShort(value.getSimVariant());
					}
				}
			}
		} catch (IOException e) {
			_debug
					.error("Fehler beim Schreiben der Datenzeit: " + RESTART_TIME_FILE_NAME + " ist nicht beschreibbar.", e);
		}
	}

	/**
	 * Lädt die mit {@link #saveUnsubscriptionTime()} gespeicherten Zeiten wieder ein und speichert sie im DataIdentTree. Hierbei können überflüssige Knoten entstehen, aber das ist nicht zu verhindern da wir nicht immer wissen,
	 * welche Zeiten wir noch brauchen.
	 *
	 * @throws IOException Allgemeiner IO-Fehler
	 */
	private void loadUnsubscriptionTime() throws IOException {
		try (DataInputStream stream = new DataInputStream(new BufferedInputStream(new GZIPInputStream(Files.newInputStream(
				_unsubscriptionFile))))) {
			int numEntries = stream.readInt();
			for (int i = 0; i < numEntries; i++) {
				long time = stream.readLong();
				int numValues = stream.readInt();
				for (int f = 0; f < numValues; f++) {
					long objId = stream.readLong();
					long atgId = stream.readLong();
					long aspId = stream.readLong();
					int sv = stream.readShort();
					_dataIdentTree.get(new IdDataIdentification(objId, atgId, aspId, sv)).setUnsubscriptionTime(time);
				}
			}
		}
	}

	/**
	 * Ermittelt die Header vom letzten Container (vom Container mit der größten ID)
	 *
	 * @param containerDirectory Referenz auf die gelockte Datenidentifikation und Datenart für den Zugriff auf Containerdaten
	 *                           (Datenidentifikation + ADK, unabhängig vom spezifischen Persistenzverzeichnis)
	 * @return ContainerHeaders
	 * @throws IndexException wenn der Indexzugriff fehlschlägt
	 */
	@Nullable
	public ContainerHeaders getLastContainerHeaders(final LockedContainerDirectory containerDirectory) throws IndexException {
		List<? extends PersistenceDirectory> persistenceDirectories = getPersistenceDirectories(containerDirectory.getSimVariant());

		for (PersistenceDirectory persistenceDirectory : Lists.reverse(persistenceDirectories)) {
			ContainerHeaders headers = persistenceDirectory.getLastContainerHeaders(containerDirectory);
			if (headers != null) {
				return headers;
			}
		}
		return null;
	}

	/**
	 * Ermittelt den zuletzt archivierten Datensatz einer Datenidentifikation und Datenart
	 *
	 * @param containerDirectory Referenz auf die gelockte Datenidentifikation und Datenart für den Zugriff auf Containerdaten
	 *                           (Datenidentifikation + ADK, unabhängig vom spezifischen Persistenzverzeichnis)
	 * @return ContainerHeaders
	 * @throws IndexException                 wenn der Indexzugriff fehlschlägt
	 * @throws SynchronizationFailedException Synchronisierung fehlgeschlagen
	 * @throws PersistenceException           Persistenzfehler
	 */
	@Nullable
	public ContainerDataResult getLastDataSet(final LockedContainerDirectory containerDirectory) throws IndexException, PersistenceException, SynchronizationFailedException {
		List<? extends PersistenceDirectory> persistenceDirectories = getPersistenceDirectories(containerDirectory.getSimVariant());

		for (PersistenceDirectory persistenceDirectory : Lists.reverse(persistenceDirectories)) {
			ContainerHeaders headers = persistenceDirectory.getLastContainerHeaders(containerDirectory);
			if (headers != null) {
				long contId = headers.getContainerHeaderParamAsLong(ContainerManagementInformation.CHP_CONT_ID);
				try (ContainerFileHandle containerFileHandle = persistenceDirectory.accessContainer(containerDirectory, contId)) {
					DataIterator iterator = containerFileHandle.iterator();
					ContainerDataResult result = new ContainerDataResult();
					while (!iterator.isEmpty()) {
						iterator.poll(result);
					}
					iterator.close();
					return result;
				}
			}
		}
		return null;
	}

	@Override
	public SyncKey<IdDataIdentification> lockIndex(final IdDataIdentification dataIdentification) throws SynchronizationFailedException {
		return _indexLockManager.acquireWriteKey(dataIdentification);
	}

	/**
	 * Gibt alle aktuell genutzten Locks zur Synchronisation auf die Datenidentifikationen zurück.
	 *
	 * @return Locks
	 */
	public SetMultimap<IdDataIdentification, SyncKey<IdDataIdentification>> getIndexLocks() {
		return _indexLockManager.getLocks();
	}

	/**
	 * Testmethode fürs Debugging, stellt sicher, dass aktuell keine Locks vom aktuellen Thread gehalten werden. Dieser Aufruf kann eingefügt werden,
	 * wenn eine Methode einen blockierenden Aufruf macht (z. B. auf Netzwerkantwort warten). Dies sollte aus Deadlock-Gefahr-Gründen nicht gemacht
	 * werden, während ein Index gelockt ist.
	 */
	public void assertNoLocks() {
		assert assertNoLocksInternal(Thread.currentThread());
	}

	@SuppressWarnings("SameReturnValue")
	private boolean assertNoLocksInternal(final Thread currentThread) {
		Collection<SyncKey<IdDataIdentification>> values = getIndexLocks().values();
		for (SyncKey<IdDataIdentification> value : values) {
			if (value.getThread() == currentThread) {
				throw new AssertionError("Thread " + currentThread + " darf keine Locks halten: " + value);
			}
		}
		return true;
	}

	/**
	 * Gibt das aktive Persistenzverzeichnis (in das gerade aktiviert wird) zurück.
	 *
	 * @param simVariant Simulationsvariante
	 * @return Persistenzverzeichnis, oder null, wenn gerade keines zum Beschreiben benutzt wird.
	 */
	@Nullable
	public ActivePersistenceDirectory getActivePersistenceDirectory(int simVariant) {
		if (simVariant == 0) {
			return getPersistenceDirectoryManager().getActivePersistenceDirectory();
		}
		return getPersistenceDirectoryManager().getSimulationPersistenceDirectory(simVariant);
	}

	/**
	 * Ermittelt relevante Persistenzverzeichnisse. Achtung: Die zurückgegebenen
	 * Verzeichnisse sind grob nach der angegebenen {@link SequenceSpecification} gefiltert. D. h. der Anfangs-Zustand
	 * und ein ggf. nachfolgender Datensatz fehlen in sehr ungünstigen Fällen eventuell, und zwar wenn die Grenze des
	 * Anfragebereichs genau auf einem Wechsel des Wochenverzeichnisses liegt.
	 * <p>
	 * Daher muss in diesen Fällen #validateIndexResult aufgerufen werden.
	 * </p>
	 *
	 * @param simVariant            Simulationsvariante (oder 0 für "normale" Daten)
	 * @param sequenceSpecification Anfragebereich
	 * @return Liste mit Persistenzverzeichnissen
	 */
	public List<? extends PersistenceDirectory> getPersistenceDirectories(int simVariant, SequenceSpecification sequenceSpecification) {
		return getPersistenceDirectoryManager().getPersistenceDirectories(simVariant, sequenceSpecification);
	}

	/**
	 * Gibt alle Persistenzverzeichnisse einer Simulationsvariante zurück.
	 *
	 * @param simVariant Simulationsvariante (oder 0 für "normale" Daten)
	 * @return alle Persistenzverzeichnisse, die zurückgegebene Liste ist immutable.
	 */
	public List<? extends PersistenceDirectory> getPersistenceDirectories(int simVariant) {
		return getPersistenceDirectoryManager().getPersistenceDirectories(simVariant, AllDataSpecification.Instance);
	}

	/**
	 * Führt eine Index-Abfrage über mehrere Persistenzverzeichnisse durch
	 *
	 * @param containerDirectory       Gelocktes Containerverzeichnis
	 * @param archiveTimeSpecification Archivzeitspezifikation
	 * @return Index-Ergebnis
	 * @throws IndexException Fehler beim Index-Zugriff
	 */
	public LocatedIndexResult<IndexValues> getIndexResult(LockedContainerDirectory containerDirectory,
	                                                      ArchiveTimeSpecification archiveTimeSpecification)
			throws IndexException {

		// Indexe auf den aktuellen Stand bringen
		ActivePersistenceDirectory persistenceDirectory = getActivePersistenceDirectory(containerDirectory.getSimVariant());
		if (persistenceDirectory != null) {
			persistenceDirectory.updateStandardIndexes(containerDirectory);
		}

		SequenceSpecification spec = createSequenceFromArchiveTimeSpecification(archiveTimeSpecification, containerDirectory, false);

		return getIndexResult(containerDirectory, spec);
	}

	/**
	 * Führt eine Index-Abfrage über mehrere Persistenzverzeichnisse durch
	 *
	 * @param containerDirectory    Gelocktes Containerverzeichnis
	 * @param sequenceSpecification Abfragebereich
	 * @return Index-Ergebnis
	 * @throws IndexException Fehler beim Index-Zugriff
	 */
	public LocatedIndexResult<IndexValues> getIndexResult(LockedContainerDirectory containerDirectory,
	                                                       SequenceSpecification sequenceSpecification)
			throws IndexException {

		if (sequenceSpecification instanceof DataIndexSequenceSpecification spec) {
			return getCompoundDataIndexIndex(containerDirectory, sequenceSpecification).getContainerIDByDataIndex(spec.minimumIndex(), spec.maximumIndex());
		} else if (sequenceSpecification instanceof ArchiveTimeSequenceSpecification spec) {
			return getCompoundArchiveTimeIndex(containerDirectory, sequenceSpecification).getContainerIDByArchiveTime(spec.minimumTime(), spec.maximumTime());
		} else if (sequenceSpecification instanceof AllDataSpecification) {
			return getCompoundDataIndexIndex(containerDirectory, sequenceSpecification).getContainerIDByDataIndex(0, Long.MAX_VALUE);
		}
		throw new AssertionError();
	}

	/**
	 * Konvertiert eine {@link ArchiveTimeSpecification} (aus einer Anfrage) in eine {@link SequenceSpecification}.
	 * Es werden nur absolute {@link ArchiveTimeSpecification}-Objekte unterstützt.
	 *
	 * @param ats                Archivzeitspezifikation
	 * @param containerDirectory Containerverzeichnis
	 * @return SequenceSpecification
	 * @throws IndexException Fehler beim Index-Zugriff
	 */
	public SequenceSpecification createSequenceFromArchiveTimeSpecification(ArchiveTimeSpecification ats, LockedContainerDirectory containerDirectory) throws IndexException {
		return createSequenceFromArchiveTimeSpecification(ats, containerDirectory, true);
	}

	/**
	 * Konvertiert eine {@link ArchiveTimeSpecification} (aus einer Anfrage) in eine {@link SequenceSpecification}.
	 * Es werden nur absolute {@link ArchiveTimeSpecification}-Objekte unterstützt.
	 *
	 * @param ats                Archivzeitspezifikation
	 * @param containerDirectory Containerverzeichnis
	 * @param needsIndexUpdate Müssen die Indexe vorher anhand der gecachten {@link OpenContainerData} auf den aktuellen Stand gebracht werden?   
	 * @return SequenceSpecification
	 * @throws IndexException Fehler beim Index-Zugriff
	 */
	private SequenceSpecification createSequenceFromArchiveTimeSpecification(ArchiveTimeSpecification ats, LockedContainerDirectory containerDirectory, boolean needsIndexUpdate) throws IndexException {

		if (ats.isStartRelative())
			throw new UnsupportedOperationException("Ein relativer Intervallstart wird bei dieser Aktion nicht unterstützt.");

		if (ats.getTimingType().equals(TimingType.ARCHIVE_TIME)) {
			return new ArchiveTimeSequenceSpecification(ats.getIntervalStart(), ats.getIntervalEnd());
		}
		if (ats.getTimingType().equals(TimingType.DATA_INDEX)) {
			return new DataIndexSequenceSpecification(ats.getIntervalStart(), ats.getIntervalEnd());
		}
		if (ats.getTimingType().equals(TimingType.DATA_TIME)) {

			if (needsIndexUpdate) {
				// Indexe auf den aktuellen Stand bringen
				ActivePersistenceDirectory persistenceDirectory = getActivePersistenceDirectory(containerDirectory.getSimVariant());
				if (persistenceDirectory != null) {
					persistenceDirectory.updateStandardIndexes(containerDirectory);
				}
			}

			CompoundDataTimeIndex index = getCompoundDataTimeIndex(containerDirectory, ats.getIntervalStart(), ats.getIntervalEnd());
			IndexResult<IndexValues> contIDs = index.getContainerIDByDataTime(ats.getIntervalStart(), ats.getIntervalEnd());
			if (contIDs.isEmpty()) {
				// Falls keine Container ermittelt werden konnte, den ersten Container zurückliefern, damit der erste Datensatz übertragen werden kann.
				// Die Anforderungen sind nicht ganz klar, aber der Test de.bsvrz.ars.ars.TestArchiveRelative_Request.testEndeVorKleinstem()
				// deutet darauf hin, dass bei einer Archivanfrage mit DATA-TIME der erste Datensatz zurückgeliefert werden soll, wenn der Anfragebereich
				// vor dem ersten Datensatz liegt.
				return new DataIndexSequenceSpecification(0, 0);
			}
			// Daten, die zwischen Anfang und Ende liegen, aber wegen Zeitrücksprüngen nicht mehr im Anfragebereich liegen, trotzdem mitberücksichtigen,
			// daher hier nochmal anhand des DatenIndex-Indexes die Container bestimmen.
			return new DataIndexSequenceSpecification(contIDs.getMin(IndexValues.DataIndexMin), contIDs.getMax(IndexValues.DataIndexMax));
		}
		throw new AssertionError();
	}


	private CompoundArchiveTimeIndex getCompoundArchiveTimeIndex(LockedContainerDirectory containerDirectory, SequenceSpecification sequenceSpecification) throws IndexException {

		int simVariant = containerDirectory.getSimVariant();

		// Alle Verzeichnisse ermitteln, die wahrscheinlich benötigt werden
		List<? extends PersistenceDirectory> reducedDirectories = getPersistenceDirectories(simVariant, sequenceSpecification);
		List<? extends PersistenceDirectory> allDirectories = getPersistenceDirectories(simVariant);

		// Es mit den (potenziell) reduzierten Verzeichnissen versuchen
		CompoundArchiveTimeIndex result = createArchiveTimeIndex(containerDirectory, reducedDirectories, sequenceSpecification);
		if (reducedDirectories.equals(allDirectories) || result.isComplete(sequenceSpecification)) {
			// Es wurden entweder schon alle Verzeichnisse berücksichtigt, oder die oben bestimmten Verzeichnisse enthalten alle Daten
			// → Also einfach zurückgeben
			return result;
		}

		// Notfalls müssen alle Indexdateien in allen Wochenverzeichnissen analysiert werden
		return createArchiveTimeIndex(containerDirectory, allDirectories, sequenceSpecification);
	}

	@NotNull
	private static CompoundArchiveTimeIndex createArchiveTimeIndex(LockedContainerDirectory containerDirectory, Collection<? extends PersistenceDirectory> directories, SequenceSpecification sequenceSpecification) throws IndexException {

		long minArchiveTime = Long.MIN_VALUE;
		long maxArchiveTime = Long.MAX_VALUE;
		if (sequenceSpecification instanceof ArchiveTimeSequenceSpecification archiveTimeSequenceSpecification) {
			minArchiveTime = archiveTimeSequenceSpecification.minimumTime();
			maxArchiveTime = archiveTimeSequenceSpecification.maximumTime();
		}


		IndexAggregator<ArchiveTimeIndex> aggregator = new IndexAggregator<>(ValidDataRange::minArchiveTime, ValidDataRange::maxArchiveTime, IndexTree::getArchiveTimeIndex);

		aggregator.aggregate(directories, containerDirectory, minArchiveTime, maxArchiveTime);

		return new CompoundArchiveTimeIndex(aggregator);
	}

	private CompoundDataIndexIndex getCompoundDataIndexIndex(LockedContainerDirectory containerDirectory, SequenceSpecification sequenceSpecification) throws IndexException {

		int simVariant = containerDirectory.getSimVariant();

		// Alle Verzeichnisse ermitteln, die wahrscheinlich benötigt werden
		List<? extends PersistenceDirectory> reducedDirectories = getPersistenceDirectories(simVariant, sequenceSpecification);
		List<? extends PersistenceDirectory> allDirectories = getPersistenceDirectories(simVariant);

		// Es mit den (potenziell) reduzierten Verzeichnissen versuchen
		CompoundDataIndexIndex result = createDataIndexIndex(containerDirectory, reducedDirectories, sequenceSpecification);
		if (reducedDirectories.equals(allDirectories) || result.isComplete(sequenceSpecification)) {
			// Hinweis: Die erste Bedingung ist aktuell immer true, da aktuell Filterung der Wochenverzeichnisse nach Datenindex implementiert ist.
			// Es wurden entweder schon alle Verzeichnisse berücksichtigt, oder die oben bestimmten Verzeichnisse enthalten alle Daten
			// → Also einfach zurückgeben
			return result;
		}

		// Notfalls müssen alle Indexdateien in allen Wochenverzeichnissen analysiert werden
		return createDataIndexIndex(containerDirectory, allDirectories, sequenceSpecification);
	}

	@NotNull
	private static CompoundDataIndexIndex createDataIndexIndex(LockedContainerDirectory containerDirectory, Collection<? extends PersistenceDirectory> directories, SequenceSpecification sequenceSpecification) throws IndexException {

		long minDataIndex = Long.MIN_VALUE;
		long maxDataIndex = Long.MAX_VALUE;
		if (sequenceSpecification instanceof DataIndexSequenceSpecification dataIndexSequenceSpecification) {
			minDataIndex = dataIndexSequenceSpecification.minimumIndex();
			maxDataIndex = dataIndexSequenceSpecification.maximumIndex();
		}


		IndexAggregator<DataIndexIndex> aggregator = new IndexAggregator<>(ValidDataRange::minDataIndex, ValidDataRange::maxDataIndex, IndexTree::getDataIndexIndex);

		aggregator.aggregate(directories, containerDirectory, minDataIndex, maxDataIndex);

		return new CompoundDataIndexIndex(aggregator);
	}

	private CompoundDataTimeIndex getCompoundDataTimeIndex(LockedContainerDirectory containerDirectory, long minDataTime, long maxDataTime) throws IndexException {
		Collection<? extends PersistenceDirectory> directories = getPersistenceDirectories(containerDirectory.getSimVariant());

		IndexAggregator<DataTimeIndex> aggregator = new IndexAggregator<>(ValidDataRange::minDataTime, ValidDataRange::maxDataTime, IndexTree::getDataTimeIndex);

		aggregator.aggregate(directories, containerDirectory, minDataTime, maxDataTime);
		
		return new CompoundDataTimeIndex(aggregator);
	}

	public Path getBackupConfigurationDirectory() {
		return backupDirectory;
	}

	/**
	 * Löscht das Persistenzverzeichnis von einer Simulation komplett vom Datenträger. Es dürfen keine Verzeichnisse
	 * übergeben werden, die nicht von Simulationen können, dann wird ein Fehler geworfen.
	 *
	 * @param directory Persistenzverzeichnis der Simulation.
	 * @throws PersistenceException Fehler beim Löschen
	 */
	public void deletePersistenceDirectory(ActivePersistenceDirectory directory) throws PersistenceException {
		if (directory.getSimulationVariant() == 0) {
			throw new IllegalArgumentException();
		}

		try {
			getPersistenceDirectoryManager().deleteSimulationDirectory(directory);
			deletePath(directory.getBasePath());
		} catch (IOException e) {
			throw new PersistenceException("Simulationsdatenverzeichnis '" + directory + "'konnte nicht gelöscht werden.", e);
		}
	}

	/**
	 * Diese Methode wird aufgerufen, um die aktuelle Archivzeit zu setzen. Diese Methode wird aufgerufen,
	 * bevor ein Datensatz archiviert wird, damit die Persistenzschicht die erforderlichen Arbeiten durchführen kann,
	 * also z. B. ein neues Wochenverzeichnis anzulegen.
	 *
	 * @param archTime Aktuelle Archivzeit
	 * @throws PersistenceException Fehler beim Erstellen eines neuen Persistenzverzeichnisses
	 */
	public void updateArchTime(long archTime) throws PersistenceException {
		try {
			getPersistenceDirectoryManager().updatePersistenceDirectories(archTime);
		} catch (IOException | InterruptedException e) {
			throw new PersistenceException("Fehler beim Erstellen eines neuen Persistenzverzeichnisses", e);
		}
	}

	/**
	 * Diese Methode wird aufgerufen, um die aktuelle Archivzeit zu setzen und gibt gleichzeitig das zugehörige aktive
	 * Persistenzverzeichnis zurück. Diese Methode wird aufgerufen, bevor ein Datensatz archiviert wird,
	 * damit die Persistenzschicht die erforderlichen Arbeiten durchführen kann,
	 * also z. B. ein neues Wochenverzeichnis anzulegen.
	 * <p>
	 * Im Gegensatz zu {@link #getActivePersistenceDirectory(int)} wird nie {@code null} zurückgegeben.
	 * </p>
	 *
	 * @param archTime   Aktuelle Archivzeit
	 * @param simVariant Simulationsvariante
	 * @return Persistenzverzeichnis für die angegebenen Dateninformationen.
	 * @throws PersistenceException Fehler beim Erstellen eines neuen Persistenzverzeichnisses
	 */
	@NotNull
	public ActivePersistenceDirectory updateAndGetActivePersistenceDirectory(long archTime, int simVariant) throws PersistenceException {
		updateArchTime(archTime);
		ActivePersistenceDirectory directory = getActivePersistenceDirectory(simVariant);
		if (directory == null) {
			if (simVariant == 0) {
				throw new PersistenceException("Das aktuelle Persistenzverzeichnis konnte nicht gelesen werden.");
			} else {
				throw new AssertionError("Für Simulation " + simVariant + " wurde noch kein Verzeichnis angelegt.");
			}
		}
		return directory;
	}

	/**
	 * Gibt das Wurzelverzeichnis der Persistenz zurück.
	 * @return Wurzelverzeichnis
	 */
	public Path getRootPath() {
		return getPersistenceDirectoryManager().getRootPath();
	}

	/**
	 * Gibt die Verwaltung der einzelnen Unterverzeichnisse zurück.
	 * @return Verwaltungsschicht für Verzeichnisse
	 */
	public PersistenceDirectoryManager getPersistenceDirectoryManager() {
		return _persistenceDirectoryManager;
	}

	@Override
	public DataIdentTree getDataIdentTree() {
		return _dataIdentTree;
	}

	@Override
	public int getIndexCacheMaxSize() {
		return  archivMgr.getIndexCacheMaxSize();
	}

	/**
	 * Die Methode formatObj gibt eine Objekt-ID als lesbaren String aus (z. B. ermitteln der Pid falls möglich)
	 *
	 * @param objId von Typ long
	 * @return String
	 */
	@Override
	public String formatObj(final long objId) {
		if (archivMgr instanceof ArchiveManager am) {
			ClientDavInterface davCon = am.getDavCon();
			if (davCon != null) {
				DataModel dataModel = davCon.getDataModel();
				if (dataModel != null) {
					SystemObject obj = dataModel.getObject(objId);
					if (obj != null) {
						return obj.getPidOrId();
					}
				}
			}
		}
		return String.valueOf(objId);
	}

	@Override
	public boolean isRangeUnavailable(long fromArchiveTime, long toArchiveTime) {
		return getPersistenceDirectoryManager().isRangeUnavailable(fromArchiveTime, toArchiveTime);
	}

	@Override
	public boolean wasTerminated() {
		return archivMgr.wasTerminated();
	}

	@Override
	public int getNumCloseIndexThreads() {
		return archivMgr.getNumCloseIndexThreads();
	}

	@Override
	@Nullable
	public RuntimeControl getRuntimeControl() {
		return archivMgr.getRuntimeControl();
	}

	@Override
	public void suspendTaskIfNecessary(Task task) throws InterruptedException {
		archivMgr.suspendTaskIfNecessary(task);
	}

	@Override
	public long countDataInQueues() {
		return archivMgr.countDataInQueues();
	}

	@Override
	public long estimateQueueMemoryUsage() {
		return archivMgr.estimateQueueMemoryUsage();
	}

	@Override
	public int getNumCheckPersistenceThreads() {
		return archivMgr.getNumCheckPersistenceThreads();
	}

	@Override
	public PersistenceManager getPersistenceManager() {
		return this;
	}

	/**
	 * Klasse die Statistiken wie Queue-Größe oder Speicherverbrauch enthält,
	 * die z. B. laufend über Debug oder Telnet ausgegeben werden können
	 */
	public class Statistics {

		private final long numNodes = _dataIdentTree.size();
		private final long indexLocks = _indexLockManager.getLocks().size();
		private final long numOpenContainerData;
		private final long numFlagFiles;
		private final long queueMemorySize;
		private final long queueMemory;

		{
			ActivePersistenceDirectory persistenceDirectory = _persistenceDirectoryManager.getActivePersistenceDirectory();
			if (persistenceDirectory == null) {
				numOpenContainerData = 0;
				numFlagFiles = 0;
			} else {
				numOpenContainerData = persistenceDirectory.openContainerDataSize();
				numFlagFiles = persistenceDirectory.getDirtyDirectoriesSize();
			}

			queueMemorySize = archivMgr.countDataInQueues();
			queueMemory = archivMgr.estimateQueueMemoryUsage();
		}

		private final CacheManager.CacheMemoryUsage cachedMemory = CacheManager.getInstance().getCachedMemory();

		/**
		 * Erzeugt eine neue Statistics-Instanz.
		 */
		private Statistics() {
		}

		/**
		 * Gibt numNodes von diesem Statistics-Objekt zurück.
		 *
		 * @return numNodes (Typ long)
		 */
		public long getNumNodes() {
			return numNodes;
		}

		/**
		 * Gibt nodeMemory von diesem Statistics-Objekt zurück.
		 *
		 * @return nodeMemory (Typ long)
		 */
		public long getNodeMemory() {
			return 108 * numNodes; // überprüft mit visualvm
		}

		/**
		 * Gibt indexLocks von diesem Statistics-Objekt zurück.
		 *
		 * @return indexLocks (Typ long)
		 */
		public long getIndexLocks() {
			return indexLocks;
		}

		/**
		 * Gibt lockMemory von diesem Statistics-Objekt zurück.
		 *
		 * @return lockMemory (Typ long)
		 */
		public long getLockMemory() {
			return getIndexLocks() * 90; // 90 = ungefährer Speicherverbrauch eines SyncKey mit Referenz usw.
		}

		/**
		 * Gibt numOpenContainerData von diesem Statistics-Objekt zurück.
		 *
		 * @return numOpenContainerData (Typ long)
		 */
		public long getNumOpenContainerData() {
			return numOpenContainerData;
		}

		/**
		 * Gibt den Speicherverbrauch der Verwaltungsdaten für offene Container und Indexe zurück
		 *
		 * @return openContainerMemory (Typ long)
		 */
		public long getActiveMemory() {
			return numOpenContainerData * 192 // Geschätzt mit visualvm 
					+ numFlagFiles * 108; // ungefährer Speicherverbrauch einer HashMap-Node mit enthaltenen Objekten
		}

		/**
		 * Gibt cachedMemory von diesem Statistics-Objekt zurück.
		 *
		 * @return cachedMemory
		 */
		public CacheManager.CacheMemoryUsage getCachedMemory() {
			return cachedMemory;
		}

		public long getQueueSize() {
			return queueMemorySize;
		}

		public long getQueueMemory() {
			return queueMemory;
		}

		@Override
		public String toString() {
			return "Statistics{" +
					"numNodes=" + numNodes +
					", indexLocks=" + indexLocks +
					", numOpenContainerData=" + numOpenContainerData +
					", numFlagFiles=" + numFlagFiles +
					", queueMemorySize=" + queueMemorySize +
					", queueMemory=" + queueMemory +
					", cachedMemory=" + cachedMemory +
					'}';
		}
	}

}
