/*
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */
package de.bsvrz.ars.ars.persistence.directories;

import com.google.common.util.concurrent.UncheckedExecutionException;
import de.bsvrz.ars.ars.mgmt.datatree.synchronization.SyncKey;
import de.bsvrz.ars.ars.mgmt.datatree.synchronization.SynchronizationFailedException;
import de.bsvrz.ars.ars.persistence.*;
import de.bsvrz.ars.ars.persistence.directories.cache.DataRange;
import de.bsvrz.ars.ars.persistence.directories.cache.SimpleDataRange;
import de.bsvrz.ars.ars.persistence.directories.cache.ValidDataRange;
import de.bsvrz.ars.ars.persistence.index.ArchiveTimeIndex;
import de.bsvrz.ars.ars.persistence.index.ContainerManagementIndex;
import de.bsvrz.ars.ars.persistence.index.IndexException;
import de.bsvrz.ars.ars.persistence.index.IndexValues;
import de.bsvrz.ars.ars.persistence.index.backend.management.BaseIndex;
import de.bsvrz.ars.ars.persistence.index.result.IndexResult;
import de.bsvrz.ars.ars.persistence.iter.DataIterator;
import de.bsvrz.ars.ars.persistence.layout.PersistenceDirectoryLayoutInstance;
import de.bsvrz.ars.ars.persistence.sequence.SequenceSpecification;
import de.bsvrz.ars.ars.persistence.walk.DataIdentificationDirWalk;
import de.bsvrz.ars.ars.persistence.walk.internal.StatusPrinter;
import de.bsvrz.dav.daf.main.archive.ArchiveDataKind;
import de.bsvrz.dav.daf.main.archive.ArchiveDataKindCombination;
import de.bsvrz.sys.funclib.debug.Debug;
import de.bsvrz.sys.funclib.kappich.annotations.NotNull;
import de.bsvrz.sys.funclib.kappich.annotations.Nullable;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.time.Duration;
import java.time.Instant;
import java.util.*;
import java.util.concurrent.*;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Supplier;

/**
 * Diese Klasse steht für ein "aktives" (noch nicht abgeschlossenes) Persistenzverzeichnis, in das laufend Online-Daten
 * archiviert werden. Daher bietet diese Klasse gegenüber der Basisklasse {@link PersistenceDirectory}, die für allgemeine
 * Persistenzverzeichnisse (insbesondere im Nur-Lesen-Modus) benutzt wird weitere Funktionalitäten.
 */
public final class ActivePersistenceDirectory extends PersistenceDirectory {

	/**
	 * Dummy-Objekt als Wert von {@link #openContainerData}, das angibt, dass es keinen offenen Container gibt
	 */
	private static final OpenContainerData.NoOpenContainer NO_OPEN_CONTAINER = OpenContainerData.NoOpenContainer.Instance;

	/**
	 * Map, die die typischen Index-Werte für den offenen Container zwischenspeichert, damit diese Daten
	 * nicht ständig in der Containerdatei und den Indexdateien vorgehalten werden müssen.
	 * <p>
	 * Als Key dient ein
	 * {@link IdContainerFileDir}-Eintrag, der ein Containerverzeichnis darstellt (je Containerverzeichnis gibt es
	 * maximal einen offenen Container).
	 * <p>
	 * Als Wert je Containerverzeichnis gibt es folgende Möglichkeiten:
	 * <ul>
	 *     <li>Der Entry ist nicht enthalten: In dem Fall gibt es keine Aussage und die Containerdatei muss ggf. manuell gelesen werden.</li>
	 *     <li>Die spezielle Instanz NO_OPEN_CONTAINER: In dem Fall weiß die Software sicher, dass es keinen offenen Container gibt, braucht dann also auch nicht erneut nachgucken.</li>
	 *     <li>Ein sonstiges {@link StandardOpenContainerData}: Daten des offenen Containers</li>
	 * </ul>
	 */
	private final Map<IdContainerFileDir, OpenContainerData> openContainerData = new ConcurrentHashMap<>();

	/**
	 * Klasse, die bei der Erzeugung von neuen Containern hilft, also insbesondere neue IDs vergibt. In laufenden System
	 * ist dies der {@link PersistenceManager}.
	 */
	private final ContainerCreator containerCreator;

	/**
	 * Eine Datei dieses Namens wird pro Verzeichnis einer Datenidentifikation/Datensatzart angelegt, wenn ein neuer Datensatz archiviert wurde und
	 * der Index möglicherweise (wg. Caching) noch nicht auf die Platte durchgeschrieben wurde. Beim Herunterfahren des Systems werden alle Indexe
	 * durchgeschrieben und die Flag-Dateien gelöscht. Existiert diese Datei beim Systemstart noch, ist dies ein Hinweis auf eine unkorrekte
	 * Beendigung des Archivsystems. Für jede DId, für die die Datei existiert, werden die Indexe verworfen bzw. neu aufgebaut.
	 */
	public static final String REBUILD_INDEX_FILE_FLAG_NAME = "_rebuildIndex.flag";

	/**
	 * Menge von Verzeichnissen, in denen die Indexe im Vergleich zum Stand auf der Festplatte modifiziert wurden,
	 * also nach einem Absturz ggf. wiederhergestellt werden müssen.
	 * <p>
	 * Dies wird eigentlich über Flag-Dateien ({@link #REBUILD_INDEX_FILE_FLAG_NAME}) geregelt. Diese Menge ist dazu da,
	 * sich zu merken, wo bereits Dateien angelegt wurden, um nicht ständig überprüfen zu müssen,
	 * ob so eine Flag-Datei schon existiert.
	 */
	private final Set<IdContainerFileDir> dirtyDirectories = new HashSet<>();

	/**
	 * Erstellt ein neues ActivePersistenceDirectory.
	 *
	 * @param containerCreator Interface mit den benötigten Funktionen zum Erstellen von neuen Containern
	 * @param layoutInstance   Konkrete Instanz des zu benutzenden Verzeichnis-Layouts
	 */
	public ActivePersistenceDirectory(@NotNull ContainerCreator containerCreator, @NotNull PersistenceDirectoryLayoutInstance layoutInstance) {
		super(containerCreator, layoutInstance);
		this.containerCreator = containerCreator;
	}

	@Override
	protected void handleUnclosedContainer(ContainerManagementIndex managementIndex, LockedContainerDirectory containerDirectory, long contId, BaseIndex<IndexValues> index) throws IndexException {
		// Offener Container steht nicht im Index
		StandardOpenContainerData openContainerData = getOpenContainerData(containerDirectory, managementIndex);
		if (openContainerData == null || openContainerData.getContainerId() != contId) {
			throw new IndexException("Inkonsistente Indexe, offener Container ist nicht eindeutig definiert", managementIndex.getFile());
		}

		index.setInsertValue(IndexValues.DataIndexMin, openContainerData.getMinDataIdx());
		index.setInsertValue(IndexValues.DataIndexMax, openContainerData.getMaxDataIdx());
		index.setInsertValue(IndexValues.DataTimeMin, openContainerData.getMinDataTime());
		index.setInsertValue(IndexValues.DataTimeMax, openContainerData.getMaxDataTime());
		index.setInsertValue(IndexValues.ArchiveTimeMin, openContainerData.getMinArcTime());
		index.setInsertValue(IndexValues.ArchiveTimeMax, openContainerData.getMaxArcTime());
	}

	/**
	 * Die Methode updateContainerIndex aktualisiert den Verwaltungsdatenindex
	 *
	 * @param containerFile Containerdaten, die in den Index eingefügt werden sollen (ersetzt ggf. vorhandene Einträge mit gleicher ContainerID)
	 * @param index         Verwaltungsdatenindex
	 * @throws PersistenceException wenn ein Fehler auftritt
	 */
	protected static void updateContainerIndex(final ContainerManagementData containerFile, final ContainerManagementIndex index)
			throws PersistenceException {
		for (ContainerManagementInformation information : ContainerManagementInformation.values()) {
			if (information.isNumeric()) {
				long paramAsLong = containerFile.getContainerHeaderParamAsLong(information);
				index.setInsertValue(information, paramAsLong);
			} else {
				String paramAsString = containerFile.getContainerHeaderParamAsString(information);
				index.setInsertValue(information, paramAsString);
			}
		}

		try {
			index.insertOrReplace();
		} catch (IndexException e) {
			throw new PersistenceException("Fehler beim Aktualisieren der Header-Informationen in der Indexdatei", e);
		}
	}

	/**
	 * Greift auf den offenen Container zu. Die Methode darf nicht für geschlossene Container verwendet werden, da dann die Synchronisation nicht
	 * sichergestellt ist.
	 *
	 * @param containerDirectory Verzeichnis des Containers
	 * @param openContID         Id des offenen Containers.
	 * @return Containerhandle
	 * @throws PersistenceException Lesefehler
	 */
	public ContainerFileHandle accessOpenContainer(final ContainerDirectory containerDirectory,
	                                               final long openContID) throws PersistenceException {
		return new ContainerFileHandle(containerDirectory, openContID, false, getLayoutInstance());
	}

	/**
	 * Die Methode updateContainerIndex aktualisiert den Verwaltungsdatenindex
	 *
	 * @param managementData     Containerdaten, die in den Index eingefügt werden sollen (ersetzt ggf. vorhandene Einträge mit gleicher ContainerID)
	 * @param containerDirectory Referenz auf die gelockte Datenidentifikation und Datenart für den Zugriff auf Containerdaten
	 * @throws PersistenceException wenn ein Fehler auftritt
	 * @throws IndexException       wenn ein Problem beim Zugriff auf den Index auftritt
	 */
	public void updateContainerIndex(final ContainerManagementData managementData, LockedContainerDirectory containerDirectory)
			throws PersistenceException, IndexException {

		Optional<? extends ContainerManagementIndex> containerManagementIndex = getIndexTree().getContainerManagementIndex(containerDirectory);
		if (containerManagementIndex.isPresent()) {
			ActivePersistenceDirectory.updateContainerIndex(managementData, containerManagementIndex.get());
		}
	}

	/**
	 * Die Methode getOpenContainerData die zwischengespeicherten Daten für den offenen Container zurück.
	 * Dieser Overload ohne {@link ContainerManagementIndex}-Parameter öffnet den Index, falls erforderlich.
	 *
	 * @param containerFileDir Containerverzeichnis
	 * @return StandardOpenContainerData oder null, falls kein offener Container existiert
	 * @throws IndexException                 wenn ein Problem beim Zugriff auf den Index auftritt
	 * @throws SynchronizationFailedException Synchronisierung fehlgeschlagen
	 */
	@Nullable
	public StandardOpenContainerData getOpenContainerData(final ContainerDirectory containerFileDir)
			throws IndexException, SynchronizationFailedException {

		IdContainerFileDir key = IdContainerFileDir.of(containerFileDir);
		OpenContainerData data = getLoadedContainerData(key);

		if (data == null) {
			// StandardOpenContainerData muss aus Container berechnet werden
			// Der folgende Code sieht unnötig kompliziert aus, es ist aber wichtig, dass wir die initOpenContainerData()-Methode nicht innerhalb
			// von computeIfAbsent ausführen und dass umgekehrt computeIfAbsent nicht ausgeführt wird, während auf den Index synchronisiert wird.
			// Sonst kann sich die ConcurrentHashMap-interne Synchronisierung mit dem lockIndex verhaken.

			OpenContainerData tmp;

			try (LockedContainerDirectory containerDirectory = containerCreator.lockIndex(containerFileDir)) {
				Optional<? extends ContainerManagementIndex> containerManagementIndex = getIndexTree().getContainerManagementIndex(containerDirectory);
				if (containerManagementIndex.isEmpty()) {
					openContainerData.put(key, NO_OPEN_CONTAINER);
					return null;
				}
				tmp = initOpenContainerData(containerDirectory, containerManagementIndex.get());
			}

			data = openContainerData.computeIfAbsent(key, (dir) -> tmp);
		}

		if (data instanceof StandardOpenContainerData containerData) {
			return containerData;
		}
		return null;
	}


	/**
	 * Die Methode getOpenContainerData ermittelt die Daten vom offenen Container.
	 * Dies ist eine optimierte Variante von {@link #getOpenContainerData(ContainerDirectory)}
	 * wenn der {@link ContainerManagementIndex} bereits geladen wurde.
	 *
	 * @param containerDirectory       Datenidentifikation/Verzeichnis des Containers
	 * @param containerManagementIndex von Typ ContainerManagementIndex
	 * @return StandardOpenContainerData
	 */
	@Nullable
	public StandardOpenContainerData getOpenContainerData(final LockedContainerDirectory containerDirectory,
	                                                      final ContainerManagementIndex containerManagementIndex) {
		// Der folgende Code sieht unnötig kompliziert aus, es ist aber wichtig, dass wir die initOpenContainerData()-Methode nicht innerhalb
		// von computeIfAbsent ausführen und dass umgekehrt computeIfAbsent nicht ausgeführt wird, während auf den Index synchronisiert wird.
		// Sonst kann sich die ConcurrentHashMap-interne Synchronisierung mit dem Lock hier verhaken.

		IdContainerFileDir key = IdContainerFileDir.of(containerDirectory);
		OpenContainerData data = getLoadedContainerData(key);

		if (data == null) {
			// StandardOpenContainerData muss aus Container berechnet werden
			OpenContainerData tmp = initOpenContainerData(containerDirectory, containerManagementIndex);

			data = openContainerData.computeIfAbsent(key, (dir) -> tmp);
		}

		if (data instanceof StandardOpenContainerData containerData) {
			return containerData;
		}
		return null;
	}

	/**
	 * Entspricht etwa {@link #getOpenContainerData()}, gibt die Daten aber nur zurück, wenn sie bereits bekannt sind,
	 * d.h. es wird nicht im letzten Container nachgesehen.
	 *
	 * @param containerDirectory Containerverzeichnis
	 * @return null, wenn kein {@link OpenContainerData} bekannt ist, {@link #NO_OPEN_CONTAINER}, wenn bekannt ist,
	 * dass es keinen offenen Container gibt, sonst die Daten des offenen Containers
	 */
	@Nullable
	public OpenContainerData getLoadedContainerData(ContainerDirectory containerDirectory) {
		return openContainerData.get(IdContainerFileDir.of(containerDirectory));
	}

	/**
	 * Die Methode initOpenContainerData initialisiert die Daten eines geöffneten Containers. Da die Header des offenen Containers nicht aussagekräftig sind, muss im offenen Container im Regelfall über die Datensätze iteriert werden.
	 * <p>
	 *
	 * @param containerDirectory Referenz auf die gelockte Datenidentifikation und Datenart für den Zugriff auf Containerdaten
	 *                           Containerdatenverzeichnis
	 * @param headerIndex        Verwaltungsdatenindex, um zu bestimmen welcher Container der offene ist (oder ob es überhaupt einen offenen gibt)
	 * @return StandardOpenContainerData
	 */
	protected OpenContainerData initOpenContainerData(LockedContainerDirectory containerDirectory, final ContainerManagementIndex headerIndex) {

		try {
			if (headerIndex.numEntries() == 0) {
				// Diese DID wurde überhaupt noch nie archiviert, daher letzten Eintrag nicht weiterverwenden (welchen auch?)
				return NO_OPEN_CONTAINER;
			}
			long lastContId = headerIndex.getLast(ContainerManagementInformation.CHP_CONT_ID);
			boolean isOpen = (int) headerIndex.getLast(ContainerManagementInformation.CHP_ANZ_DS) == ContainerHdr.CONT_UNCLOSED;

			if (!isOpen) {
				// Container abgeschlossen, letzten Eintrag nicht weiterverwenden
				return NO_OPEN_CONTAINER;
			}

			return initOpenContainerDataFromFile(containerDirectory, lastContId);
		} catch (Exception e) {
			throw new UncheckedExecutionException(e);
		}
	}

	private OpenContainerData initOpenContainerDataFromFile(LockedContainerDirectory dir, final long containerId) throws SynchronizationFailedException, PersistenceException {

		if (!dir.lock().isValid()) {
			throw new IllegalArgumentException("Der Parameter \"indexLock\" ist ungültig:" + dir.lock());
		}

		// es existiert ein offener Container
		try (ContainerFileHandle handle = accessContainer(dir, containerId)) {
			// Anzahl im Container nachzählen, dabei lastNoSrc-Flag mitnehmen
			int anz = 0;
			long minArchiveTime = Long.MAX_VALUE;
			long maxArchiveTime = -1;
			long minDataTime = Long.MAX_VALUE;
			long maxDataTime = -1;
			long minDataIndex = Long.MAX_VALUE;
			long maxDataIndex = -1;
			boolean lastDataIsNoSrc = false;
			try (DataIterator iter = handle.iterator()) {
				ContainerDataResult result = new ContainerDataResult();
				while (!iter.isEmpty()) {
					anz++;
					iter.poll(result);

					minArchiveTime = Math.min(minArchiveTime, result.getArchiveTime());
					maxArchiveTime = Math.max(maxArchiveTime, result.getArchiveTime());
					minDataTime = Math.min(minDataTime, result.getDataTime());
					maxDataTime = Math.max(maxDataTime, result.getDataTime());
					minDataIndex = Math.min(minDataIndex, result.getDataIndex());
					maxDataIndex = Math.max(maxDataIndex, result.getDataIndex());

					lastDataIsNoSrc = result.isNoSource();
				}
			} catch (PersistenceException ex) {
				debug.fine("Offener Container kann nicht gelesen werden", ex);
				// Wenn der Container hier nicht vollständig lesbar ist, kann das ignoriert werden
				// der Container (und der zugehörige Index-Eintrag) wird später in #checkContainerFile korrigiert.
			}

			if (anz == 0) {
				return NO_OPEN_CONTAINER;
			}

			if (dir.archiveDataKind() == ArchiveDataKind.ONLINE) {
				containerCreator.getDataIdentTree().get(dir.dataIdentification()).setLastOAWasNoSource(lastDataIsNoSrc);
			}

			// Container nicht abgeschlossen, letzten Eintrag weiterverwenden:
			return new StandardOpenContainerData(anz, minArchiveTime, maxArchiveTime, minDataTime, maxDataTime, minDataIndex, maxDataIndex,
					containerId);
		}
	}

	/**
	 * Gibt eine Map zurück, die je Datenidentifikation die 4 möglichen {@link StandardOpenContainerData} (je ADK) enthält.
	 * <p>
	 * Es werden nur Datenidentifikationen betrachtet, wo es offene Container gibt.
	 *
	 * @return Map
	 */
	Map<IdDataIdentification, StandardOpenContainerData[]> getOpenContainerData() {
		// OpenContainerData nach Datenidentifikation ummodeln, damit jede Datenidentifikation zusammen geschlossen wird
		// Da ArchiveDataKind kein Enum ist, wird hier statt einer EnumMap ein Array genutzt, die Indizes entsprechen `ArchiveDataKind.getCode() - 1`

		final Map<IdDataIdentification, StandardOpenContainerData[]> tmpMap = new HashMap<>();
		for (Map.Entry<IdContainerFileDir, OpenContainerData> entry : openContainerData.entrySet()) {
			IdDataIdentification dataIdentification = entry.getKey().dataIdentification();
			ArchiveDataKind archiveDataKind = entry.getKey().archiveDataKind();
			OpenContainerData openContainerData = entry.getValue();

			if (openContainerData instanceof StandardOpenContainerData containerData) {
				StandardOpenContainerData[] data =
						tmpMap.computeIfAbsent(dataIdentification, it -> new StandardOpenContainerData[4]);
				data[archiveDataKind.getCode() - 1] = containerData;			
			}
		}
		return tmpMap;
	}


	/**
	 * Setzt die Daten eines offenen Containers.
	 *
	 * @param containerDirectory Datenidentifikation/Verzeichnis des Containers
	 * @param data               Neue Daten
	 */
	public void setOpenContainerData(final LockedContainerDirectory containerDirectory, StandardOpenContainerData data) {
		openContainerData.put(IdContainerFileDir.of(containerDirectory), data);
	}

	/**
	 * Setzt, dass es aktuell für eine Datenidentifikation keinen offenen Container mehr gibt.
	 *
	 * @param containerDirectory Datenidentifikation/Verzeichnis des Containers
	 */
	public void removeOpenContainerData(final ContainerDirectory containerDirectory) {
		openContainerData.put(IdContainerFileDir.of(containerDirectory), NO_OPEN_CONTAINER);
	}

	/**
	 * Löscht die Daten eines offenen Containers komplett, also auch die Information, dass es keinen offenen Container gibt.
	 * Hierdurch wird der Speicher freigegeben.
	 *
	 * @param lock Synchroniserungszugriffsschlüssel
	 * @param adk  Archivdatenart
	 */
	public void deleteOpenContainerData(final SyncKey<IdDataIdentification> lock, final ArchiveDataKind adk) {
		IdDataIdentification did = lock.getElement();
		openContainerData.remove(did.resolve(adk));
	}

	/**
	 * Gibt die Anzahl der Einträge in {@link #openContainerData} zurück.
	 *
	 * @return die Anzahl der Einträge in {@link #openContainerData}
	 */
	public long openContainerDataSize() {
		return openContainerData.size();
	}

	/**
	 * Die Methode getRebuildIdxFile ermittelt den Speicherort für die _rebuildIndex.flag
	 *
	 * @param directory Datenidentifikation/Verzeichnis der Flag-Datei
	 * @return Path
	 */
	@NotNull
	public Path getRebuildIdxFile(final ContainerDirectory directory) {
		return getPath(directory).resolve(
				REBUILD_INDEX_FILE_FLAG_NAME);
	}

	/**
	 * Legt im Verzeichnis der Datenidentifikation ein FLAG-Datei an.
	 *
	 * @param directory Datenidentifikation/Verzeichnis der Flag-Datei
	 * @throws PersistenceException Lesefehler im Persistenzverzeichnis
	 */
	public void createRebuildIxdFlagFile(final ContainerDirectory directory) throws PersistenceException {
		Path idxFile = getRebuildIdxFile(directory);
		try {
			if (!Files.exists(idxFile)) {
				Files.createFile(idxFile);
			} else {
				debug.fine(idxFile + ": Datei existiert bereits.");
			}
		} catch (IOException e) {
			throw new PersistenceException("Fehler beim Anlegen der '" + idxFile + "'", e);
		}
	}

	/**
	 * Loescht die FLAG-Datei.
	 *
	 * @param didPath Containerverzeichnis
	 */
	void removeRebuildIxdFlagFile(Path didPath) {
		Path file = didPath.resolve(REBUILD_INDEX_FILE_FLAG_NAME);
		try {
			Files.deleteIfExists(file);
		} catch (IOException e) {
			debug.warning("Datei konnte nicht gelöscht werden: " + file, e);
		}
	}

	/**
	 * Erzeugt die Kennzeichen-Datei, ob die Indexe verändert wurden, um nach einem Absturz den Wiederanlauf zu beschleunigen.
	 *
	 * @param directory Datenidentifikation/Verzeichnis, das markiert werden soll
	 * @throws PersistenceException Lesefehler im Persistenzverzeichnis
	 */
	public void markAsDirty(ContainerDirectory directory) throws PersistenceException {
		synchronized (dirtyDirectories) {
			if (!dirtyDirectories.add(IdContainerFileDir.of(directory))) {
				createRebuildIxdFlagFile(directory);
			}
		}
	}

	/**
	 * Entfernt die Kennzeichen-Datei, ob die Indexe verändert wurden, um nach einem Absturz den Wiederanlauf zu beschleunigen.
	 *
	 * @param directory Datenidentifikation/Verzeichnis, mit Flag-Datei
	 */
	public void markAsClean(ContainerDirectory directory) {
		synchronized (dirtyDirectories) {
			if (dirtyDirectories.remove(IdContainerFileDir.of(directory))) {
				removeRebuildIxdFlagFile(getPath(directory));
			}
		}
	}

	/**
	 * Schließt die Indexe der Datensatzart des DataIndentNode.
	 *
	 * @param data Daten
	 * @throws IndexException Lesefehler der Indexe (z. B. korrupt)
	 */
	void closeIndex(final LockedContainerDirectory containerDirectory, @Nullable StandardOpenContainerData data) throws IndexException {
		if (data != null) {
			updateStandardIndexes(containerDirectory, data);
		}
		getIndexTree().closeIndexes(containerDirectory);
		markAsClean(containerDirectory);
	}

	/**
	 * Schreibt den aktuellen Stand der Indexe auf die Festplatte
	 *
	 * @param containerDirectory Datenidentifikation/Verzeichnis, in dem die Indexe geflusht werden sollen
	 */
	public void flushIndexes(LockedContainerDirectory containerDirectory) {
		getIndexTree().flushIndexes(containerDirectory);
	}

	/**
	 * Die Methode maxATime gibt die maximale Archivzeit einer Datenidentifikation zurück
	 *
	 * @param containerDirectory Referenz auf die gelockte Datenidentifikation und Datenart für den Zugriff auf Containerdaten
	 *                           Containerverzeichnis
	 * @return long
	 * @throws IndexException                 wenn ein Problem beim Zugriff auf den Index auftritt
	 * @throws SynchronizationFailedException Synchronisierung auf Indexe fehlgeschlagen
	 */
	public long maxATime(final ContainerDirectory containerDirectory) throws IndexException, SynchronizationFailedException {
		OpenContainerData openContainerData = getOpenContainerData(containerDirectory);
		if (openContainerData instanceof StandardOpenContainerData containerData) {
			return containerData.getMaxArcTime();
		} else {
			try (LockedContainerDirectory lockedContainerDirectory = containerCreator.lockIndex(containerDirectory)) {
				Optional<? extends ArchiveTimeIndex> archiveTimeIndex = getIndexTree().getArchiveTimeIndex(lockedContainerDirectory);
				if (archiveTimeIndex.isEmpty()) {
					return Long.MIN_VALUE;
				}
				return archiveTimeIndex.get().getLast(IndexValues.ArchiveTimeMax);
			}
		}
	}

	/**
	 * Die Methode getOpenContID gibt die ID des offenen Containers für eine Datenidentifikation zurück
	 *
	 * @param containerDirectory Referenz auf die gelockte Datenidentifikation und Datenart für den Zugriff auf Containerdaten
	 *                           Containerverzeichnis
	 * @return long oder -1 falls es keinen offenen Container gibt
	 * @throws IndexException                 wenn ein Problem beim Zugriff auf den Index auftritt
	 * @throws SynchronizationFailedException Synchronisierung auf Indexe fehlgeschlagen
	 */
	public long getOpenContID(final ContainerDirectory containerDirectory) throws IndexException, SynchronizationFailedException {
		OpenContainerData openContainerData = getOpenContainerData(containerDirectory);
		if (openContainerData instanceof StandardOpenContainerData containerData) {
			return containerData.getContainerId();
		} else {
			return -1;
		}
	}

	/**
	 * Die Methode getIndexResult führt eine Index-Abfrage durch (für Archivanfragen) und aktualisiert dabei die Daten vom offenen Container im Index
	 *
	 * @param sequenceSpecification von Typ ArchiveTimeSpecification
	 * @return IndexResult&lt;IndexValues&gt;
	 * @throws IndexException wenn ein Problem beim Zugriff auf den Index auftritt
	 */
	@Override
	public IndexResult<IndexValues> getIndexResult(final LockedContainerDirectory containerDirectory, final SequenceSpecification sequenceSpecification) throws IndexException, SynchronizationFailedException {

		StandardOpenContainerData openContainerData = getOpenContainerData(containerDirectory);
		if (openContainerData != null) {
			// Indexe auf den aktuellen Stand bringen, damit die Anfrage aktuelle Daten zurückliefert 
			updateStandardIndexes(containerDirectory, openContainerData);
		}

		return super.getIndexResult(containerDirectory, sequenceSpecification);
	}

	@Override
	public long estimate() {
		return Math.max(super.estimate(), openContainerData.size());
	}

	@Override
	public DataRange getDataRange(LockedContainerDirectory directory) throws IndexException {
		DataRange dataRange = computeDataRange(directory);
		try {
			StandardOpenContainerData openContainerData = getOpenContainerData(directory);
			if (openContainerData != null) {
				if (dataRange instanceof ValidDataRange validDataRange) {
					// Noch nicht geschriebene neuere Daten berücksichtigen und "verheiraten"
					return new SimpleDataRange(
							Math.min(validDataRange.minArchiveTime(), openContainerData.getMinArcTime()),
							Math.max(validDataRange.maxArchiveTime(), openContainerData.getMaxArcTime()),
							Math.min(validDataRange.minDataTime(), openContainerData.getMinDataTime()),
							Math.max(validDataRange.maxDataTime(), openContainerData.getMaxDataTime()),
							Math.min(validDataRange.minDataIndex(), openContainerData.getMinDataIdx()),
							Math.max(validDataRange.maxDataIndex(), openContainerData.getMaxDataIdx())
					);
				} else {
					// Nur OpenContainerData berücksichtigen
					return new SimpleDataRange(
							openContainerData.getMinArcTime(),
							openContainerData.getMaxArcTime(),
							openContainerData.getMinDataTime(),
							openContainerData.getMaxDataTime(),
							openContainerData.getMinDataIdx(),
							openContainerData.getMaxDataIdx()
					);
				}
			}
		} catch (SynchronizationFailedException e) {
			// Wir sind bereits synchronisiert!
			throw new AssertionError(e);
		}
		return dataRange;
	}

	/**
	 * Die Methode addIndexEntries fügt einem Index Werte hinzu.
	 *
	 * @param containerDirectory Datenidentifikation/Verzeichnis, in dem die Indexwerte hinzugefügt werden sollen
	 * @param dataIdx            von Typ long
	 * @param arsTime            von Typ long
	 * @param dataTime           von Typ long
	 * @param openContID         von Typ long
	 * @throws IndexException wenn ein Problem beim Zugriff auf den Index auftritt
	 */
	public void addIndexEntries(final LockedContainerDirectory containerDirectory, final long dataIdx, final long arsTime, final long dataTime, final long openContID) throws IndexException {
		final StandardOpenContainerData openContainerData = new StandardOpenContainerData(1, arsTime, arsTime, dataTime, dataTime, dataIdx, dataIdx, openContID);
		setOpenContainerData(containerDirectory, openContainerData);
		updateStandardIndexes(containerDirectory, openContainerData);
	}


	/**
	 * Die Methode updateStandardIndexes aktualisiert die Indexe mit dem zuletzt gesicherten {@link OpenContainerData}.
	 *
	 * @param containerDirectory Datenidentifikation/Verzeichnis, in dem die Indexwerte aktualisiert werden sollen
	 * @throws IndexException wenn ein Problem beim Zugriff auf den Index auftritt
	 */
	public void updateStandardIndexes(final LockedContainerDirectory containerDirectory)
			throws IndexException {
		OpenContainerData data = getLoadedContainerData(IdContainerFileDir.of(containerDirectory));
		if (data instanceof StandardOpenContainerData containerData) {
			updateStandardIndexes(containerDirectory, containerData);
		}
	}

	/**
	 * Die Methode updateStandardIndexes fügt einem Index Werte hinzu oder aktualisiert die vorhandenen Werte.
	 *
	 * @param containerDirectory Datenidentifikation/Verzeichnis, in dem die Indexwerte aktualisiert werden sollen
	 * @param data               von Typ OpenContainerData
	 * @throws IndexException wenn ein Problem beim Zugriff auf den Index auftritt
	 */
	public void updateStandardIndexes(final LockedContainerDirectory containerDirectory, final StandardOpenContainerData data)
			throws IndexException {
		final List<BaseIndex<IndexValues>> indices = new ArrayList<>(3);
		Supplier<IndexException> errorHandler = () -> new IndexException("Indexe fehlen", getPath(containerDirectory));
		indices.add(getIndexTree().getArchiveTimeIndex(containerDirectory).orElseThrow(errorHandler));
		indices.add(getIndexTree().getDataTimeIndex(containerDirectory).orElseThrow(errorHandler));
		if (containerDirectory.archiveDataKind().isRequested()) {
			indices.add(getIndexTree().getDataIndexIndex(containerDirectory).orElseThrow(errorHandler));
		}
		for (BaseIndex<IndexValues> index : indices) {
			index.setInsertValue(IndexValues.ContainerId, data.getContainerId());
			index.setInsertValue(IndexValues.DataTimeMin, data.getMinDataTime());
			index.setInsertValue(IndexValues.DataTimeMax, data.getMaxDataTime());
			index.setInsertValue(IndexValues.ArchiveTimeMin, data.getMinArcTime());
			index.setInsertValue(IndexValues.ArchiveTimeMax, data.getMaxArcTime());
			index.setInsertValue(IndexValues.DataIndexMin, data.getMinDataIdx());
			index.setInsertValue(IndexValues.DataIndexMax, data.getMaxDataIdx());
			index.insertOrReplace(true); // Erstmal prüfen, ob in alle Indexe eingefügt werden kann
		}
		for (BaseIndex<IndexValues> index : indices) {
			index.insertOrReplace(); // Einfügen
		}
	}

	/**
	 * Die Methode speichert im {@link OpenContainerData} neue Maximal-Werte für den offenen Container (Maximal-Datenindex usw.)
	 *
	 * @param containerDirectory Containerverzeichnis
	 * @param dataIdx            Datenindex
	 * @param arsTime            Archivzeit
	 * @param dataTime           Datenzeit
	 * @throws IndexException                 wenn ein Problem beim Zugriff auf den Index auftritt
	 * @throws SynchronizationFailedException Synchronisierung auf Indexe fehlgeschlagen
	 */
	public void updateMaxValues(ContainerDirectory containerDirectory, final long dataIdx, final long arsTime,
	                            final long dataTime) throws IndexException, SynchronizationFailedException {
		StandardOpenContainerData openContainerData = getOpenContainerData(containerDirectory);
		if (openContainerData != null) {
			openContainerData.updateMax(dataIdx, arsTime, dataTime);
		}
	}

	/**
	 * Schließt/Flusht alle Indexdateien und wartet bis alle Indexe geschlossen wurden
	 *
	 * @param numThreads Anzahl Threads fürs parallele Schließen
	 * @throws InterruptedException Unterbrochen beim Warten auf die Threads
	 */
	public void closeIndexes(int numThreads) throws InterruptedException {

		Instant start = Instant.now();
		final AtomicLong run = new AtomicLong();
		try (ExecutorService threadPool = new ThreadPoolExecutor(numThreads, numThreads,
				0L, TimeUnit.MILLISECONDS,
				new ArrayBlockingQueue<>(100))) {

			AtomicLong indexed = new AtomicLong();
			AtomicBoolean indexFinished = new AtomicBoolean();
			// Indexdateien in mehreren Threads schließen
			final Map<IdDataIdentification, StandardOpenContainerData[]> tmpMap = getOpenContainerData();

			threadPool.submit(() -> {
				for (Map.Entry<IdDataIdentification, StandardOpenContainerData[]> entry : tmpMap.entrySet()) {
					StandardOpenContainerData[] value = entry.getValue();
					threadPool.submit(new IndexCloserWorker(run, entry.getKey(), value));
					indexed.getAndIncrement();
				}
				indexFinished.set(true);
			});

			// Abwarten, bis alle Threads fertig sind.
			threadPool.shutdown();
			while (!threadPool.awaitTermination(60, TimeUnit.SECONDS)) {
				if (indexFinished.get()) {
					debug.info("Indexe werden geschlossen: " + run.get() + " von " + indexed + " Datenidentifikationen\n" +
							StatusPrinter.estimateRuntime(
									Duration.between(start, Instant.now()),
									run.get(),
									indexed.get()
							));
				} else {
					debug.info("Indexe werden geschlossen: " + run.get() + " von mindestens " + indexed + " Datenidentifikationen\n" +
							StatusPrinter.estimateMinRuntime(
									Duration.between(start, Instant.now()),
									run.get(),
									indexed.get()
							));
				}
			}
		}
		debug.info("Alle Indexe in " + StatusPrinter.formatDuration(Duration.between(start, Instant.now())) + " geschlossen: " + run.get());
	}


	/**
	 * Schließt den offenen Container in einem Verzeichnis ab.
	 *
	 * @param containerDirectory Verzeichnis
	 * @throws IndexException                 Fehler beim Schreiben der Indexe
	 * @throws SynchronizationFailedException Fehler beim Synchronisieren auf die Indexe der Datenidentifikation
	 */
	public void closeOpenContainer(LockedContainerDirectory containerDirectory) throws IndexException, SynchronizationFailedException {

		StandardOpenContainerData openContainerData = getOpenContainerData(containerDirectory);

		if (openContainerData != null) {
			try (var containerFileHandle = accessOpenContainer(containerDirectory, openContainerData.getContainerId())) {
				ContainerFile cf = containerFileHandle.getContainerFile();
				cf.closeContainer(openContainerData);
				updateContainerIndex(containerFileHandle, new LockedContainerDirectory(containerDirectory.lock(), ((StandaloneContainerFileHandle) containerFileHandle).getLocation().archiveDataKind()));
				updateStandardIndexes(containerDirectory, openContainerData);
				removeOpenContainerData(containerDirectory);
			} catch (PersistenceException e) {
				debug.error("Fehler beim Abschliessen eines Containers: " + e.getMessage() + Debug.NEWLINE + this, e);
			}
		}
	}

	/**
	 * Schließt dieses Persistenz-Wochenverzeichnis dauerhaft ab, schließt also alle Container und Indexe.
	 *
	 * @throws InterruptedException Unterbrochen beim Warten auf Beendigung des Abschließens
	 * @throws PersistenceException Schreibfehler beim Container schließen
	 */
	public void closePermanently() throws InterruptedException, PersistenceException {
		DataIdentificationDirWalk walk = DataIdentificationDirWalk.allDirectories(this);
		walk.setStatusPrinter(null);
		int numThreads = this.containerCreator.getCloseThreadCount();
		walk.execute("Container abschließen", numThreads, (dataIdentificationDir, walk1) -> {
			try (SyncKey<IdDataIdentification> key = containerCreator.lockIndex(dataIdentificationDir.getDataIdentification())) {
				for (ArchiveDataKind archiveDataKind : ArchiveDataKindCombination.all()) {
					closeOpenContainer(new LockedContainerDirectory(key, archiveDataKind));
				}
			}
		});
		closeIndexes(numThreads);
	}

	public long getDirtyDirectoriesSize() {
		return this.dirtyDirectories.size();
	}

	class IndexCloserWorker implements Runnable {
		private final AtomicLong _run;
		private final IdDataIdentification _dataIdentification;

		/**
		 * Array mit OpenContainerData je Datenart. Der Index entspricht {@link ArchiveDataKind#getCode()} {@code - 1}
		 */
		private final StandardOpenContainerData[] _data;

		/**
		 * Erzeugt eine neue IndexCloserWorker-Instanz.
		 *
		 * @param run                von Typ AtomicLong
		 * @param dataIdentification von Typ IdContainerFileDir
		 * @param data               von Typ StandardOpenContainerData
		 */
		public IndexCloserWorker(final AtomicLong run, final IdDataIdentification dataIdentification, @NotNull final StandardOpenContainerData[] data) {
			_run = run;
			_dataIdentification = dataIdentification;
			_data = data;
		}

		/**
		 * Die Methode run schließt Indexe
		 */
		@Override
		public void run() {
			try {
				_run.incrementAndGet();
				for (int i = 0; i < _data.length; i++) {
					ArchiveDataKind adk = ArchiveDataKind.getInstance(i + 1);
					StandardOpenContainerData openContainerData = _data[i];
					if (openContainerData != null) {
						closeIndexHelper(openContainerData, adk);
					}
				}
			} catch (IndexException | SynchronizationFailedException e) {
				debug.error("Herunterfahren: Fehler beim Schliessen der Indexe: ", e);
			}
		}

		/**
		 * Die Methode closeIndexHelper schließt einen Index
		 *
		 * @param openContainerData von Typ AtomicInteger
		 * @param dataKind          von Typ ArchiveDataKind
		 * @throws IndexException wenn ein Problem beim Zugriff auf den Index auftritt
		 */
		private void closeIndexHelper(final StandardOpenContainerData openContainerData, final ArchiveDataKind dataKind) throws IndexException, SynchronizationFailedException {
			try (LockedContainerDirectory containerDirectory = containerCreator.lockIndex(new IdContainerFileDir(_dataIdentification, dataKind))) {
				closeIndex(containerDirectory, openContainerData);
			}
		}
	}
}
