/*
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */
package de.bsvrz.ars.ars.persistence.directories.mgmt.range;

import com.google.common.collect.DiscreteDomain;
import com.google.common.collect.Range;
import de.bsvrz.sys.funclib.kappich.annotations.NotNull;
import de.bsvrz.sys.funclib.kappich.annotations.Nullable;
import org.jetbrains.annotations.Contract;

import java.nio.file.Path;
import java.time.Instant;
import java.util.List;

/**
 * Basisklasse für einen diskreten Typ für Zeitbereiche (z. B. Wochen) für die Wochenverzeichnisse.
 * <p>
 * Diese Klasse ermittelt zu einem Zeitstempel den zugehörigen Zeitbereich und konvertiert diesen in eine Ordnerstruktur
 * bzw. erlaubt es umgekehrt, basierend auf einer gegebenen Ordnerstruktur das zugehörige Zeitintervall
 * (also z. B.die Woche)
 * zu ermitteln.
 *
 * @param <T> Typ der einzelnen Zeitintervalle, z. B. {@link Week}
 */
public abstract class TimeDomain<T extends TimeRange<T>> extends DiscreteDomain<T> {

	/**
	 * Ermittelt zu einem absoluten Zeitbereich (gegeben durch 2 {@link Instant}-Objekte)
	 * die Liste mit Intervallen, in diesem Bereich. Die zurückgegebene Liste ist unveränderlich.
	 *
	 * @param timeRange Zeitbereich
	 * @return Intervalle
	 */
	public List<T> getIntervals(@NotNull Range<Instant> timeRange) {
		T start = ofEpochMillis(timeRange.lowerEndpoint().toEpochMilli());
		T end = ofEpochMillis(timeRange.upperEndpoint().toEpochMilli());
		return new RangeList<>(start, end);
	}

	/**
	 * Gibt zu einem Zeitpunkt, das Intervall zurück, das diesen Zeitpunkt enthält.
	 *
	 * @param epochMillis Zeitpunkt in Epoch-Millis
	 * @return zu einem Zeitpunkt, das Intervall
	 */
	@Contract(pure = true)
	@NotNull
	public abstract T ofEpochMillis(long epochMillis);

	/**
	 * Gibt zu einer relativen Ordnerstruktur das zugehörige Zeitintervall zurück.
	 * <p>
	 * Beispielsweise könnte {@code ofPath(Path.of("2021", "2021-01-01")); } die Woche oder
	 * den Monat zurückgeben, der am 01. 01. 2021 anliegt.
	 * <p>
	 * Handelt es sich um keine bekannte Ordnerstruktur, kann {@code null} zurückgegeben werden.
	 *
	 * @param relativePath Pfad, der relativ sein muss
	 * @return zu einem Zeitpunkt, das Intervall
	 */
	@Contract(pure = true)
	@Nullable
	public abstract T ofPath(Path relativePath);

	/**
	 * Gibt zu einem gegebenen Zeitintervall den Pfad zurück, indem Daten dieses Zeitintervalls gespeichert werden sollen.
	 * Dieser Pfad muss zu jedem möglichen Zeitintervall eindeutig sein, sonst könnte die umgekehrte Operation
	 * {@link #ofPath(Path)} nicht funktionieren.
	 *
	 * @param timeRange Zeitbereich
	 * @return Pfad
	 */
	@Contract(pure = true)
	@NotNull
	public abstract Path getPath(T timeRange);

	/**
	 * Gibt die Maximaltiefe der Ordnerstruktur zurück, die diese {@link TimeDomain} verwendet. Wenn z. B. Daten in
	 * Jahresverzeichnissen und darunter in Wochenverzeichnissen gespeichert werden sollen, mull
	 * 2 zurückgegeben werden. Innerhalb dieser Tiefe sucht das Archivsystem nach gültigen Persistenzverzeichnissen.
	 *
	 * @return Maximaltiefe
	 */
	public abstract int getMaxDepth();
}
