/*
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */
package de.bsvrz.ars.ars.persistence.directories.mgmt.range;

import de.bsvrz.sys.funclib.debug.Debug;
import de.bsvrz.sys.funclib.kappich.annotations.NotNull;

import java.nio.file.Path;
import java.time.Instant;
import java.time.LocalDate;
import java.time.ZoneOffset;
import java.time.format.DateTimeParseException;
import java.time.temporal.ChronoUnit;

/**
 * Domain-Klasse für Wochen-Zeitbereiche
 */
public final class WeekDomain extends TimeDomain<Week> {

	private static final Debug _debug = Debug.getLogger();

	/**
	 * Konvertiert ein {@link LocalDate} in eine Woche
	 *
	 * @param date Datum
	 * @return Woche
	 */
	public Week ofDate(LocalDate date) {
		return new Week(Math.floorDiv(ChronoUnit.DAYS.between(Week.root, date), 7));
	}

	/**
	 * Konvertiert einen Epoch-Milli-zeitstempel in eine Woche
	 *
	 * @param epochMillis Zeitpunkt in Epoch-Millis
	 * @return Woche
	 */
	@NotNull
	public Week ofEpochMillis(long epochMillis) {
		return ofDate(Instant.ofEpochMilli(epochMillis).atOffset(ZoneOffset.UTC).toLocalDate());
	}

	@Override
	public Week ofPath(Path relativePath) {
		if (relativePath.isAbsolute()) {
			throw new IllegalArgumentException("Pfad ist absolut: " + relativePath);
		}
		if (relativePath.getNameCount() != 2) {
			_debug.fine("Ignoriere Verzeichnis: Benötige 2 Komponenten: " + relativePath);
			return null;
		}
		String year = relativePath.getName(0).toString();
		String weekStart = relativePath.getName(1).toString();
		try {
			LocalDate date = LocalDate.parse(weekStart);
			Week week = ofDate(date);
			if (week.getFirstDay().equals(date)) {
				if (!String.valueOf(date.getYear()).equals(year)) {
					_debug.fine("Ignoriere Verzeichnis: Falsches Jahr: " + relativePath);
					return null;
				}
				return week;
			}
		} catch (DateTimeParseException e) {
			_debug.fine("Ignoriere Verzeichnis: Kein gültiger Zeitbereich: " + weekStart, e);
			return null;
		}
		_debug.fine("Ignoriere Verzeichnis: Kein gültiger Zeitbereich: " + weekStart);
		return null;
	}

	@NotNull
	@Override
	public Path getPath(Week timeRange) {
		LocalDate firstDay = timeRange.getFirstDay();
		return Path.of(String.valueOf(firstDay.getYear()), firstDay.toString());
	}

	@Override
	public int getMaxDepth() {
		return 2; // Jahresverzeichnis und Woche
	}

	@NotNull
	@Override
	public Week next(Week value) {
		return value.next();
	}

	@NotNull
	@Override
	public Week previous(Week value) {
		return value.previous();
	}

	@Override
	public long distance(Week start, Week end) {
		return end.internalWeekNumber() - start.internalWeekNumber();
	}

	/**
	 * Gibt {@code true} zurück, wenn es sich um ein Jahresverzeichnis handelt.
	 *
	 * @param path Zu prüfendes Verzeichnis
	 * @return {@code true}, wenn es sich um ein Jahresverzeichnis handelt, sonst {@code false}
	 */
	public static boolean isYearDir(Path path) {
		try {
			int year = Integer.parseInt(path.getFileName().toString());
			return year > 1900 && year <= 9999;
		} catch (Exception e) {
			return false;
		}
	}
}
