/*
 * Copyright 2019-2020 by Kappich Systemberatung, Aachen
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.ars.ars.persistence.gap;

import com.google.common.base.Splitter;
import com.google.common.collect.Range;
import de.bsvrz.ars.ars.mgmt.tasks.DataIndexRangeSet;
import de.bsvrz.ars.ars.persistence.IdDataIdentification;
import de.bsvrz.ars.ars.persistence.PersistenceManager;
import de.bsvrz.ars.ars.persistence.gap.util.RangeMultimap;
import de.bsvrz.sys.funclib.debug.Debug;
import de.bsvrz.sys.funclib.losb.util.Util;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.IOException;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.NoSuchFileException;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.util.Iterator;
import java.util.Map;
import java.util.SortedSet;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * Datenlücken-Datei für das nachfordern, die in einem Datenidentifikations-Ordner (neben oa, on, etc.) abgespeichert wird.
 */
public class GapFileImpl implements GapFile{

	private static final String GAP_KEYWORD = "<not_in>";

	private static final String GAP_IDX_SEPARATOR = "-";

	private static final String GAP_ARCH_SEPARATOR = "#";
	private static final Splitter ARS_SPLITTER = Splitter.on(GAP_ARCH_SEPARATOR);

	private static final Pattern GAP_PAT = Pattern.compile("^(\\d+)" + GAP_IDX_SEPARATOR + "(\\d+)" + GAP_KEYWORD + "(\\S+)$");
	private static final Charset CHARSET = StandardCharsets.ISO_8859_1;

	private final Path _path;
	
	private static final Debug _debug = Debug.getLogger();

	/**
	 * Konstruktor mit Datei. Wenn der Pfad nicht bekannt ist, stattdessen die Factory-Methode {@link #getInstance(PersistenceManager, IdDataIdentification)}
	 * benutzen.
	 *
	 * @param path Dateipfad
	 */
	public GapFileImpl(final Path path) {
		_path = path;
	}


	@Override
	public void readGaps(final Map<? super RemoteArchive, DataIndexRangeSet> map) {
		Path gapFile = _path;
		try {
			try(BufferedReader reader = Files.newBufferedReader(gapFile, CHARSET)) {
				readGaps(map, reader);
			}
		}
		catch(NoSuchFileException ignored){
		}
		catch(IOException e) {
			_debug.warning("Lueckendatei '" + gapFile + " konnte nicht gelesen werden.", e);
		}
	}

	@Override
	public void writeGaps(final Map<? extends RemoteArchive, ? extends DataIndexRangeSet> map) {
		Path tmpGapFile = _path.resolveSibling(_path.getFileName() + ".tmp");
		try {
			Path parent = tmpGapFile.getParent();
			if(!Files.exists(parent)) {
				Files.createDirectories(parent);
			}
			try(BufferedWriter writer = Files.newBufferedWriter(tmpGapFile, CHARSET)) {
				writeGaps(map, writer);
			}
			Files.move(tmpGapFile, _path, StandardCopyOption.REPLACE_EXISTING);
		}
		catch(IOException e) {
			_debug.warning("Temporaere Lueckendatei '" + tmpGapFile + " konnte nicht angelegt werden: " + e.getMessage());
		}	
	}

	private void readGaps(final Map<? super RemoteArchive, DataIndexRangeSet> resultMap, final BufferedReader reader) throws IOException {
		while(true) {
			String line = reader.readLine();
			if(line == null) {
				break;
			}
			Matcher m = GAP_PAT.matcher(line);
			if(m.matches()) {
				long start = Long.parseLong(m.group(1));
				long end = Long.parseLong(m.group(2));
				Iterable<String> pids = ARS_SPLITTER.split(m.group(3));
				for(String pid : pids) {
					RemoteArchive object = RemoteArchive.open(pid);
					DataIndexRangeSet ranges = resultMap.computeIfAbsent(object, (key) -> new DataIndexRangeSet());
					ranges.addRange(Util.dIdxNoModBits(start) + 1, Util.dIdxNoModBits(end));
				}
			}
		}
	}

	private static void writeGaps(final Map<? extends RemoteArchive, ? extends DataIndexRangeSet> inputMap, final BufferedWriter writer) throws IOException {
		RangeMultimap<Long, RemoteArchive> gapFileContent = new RangeMultimap<>();
		for(Map.Entry<? extends RemoteArchive, ? extends DataIndexRangeSet> entry : inputMap.entrySet()) {
			for(Range<Long> range : entry.getValue()) {
				gapFileContent.put(range, entry.getKey());
			}
		}

		for (Map.Entry<Range<Long>, SortedSet<RemoteArchive>> entry : gapFileContent.entries()) {
			Range<Long> range = entry.getKey();
			SortedSet<RemoteArchive> objects = entry.getValue();
			writer.write(String.valueOf(Util.dIdxAppendZeroModBits(range.lowerEndpoint()) - 2));
			writer.write(GAP_IDX_SEPARATOR);
			writer.write(String.valueOf(Util.dIdxAppendZeroModBits(range.upperEndpoint())));
			writer.write(GAP_KEYWORD);
			for(Iterator<RemoteArchive> iterator = objects.iterator(); iterator.hasNext(); ) {
				final RemoteArchive object = iterator.next();
				writer.write(object.getPid());
				if(iterator.hasNext()) {
					writer.write(GAP_ARCH_SEPARATOR);
				}
			}
			writer.newLine();
		}
	}
}
