/*
 * Copyright 2019-2020 by Kappich Systemberatung, Aachen
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.ars.ars.persistence.gap.util;

import com.google.common.collect.*;
import de.bsvrz.sys.funclib.kappich.annotations.Nullable;

import java.util.*;

/**
 * Eine {@link RangeMap} als {@link com.google.common.collect.Multimap}.
 *
 * @author Kappich Systemberatung
 * @param <K> Typ, von dem die Range-Keys sind (z. B. Double)
 * @param <V> Typ von dem die enthaltenen Werte sind, z. B. String, muss Comparable sein
 */
public class RangeMultimap<K extends Comparable<? super K>, V extends Comparable<V>> {

	private static final Object EMPTY_SORTED_SET = Collections.unmodifiableSortedSet(new TreeSet<>());
	
	private final RangeMap<K, TreeSet<V>> _delegate = TreeRangeMap.create();

	/**
	 * Gibt alle Elemente zurück, die am angegebenen Punkt existieren
	 *
	 * @param key Punkt auf der Gerade, über die die Keys gehen
	 * @return Werte an diesem Punkt
	 */
	@Nullable
	public SortedSet<V> get(final K key) {
		TreeSet<V> s = _delegate.get(key);
		if (s == null) {
			//noinspection unchecked
			return (SortedSet<V>) EMPTY_SORTED_SET;
		}
		return Collections.unmodifiableSortedSet(s);
	}

	/**
	 * Gibt den Wertebereich zurück, siehe {@link RangeMap#span()}
	 * @return Span
	 */
	public Range<K> span() {
		return _delegate.span();
	}

	/**
	 * Fügt auf einem Bereich einen Wert ein
	 * @param range Bereich
	 * @param value Wert
	 */
	public void put(final Range<K> range, final V value) {
		addValueTo(value, range);
	}

	/**
	 * Leert die {@link RangeMultimap}
	 */
	public void clear() {
		_delegate.clear();
	}

	/**
	 * Interne Funktion: Einen Wert zu einem bereich hinzufügen
	 * @param value Wert
	 * @param span  Bereich
	 */
	private void addValueTo(final V value, final Range<K> span) {
		K lowerEndPoint = span.lowerEndpoint();
		K upperEndpoint = span.upperEndpoint();
		Map.Entry<Range<K>, TreeSet<V>> lowerEntry = _delegate.getEntry(lowerEndPoint);
		Map.Entry<Range<K>, TreeSet<V>> upperEntry = _delegate.getEntry(upperEndpoint);
		
		if(lowerEntry != null && !lowerEntry.getValue().contains(value) && !lowerEntry.getKey().lowerEndpoint().equals(lowerEndPoint)) {
			_delegate.put(lowerEntry.getKey().intersection(span), new TreeSet<>(lowerEntry.getValue()));
		}

		if (upperEntry != null && !upperEntry.getValue().contains(value) && !upperEntry.getKey().upperEndpoint().equals(upperEndpoint)) {
			_delegate.put(upperEntry.getKey().intersection(span), new TreeSet<>(upperEntry.getValue()));
		}

		TreeRangeSet<K> remainingSpans = TreeRangeSet.create();
		remainingSpans.add(span);
		for (Map.Entry<Range<K>, TreeSet<V>> entry : List.copyOf(_delegate.subRangeMap(span).asMapOfRanges().entrySet())) {
			TreeSet<V> set = new TreeSet<>(entry.getValue());
			set.add(value);
			_delegate.putCoalescing(entry.getKey(), set);
			remainingSpans.remove(entry.getKey());
		}
		for(Range<K> range : remainingSpans.asRanges()) {
			TreeSet<V> set = new TreeSet<>();
			set.add(value);
			_delegate.putCoalescing(range, set);
		}
	}

	/**
	 * Siehe {@link RangeMap#asMapOfRanges()}. Die in den values zurückgegebenen TreeSets sollten nicht modifiziert werden.
	 *
	 * @return Map von Bereichen zu Werten
	 */
	public Collection<Map.Entry<Range<K>, SortedSet<V>>> entries() {
		return Collections2.transform(_delegate.asMapOfRanges().entrySet(),
				it -> Map.entry(it.getKey(), Collections.unmodifiableSortedSet(it.getValue()))
		);
	}

	@Override
	public String toString() {
		return _delegate.toString();
	}

	@Override
	public boolean equals(Object o) {
		if (this == o) return true;
		if (o == null || getClass() != o.getClass()) return false;

		RangeMultimap<?, ?> that = (RangeMultimap<?, ?>) o;

		return _delegate.equals(that._delegate);
	}

	@Override
	public int hashCode() {
		return _delegate.hashCode();
	}
}
