/*
 *
 * Copyright 2017-2020 by Kappich Systemberatung, Aachen
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.ars.ars.persistence.index;

import de.bsvrz.ars.ars.mgmt.datatree.IndexImpl;
import de.bsvrz.ars.ars.persistence.ContainerManagementInformation;
import de.bsvrz.ars.ars.persistence.index.backend.management.AbstractIndex;
import de.bsvrz.ars.ars.persistence.index.backend.management.ColumnType;
import de.bsvrz.ars.ars.persistence.index.backend.management.IndexContentDescriptor;
import java.nio.file.Path;

/**
 * Index, der alle wesentlichen Verwaltungsdaten eines Containers enthält.
 *
 * @see ContainerManagementInformation
 *
 * @author Kappich Systemberatung
 */
public class ContainerManagementIndex extends AbstractIndex<ContainerManagementInformation> {

	/** Dateiname, unter dem alle Indices dieser Art gespeichert werden */
	public static final String IDX_FILENAME = IndexImpl.ManagementData.getFileName();

	/**
	 * Spaltenbeschreibung
	 */
	public static final IndexContentDescriptor<ContainerManagementInformation> INDEX_CONTENT_DESCRIPTOR;

	static {
		INDEX_CONTENT_DESCRIPTOR = new IndexContentDescriptor<>(ContainerManagementInformation.class);
		for(ContainerManagementInformation containerHeaderParam : ContainerManagementInformation.values()) {
			ColumnType type;
			int byteLength = containerHeaderParam.getByteLength();
			if(containerHeaderParam == ContainerManagementInformation.CHP_CONT_ID) {
				type = ColumnType.StrictlyIncreasing;
			}
			else if(containerHeaderParam.isNumeric()){
				type = ColumnType.Unordered;
			}
			else {
				type = ColumnType.String;
			}
			INDEX_CONTENT_DESCRIPTOR.addColumn(byteLength, containerHeaderParam, type);
		}
	}

	/**
	 * Erstellt eine neue Instanz, die den vorhandenen Index (falls vorhanden) öffnet.
	 * @param bufferSize Puffergröße in bytes
	 * @throws CorruptIndexException Indexfehler (korrupt, nicht lesbar, ...)
	 */
	public ContainerManagementIndex(final int bufferSize, final Path file) throws CorruptIndexException {
		super(INDEX_CONTENT_DESCRIPTOR, bufferSize, file);
	}

	/**
	 * Gibt die anzahl Bytes zurück, die ein Indexeintrag belegt.
	 * @return Anzahl bytes
	 */
	public static int entrySize() {
		return INDEX_CONTENT_DESCRIPTOR.getEntryLengthBytes();
	}
}
