/*
 *
 * Copyright 2005-2008 by beck et al. projects GmbH, Munich
 * Copyright 2009-2020 by Kappich Systemberatung, Aachen
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.ars.ars.persistence.index;

import de.bsvrz.ars.ars.mgmt.datatree.IndexImpl;
import de.bsvrz.ars.ars.persistence.index.backend.management.AbstractIndex;
import de.bsvrz.ars.ars.persistence.index.backend.management.ColumnType;
import de.bsvrz.ars.ars.persistence.index.backend.management.IndexContentDescriptor;

import java.nio.file.Path;

/**
 * Diese Klasse stellt einen Datenindex-Index dar. Sie verhält sich ähnlich dem {@link DataTimeIndexImpl}. Die Folge der Eintraege (min, max) ist nicht monoton,
 * weil es beim Nachfordern Rückspruenge geben kann. Stattdessen sind die Eintraege nur nach dem min-Wert sortiert. Ein Eintrag enthält Minimum, Maximum des
 * Datenindex und die ContainerID.
 *
 * @author beck et al. projects GmbH
 * @author Phil Schrettenbrunner
 * @version $Revision$ / $Date$ / ($Author$)
 */
public final class DataIndexIndexImpl extends AbstractIndex<IndexValues> implements DataIndexIndex {

	/** Dateiname, unter dem alle Indices dieser Art gespeichert werden */
	public static final String IDX_FILENAME = IndexImpl.DataIndex.getFileName();

	private static final IndexContentDescriptor<IndexValues> INDEX_CONTENT_DESCRIPTOR;
	static {
		INDEX_CONTENT_DESCRIPTOR = new IndexContentDescriptor<>(IndexValues.class);
		INDEX_CONTENT_DESCRIPTOR.addRange(8, IndexValues.DataIndexMin, ColumnType.Increasing, IndexValues.DataIndexMax, ColumnType.Unordered);
		INDEX_CONTENT_DESCRIPTOR.addColumn(5, IndexValues.ContainerId, ColumnType.Unique);
	}

	/**
	 * Erzeugt einen neuen DatenIndex.
	 * <p>
	 * Bei startmode==APPEND_REUSE_LAST wird der Ringpuffer mit den aktuellsten Daten aus einer vorhandenen Indexdatei gefüllt. Der Eintrag mit der größten
	 * ContainerId wird als currentEntry reaktiviert. Damit kann ein Container fortgesetzt und der reaktivierte Eintrag weiter aktualisiert werden.
	 * <p>
	 * Bei startmode==ERASE_OLD wird ein leerer Index erzeugt und die bisherige Indexdatei (sofern vorhanden) geloescht.
	 *
	 * @param bufferSize Anzahl der Einträge, die der die der Index im RAM halten soll
	 * @param file Datei
	 *
	 * @throws CorruptIndexException Bei Problemen mit dem Dateizugriff
	 */
	public DataIndexIndexImpl(int bufferSize, Path file) throws CorruptIndexException {
		super(INDEX_CONTENT_DESCRIPTOR, bufferSize, file);
	}

	/**
	 * Fügt dem Index einen neuen Eintrag hinzu. Dabei wird der derzeitig aktuelle Eintrag abgeschlossen. Führt dies dazu, dass keine weiteren freien Plätze im
	 * Cache vorhanden sind, werden die Änderungen auf Platte geschrieben. Wenn der Cache noch vollständig leer ist, werden die aktuellsten Daten aus der
	 * Index-Datei geladen.
	 *
	 * @param dIMin  DatenIndex Min-Wert
	 * @param dIMax  DatenIndex Max-Wert
	 * @param contId Container ID
	 * @throws IndexException Bei Problemem mit der Index-Datei
	 */
	public void add(long dIMin, long dIMax, long contId) throws IndexException {
		checkAdd(dIMin, dIMax, contId);
		setInsertValue(IndexValues.DataIndexMin, dIMin);
		setInsertValue(IndexValues.DataIndexMax, dIMax);
		setInsertValue(IndexValues.ContainerId, contId);
		insert();
	}

	private void checkAdd(final long dIMin, final long dIMax, final long contId) throws IndexException {
		if(dIMin < 0 || dIMax < 0 || contId <= 0 ) {
			throw new CorruptIndexException("DatenIndexIndex: neuer Eintrag ungültig: (" + "dIMin=" + dIMin + ", dIMax=" + dIMax + ", contID=" + contId + ")", getFile());
		}
	}

	/**
	 * Gibt die anzahl Bytes zurück, die ein Indexeintrag belegt.
	 * @return Anzahl bytes
	 */
	public static int entrySize() {
		return INDEX_CONTENT_DESCRIPTOR.getEntryLengthBytes();
	}
}
