/*
 *
 * Copyright 2017-2020 by Kappich Systemberatung, Aachen
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.ars.ars.persistence.index.backend.management;

import com.google.common.collect.ImmutableMap;
import de.bsvrz.ars.ars.persistence.index.CorruptIndexException;
import de.bsvrz.ars.ars.persistence.index.IndexException;
import de.bsvrz.ars.ars.persistence.index.backend.storage.HybridStorage;
import de.bsvrz.ars.ars.persistence.index.result.BinaryIndexResult;
import de.bsvrz.ars.ars.persistence.index.result.IndexResult;
import de.bsvrz.sys.funclib.kappich.annotations.Nullable;

import java.io.IOException;
import java.nio.file.Path;
import java.util.List;
import java.util.Map;

/**
 * Basisimplementierung für Indexdateien im Archivsystem. Eine Indexdatei kann man sich vorstellen, wie eine Datenbank- oder Excel-Tabelle,
 * die nach bestimmten Spalten sortiert abgelegt wird um effizient Anfragen durchführen zu können.
 *
 * @author Kappich Systemberatung
 * @param <E> Enum, das die möglichen Spalten im Index definiert, z. B. {@link de.bsvrz.ars.ars.persistence.index.IndexValues}
 */
@SuppressWarnings("SuspiciousMethodCalls")
public class AbstractIndex<E extends Enum<E>> implements BaseIndex<E> {

	/**
	 * Hier wird über die sortieren Spalten eine {@link IndexImplementation} anlegt, die eine Suche/Filterung nach Werten ermöglicht
	 */
	private final Map<IndexContentDescriptor.IndexColumn, IndexImplementation> _indexes;

	/**
	 * Rohdaten-Speicher
	 */
	private final HybridStorage _storage;

	/**
	 * Spaltenbeschreibung
	 */
	private final IndexContentDescriptor<E> _indexContentDescriptor;

	/**
	 * Wiederverwendetes tmp-Objekt
	 */
	private final byte[] _tmp;

	/**
	 * Primärschlüssel (bestimmt die Sortierreihenfolge) wird automatisch aus der ersten monotonen Spalte ermittelt.
	 */
	private final IndexImplementation _mainIndex;

	/**
	 * hier werten die einzufügenden Long-Werte bis zum Aufruf von {@link #insert()} oder {@link #insertOrReplace()} usw. zwischengespeichert.
	 */
	private final long[] _insertLongValues;

	/**
	 * hier werten die einzufügenden String-Werte bis zum Aufruf von {@link #insert()} oder {@link #insertOrReplace()} usw. zwischengespeichert.
	 */
	private final String[] _insertStringValues;
	private boolean _closed;

	/**
	 * Erstellt eine neue Index-Instanz
	 *
	 * @param indexContentDescriptor Der {@link IndexContentDescriptor} gibt an, welche Spalten/informationen der Index enthalten kann
	 * @param bufferSize             Anzahl Index-Einträge, die aus der Indexdatei maximal gleichzeitig im Speicher gehalten werden (als Cache)
	 * @param file                   Index-Datei, die zu öffnen bzw. anzulegen ist. Falls null, wird der Index nur temporär im Speicher gehalten.
	 * @throws CorruptIndexException Index konnte nciht geladen werden
	 */
	public AbstractIndex(IndexContentDescriptor<E> indexContentDescriptor, int bufferSize, @Nullable Path file) throws CorruptIndexException {
		_indexContentDescriptor = indexContentDescriptor;
		try {
			_storage = new HybridStorage(_indexContentDescriptor.getEntryLengthBytes(), bufferSize, file);
		}
		catch(IOException e) {
			throw new CorruptIndexException("Kann Index nicht öffnen", file, e);
		}
		ImmutableMap.Builder<IndexContentDescriptor.IndexColumn, IndexImplementation> indexMapBuilder = ImmutableMap.builder();
		for(IndexContentDescriptor.IndexColumn column : indexContentDescriptor.getColumns()) {
			switch (column.getType()) {
				case StrictlyIncreasing -> indexMapBuilder.put(column, new StrictlyIncreasingIndex(column));
				case Increasing -> indexMapBuilder.put(column, new IncreasingIndex(column));
			}
		}
		_tmp = new byte[_storage.entryByteSize()];
		_indexes = indexMapBuilder.build();
		if(_indexes.isEmpty()) {
			throw new IllegalArgumentException("Keine Indexspalte definiert");
		}
		_mainIndex = _indexes.values().iterator().next();
		_insertLongValues = new long[indexContentDescriptor.getColumns().size()];
		_insertStringValues = new String[indexContentDescriptor.getColumns().size()];
	}

	/**
	 * Ermittelt alle Index-Zeilen, die zu der angegebenen Anfrage passen. Das Ergebnis wird performant ermittelt, indem
	 * alle bekannten Einschränkungen ausgewertet werden. Wird nach Spalten eingeschränkt, die nicht im Index vorkommen,
	 * dann werden ggf. überflüssige Zeilen/Container usw. zurückgegeben. Hier muss dann der Anfrager entsprechend selbst filtern.
	 *
	 * @param query Anfrage. Zu jeder Spalte in dem Index kann ein von/bis-Bereich angegeben werden, zu dem Container geliefert werden sollen.
	 *			  Wie für Archivdaten üblich, wird als erster Container ggf. der Container zurückgegeben, der den letzten Datensatz vor dem Anfragezeitraum
	 *			  enthält.
	 * @return Ein IndexResult, der die Ergebnistabelle enthält
	 * @throws IndexException Lesefehler oder korrupter Index
	 */
	@Override
	public IndexResult<E> query(Map<E, LongRange> query) throws IndexException {
		if(_closed) throw new IllegalStateException();
		try {
			BinaryIndexResult<E> result = new BinaryIndexResult<>(_indexContentDescriptor);

			if(_storage.numEntries() == 0) return result;

			LongRange range = indexQuery(query);
			if(range == null) return result;
			long lowerEndpoint = range.lowerEndpoint();
			long upperEndpoint = range.upperEndpoint();

			byte[] data = null;
			for(long indexIdx = lowerEndpoint; indexIdx <= upperEndpoint; indexIdx++) {
				byte[] prevData = data;
				data = new byte[_indexContentDescriptor.getEntryLengthBytes()];
				_storage.getEntries(indexIdx, 1, data);
				if(isDirectMatch(data, query)) {
					if(prevData != null && containsIntervalStart(prevData, data, query)) {
						result.add(prevData);
					}
					result.add(data);
					data = null;
				}
			}
			if (result.isEmpty() && data != null) {
				result.add(data);
			}

			if(upperEndpoint < _storage.numEntries() - 1) {
				data = new byte[_indexContentDescriptor.getEntryLengthBytes()];
				_storage.getEntries(upperEndpoint + 1, 1, data);
				result.setNext(data);
			}
			return result;
		}
		catch(IOException e) {
			throw new IndexException("Fehler bei Indexabfrage (" + query + ")", getFile(), e);
		}
	}

	/**
	 * Ermittelt alle Index-Zeilen
	 *
	 * @return Ein IndexResult, der die Ergebnistabelle enthält
	 * @throws IndexException Lesefehler oder korrupter Index
	 */
	@Override
	public IndexResult<E> query() throws IndexException {
		if(_closed) throw new IllegalStateException();
		try {
			BinaryIndexResult<E> result = new BinaryIndexResult<>(_indexContentDescriptor);

			for(long indexIdx = 0; indexIdx < _storage.numEntries(); indexIdx++) {
				byte[] data = new byte[_indexContentDescriptor.getEntryLengthBytes()];
				_storage.getEntries(indexIdx, 1, data);
				result.add(data);
			}

			return result;
		}
		catch(IOException e) {
			throw new IndexException("Fehler bei Indexabfrage (Alle Elemente)", getFile(), e);
		}
	}

	/**
	 * Setzt einen Wert der Einfügezeile, die später mit {@link #insert()} eingefügt werden kann.
	 *
	 * @param column Spalte
	 * @param value  Wert
	 */
	@Override
	public final void setInsertValue(E column, long value) {
		IndexContentDescriptor.IndexColumn tmp = _indexContentDescriptor.getColumn(column);
		if(tmp == null) return;
		if(tmp.getType() == ColumnType.String) throw new IllegalArgumentException("Versuch, einen Long-Wert in eine String-Spalte einzufügen.");
		_insertLongValues[tmp.getColumnIndex()] = value;
	}

	/**
	 * Setzt einen Wert der Einfügezeile, die später mit {@link #insert()} eingefügt werden kann.
	 *
	 * @param column Spalte
	 * @param value  Wert
	 */
	@Override
	public final void setInsertValue(E column, String value) {
		IndexContentDescriptor.IndexColumn tmp = _indexContentDescriptor.getColumn(column);
		if(tmp == null) return;
		if(tmp.getType() != ColumnType.String) throw new IllegalArgumentException("Versuch, einen String-Wert in eine Long-Spalte einzufügen.");
		_insertStringValues[tmp.getColumnIndex()] = value;
	}

	/**
	 * Gibt den ersten (der Sortierung nach am kleinsten) Eintrag zurück (der am Dateianfang gespeichert ist)
	 *
	 * @return IndexResult mit einem oder 0 Einträgen (falls der Index leer ist)
	 * @throws IndexException Lesefehler
	 */
	@Override
	public IndexResult<E> first() throws IndexException {
		if(_closed) throw new IllegalStateException();
		BinaryIndexResult<E> result = new BinaryIndexResult<>(_indexContentDescriptor);

		if(_storage.numEntries() == 0) return result;

		byte[] tmp = new byte[_indexContentDescriptor.getEntryLengthBytes()];
		try {
			_storage.getFirst(tmp);
		}
		catch(IOException e) {
			throw new IndexException("Fehler beim Ermitteln des ersten Werts", getFile(), e);
		}
		result.add(tmp);
		return result;
	}

	private boolean isDirectMatch(final byte[] data, final Map<E, LongRange> query) {
		final List<IndexContentDescriptor.IndexColumn> columns = _indexContentDescriptor.getColumns();
		for(final IndexContentDescriptor.IndexColumn column : columns) {
			LongRange range = query.get(column.getData());
			if(range == null) continue;
			long value = column.readLong(data);
			if(!range.contains(value)) return false;
		}
		return true;
	}

	/**
	 * Gibt zurück, ob der in {@code prevData} gespeicherte Datensatz den Intervallanfang enthält
	 * <p>
	 * Der Sinn dieser Funktion ist, dass bei einer Anfrage wie query(dTMin->(37..-), dTMax(-..114)), also einer Anfrage von allen Werten
	 * mit Datenzeit (dT) von 37 bis 117, festgestellt werden kann, ob noch der vorherige Container berücksichtigt werden muss.
	 * </p>
	 * <p>
	 * Beispielsweise könnte es hier Container geben mit dT=20..40, mit dT=30..50 und mit dT=40..60 und nur der letzte von diesen ist echt in diesem
	 * Bereich, obwohl theoretisch alle Zeilen passende Daten enthalten können.
	 * </p>
	 *
	 * @param prevData voriger Datensatz
	 * @param data     nachfolgender Datensatz
	 * @param query    Anfrage
	 * @return true oder false
	 */
	private boolean containsIntervalStart(final byte[] prevData, final byte[] data, final Map<E, LongRange> query) {
		final List<IndexContentDescriptor.IndexColumn> columns = _indexContentDescriptor.getColumns();
		for(final IndexContentDescriptor.IndexColumn column : columns) {
			LongRange range = query.get(column.getData());
			if(range == null || !range.hasLowerBound()) continue;

			long lowerEndpoint = range.lowerEndpoint();
			long prevValue = column.readLong(prevData);
			IndexContentDescriptor.IndexColumn minColumn = column.getMinColumn();
			long startValue = minColumn.readLong(data);

			// lowerEndpoint = Anfrage-Anfang des Intervalls
			// prevValue = Maximum-Wert des vorherigen Datensatzes
			// startValue = Minimum-Wert des aktuellen Datensatzes

			if(lowerEndpoint > prevValue
					&& lowerEndpoint < startValue) {
				return true;
			}
		}
		return false;
	}

	/**
	 * Fügt die vorher mit {@link #setInsertValue(Enum, long)} gesetzten Werte in den Index ein.
	 *
	 * @throws IndexException Lesefehler der Indexe (z. B. korrupt)
	 */
	@Override
	public void insert() throws IndexException {
		insert(false);
	}

	/**
	 * Fügt die vorher mit {@link #setInsertValue(Enum, long)} gesetzten Werte in den Index ein oder überschreibt den bisher gespeicherten Wert
	 * mit dem gleichen {@link #_mainIndex Primärschlüssel}.
	 *
	 * @throws IndexException Lesefehler der Indexe (z. B. korrupt)
	 */
	@Override
	public void insertOrReplace() throws IndexException {
		insertOrReplace(false);
	}

	/**
	 * Fügt die vorher mit {@link #setInsertValue(Enum, long)} gesetzten Werte in den Index ein oder prüft, ob das funktionieren würde.
	 *
	 * @param onlyCheck falls true wird nur geprüft, ob die neuen Werte eingefügt werden können, ohne dass die Monotoniebedingungen verletzt werden würden.
	 * @throws IndexException Lesefehler der Indexe (z. B. korrupt)
	 */
	@Override
	public void insert(final boolean onlyCheck) throws IndexException {
		if(_closed) throw new IllegalStateException();
		try {
			long entryIndex = findExistingItem();
			if(entryIndex >= 0) {
				long insertValue = _insertLongValues[_mainIndex.getColumn().getColumnIndex()];
				throw new IndexException("Index enthält bereits einen Wert " + insertValue + " in der Spalte " + _mainIndex.getColumn().getData() + " in Zeilenindex " + entryIndex, getFile());
			}
			final long position = -(entryIndex + 1);
			checkCanInsert(position);
			if(onlyCheck) return;
			serializeData();
			_storage.insertEntries(position, 1, _tmp);
		}
		catch(IOException e) {
			throw new IndexException("Fehler beim Einfügen", getFile(), e);
		}
	}

	/**
	 * Fügt die vorher mit {@link #setInsertValue(Enum, long)} gesetzten Werte in den Index ein oder überschreibt den bisher gespeicherten Wert
	 * mit dem gleichen {@link #_mainIndex Primärschlüssel}. Oder prüft, ob das funktionieren würde.
	 *
	 * @param onlyCheck falls true wird nur geprüft, ob die neuen Werte eingefügt werden können, ohne dass die Monotoniebedingungen verletzt werden würden.
	 * @throws IndexException Lesefehler der Indexe (z. B. korrupt)
	 */
	@Override
	public void insertOrReplace(final boolean onlyCheck) throws IndexException {
		if(_closed) throw new IllegalStateException();
		try {
			long entryIndex = findExistingItem();
			if(entryIndex >= 0) {
				checkCanReplace(entryIndex);
				if(onlyCheck) return;
				serializeData();
				_storage.setEntries(entryIndex, 1, _tmp);
			}
			else {
				final long position = -(entryIndex + 1);
				checkCanInsert(position);
				if(onlyCheck) return;
				serializeData();
				_storage.insertEntries(position, 1, _tmp);
			}
		}
		catch(IOException e) {
			throw new IndexException("Fehler beim Aktualisieren eines Werts", getFile(), e);
		}
	}

	private long findExistingItem() throws IOException {
		long insertValue = _insertLongValues[_mainIndex.getColumn().getColumnIndex()];
		long startIndex = _mainIndex.findEntryIndex(insertValue, _tmp);
		if(startIndex < 0) return startIndex; // Nicht gefunden
		long endIndex = startIndex;
		if(_mainIndex.getColumn().getType() == ColumnType.Increasing) {
			// Erstes passendes item suchen
			while(startIndex > 0) {
				_storage.getEntries(startIndex-1, 1, _tmp);
				if(_mainIndex.getColumn().readLong(_tmp) != insertValue) {
					break;
				}
				startIndex--;
			}	
			// Letztes passendes item suchen
			while(endIndex < _storage.numEntries() - 1) {
				_storage.getEntries(endIndex + 1, 1, _tmp);
				if(_mainIndex.getColumn().readLong(_tmp) != insertValue) {
					break;
				}
				endIndex++;
			}
		}
		for(long i = startIndex; i <= endIndex; i++) {
			_storage.getEntries(i, 1, _tmp);
			for(IndexContentDescriptor.IndexColumn column : _indexContentDescriptor.getColumns()) {
				if(column.getType() == ColumnType.Unique || column.getType() == ColumnType.StrictlyIncreasing) {
					if(column.readLong(_tmp) == _insertLongValues[column.getColumnIndex()]) {
						return i;
					}
				}
			}
		}
		return -endIndex - 2; // hinter endIndex einfügen, also (-(endindex + 1) - 1)
	}

	/**
	 * Fügt die vorher mit {@link #setInsertValue(Enum, long)} gesetzten Werte hinten an den Index an.
	 *
	 * @throws IndexException Lesefehler der Indexe (z. B. korrupt)
	 */
	@Override
	public void append() throws IndexException {
		append(false);
	}

	/**
	 * Fügt die vorher mit {@link #setInsertValue(Enum, long)} gesetzten Werte hinten an den Index an.
	 *
	 * @param onlyCheck falls true wird nur geprüft, ob die neuen Werte eingefügt werden können, ohne dass die Monotoniebedingungen verletzt werden würden und der Index wird nicht verändert.
	 * @throws IndexException Lesefehler der Indexe (z. B. korrupt)
	 */
	@Override
	public void append(final boolean onlyCheck) throws IndexException {
		if(_closed) throw new IllegalStateException();
		try {
			checkCanInsert(numEntries());
			if(onlyCheck) return;
			serializeData();
			_storage.addLast(_tmp);
		}
		catch(IOException e) {
			throw new IndexException("Fehler beim Anhängen eines Werts", getFile(), e);
		}
	}

	/**
	 * Prüft ob die mit {@link #setInsertValue(Enum, long)} gesetzte Zeile an die angegebene Position eingefügt werden darf, ohne Monotonitätsbedingungen zu verletzen.
	 *
	 * @param insertPosition Einfügeposition
	 * @throws IOException           IO-Fehler
	 * @throws CorruptIndexException Index korrupt oder nicht lesbar
	 */
	private void checkCanInsert(final long insertPosition) throws IOException, CorruptIndexException {
		boolean atStart = insertPosition == 0;
		boolean atEnd = insertPosition == numEntries();
		int entryByteSize = _storage.entryByteSize();

		checkInsertValues();

		if(!atStart && !atEnd) {
			byte[] bytes = new byte[entryByteSize * 2];
			_storage.getEntries(insertPosition - 1, 2, bytes);
			checkIsSmaller(bytes);
			checkIsLarger(bytes, entryByteSize);
		}
		else if(atStart && !atEnd) {
			byte[] bytes = new byte[entryByteSize];
			_storage.getEntries(insertPosition, 1, bytes);
			checkIsLarger(bytes, 0);
		}
		else if(!atStart) {   //  && atEnd
			byte[] bytes = new byte[entryByteSize];
			_storage.getEntries(insertPosition - 1, 1, bytes);
			checkIsSmaller(bytes);
		}
	}

	/**
	 * Prüft ob die mit {@link #setInsertValue(Enum, long)} gesetzte Zeile die Zeile an der angegebenen Position ersetzen darf, ohne Monotonitätsbedingungen zu verletzen.
	 *
	 * @param replacePosition Zu ersetzende Zeile (Index)
	 * @throws IOException           IO-Fehler
	 * @throws CorruptIndexException Index korrupt oder nicht lesbar
	 */
	private void checkCanReplace(final long replacePosition) throws IOException, CorruptIndexException {
		boolean atStart = replacePosition == 0;
		boolean atEnd = replacePosition == numEntries() - 1;
		int entryByteSize = _storage.entryByteSize();

		checkInsertValues();

		if(!atStart && !atEnd) {
			byte[] bytes = new byte[entryByteSize * 3];
			_storage.getEntries(replacePosition - 1, 3, bytes);
			checkIsSmaller(bytes);
			checkIsLarger(bytes, 2 * entryByteSize);
		}
		else if(atStart && !atEnd) {
			byte[] bytes = new byte[entryByteSize];
			_storage.getEntries(replacePosition + 1, 1, bytes);
			checkIsLarger(bytes, 0);
		}
		else if(!atStart) {   //  && atEnd
			byte[] bytes = new byte[entryByteSize];
			_storage.getEntries(replacePosition - 1, 1, bytes);
			checkIsSmaller(bytes);
		}
	}

	private void checkInsertValues() throws CorruptIndexException {
		for(IndexContentDescriptor.IndexColumn column : _indexContentDescriptor.getColumns()) {
			IndexContentDescriptor.IndexColumn minColumn = column.getMinColumn();
			if(minColumn != column) {
				long minValue = _insertLongValues[minColumn.getColumnIndex()];
				long maxValue = _insertLongValues[column.getColumnIndex()];
				if(minValue > maxValue) {
					throw new CorruptIndexException("Neuer Wert für " + column.getData() + " nicht größer als der Wert für " + minColumn.getData() + ": [" + minValue + "..." + maxValue + "]", getFile());
				}
			}
		}
	}

	private void checkIsSmaller(final byte[] bytes) throws CorruptIndexException {
		for(IndexContentDescriptor.IndexColumn column : _indexContentDescriptor.getColumns()) {
			checkIsSmaller(column, bytes);
		}
	}

	private void checkIsLarger(final byte[] bytes, final int offset) throws CorruptIndexException {
		for(IndexContentDescriptor.IndexColumn column : _indexContentDescriptor.getColumns()) {
			checkIsLarger(column, bytes, offset);
		}
	}

	private void checkIsSmaller(final IndexContentDescriptor.IndexColumn column, final byte[] bytes) throws CorruptIndexException {
		switch (column.getType()) {
			case StrictlyIncreasing -> {
				final long prevValue = column.readLong(bytes, 0);
				final long insertValue = _insertLongValues[column.getColumnIndex()];
				if (insertValue <= prevValue) {
					throw new CorruptIndexException("Neuer Wert für " + column.getData() + " nicht streng monoton steigend: " + insertValue + " (vorheriger Wert: " + prevValue + ")", getFile());
				}
			}
			case Increasing -> {
				final long prevValue = column.readLong(bytes, 0);
				final long insertValue = _insertLongValues[column.getColumnIndex()];
				if (insertValue < prevValue) {
					throw new CorruptIndexException("Neuer Wert für " + column.getData() + " nicht monoton steigend: " + insertValue + " (vorheriger Wert: " + prevValue + ")", getFile());
				}
			}
			case Unique -> {
				final long prevValue = column.readLong(bytes, 0);
				final long insertValue = _insertLongValues[column.getColumnIndex()];
				if (insertValue == prevValue) {
					throw new CorruptIndexException("Neuer Wert für " + column.getData() + " nicht eindeutig: " + insertValue + " (vorheriger Wert: " + prevValue + ")", getFile());
				}
			}
		}
	}

	private void checkIsLarger(final IndexContentDescriptor.IndexColumn column, final byte[] bytes, final int offset) throws CorruptIndexException {
		switch (column.getType()) {
			case StrictlyIncreasing -> {
				final long nextValue = column.readLong(bytes, offset);
				final long insertValue = _insertLongValues[column.getColumnIndex()];
				if (insertValue >= nextValue) {
					throw new CorruptIndexException("Neuer Wert für " + column.getData() + " nicht streng monoton steigend: " + insertValue + " (nachfolgender Wert: " + nextValue + ")", getFile());
				}
			}
			case Increasing -> {
				final long nextValue = column.readLong(bytes, offset);
				final long insertValue = _insertLongValues[column.getColumnIndex()];
				if (insertValue > nextValue) {
					throw new CorruptIndexException("Neuer Wert für " + column.getData() + " nicht monoton steigend: " + insertValue + " (nachfolgender Wert: " + nextValue + ")", getFile());
				}
			}
			case Unique -> {
				final long nextValue = column.readLong(bytes, offset);
				final long insertValue = _insertLongValues[column.getColumnIndex()];
				if (insertValue == nextValue) {
					throw new CorruptIndexException("Neuer Wert für " + column.getData() + " nicht eindeutig: " + insertValue + " (nachfolgender Wert: " + nextValue + ")", getFile());
				}
			}
		}
	}


	private void serializeData() {
		List<IndexContentDescriptor.IndexColumn> columns = _indexContentDescriptor.getColumns();
		for(int i = 0; i < columns.size(); i++) {
			IndexContentDescriptor.IndexColumn column = columns.get(i);
			if(column.getType() == ColumnType.String) {
				String value = _insertStringValues[i];
				column.writeBytes(value, _tmp);
			}
			else {
				long value = _insertLongValues[i];
				column.writeBytes(value, _tmp);
			}
		}
	}

	/**
	 * Gibt den Wert in der letzten Zeile un der angegebenen Spalte zurück
	 * @param column Spalte
	 * @return Wert
	 * @throws IndexException Lesefehler oder es gibt keinen aktuellen Eintrag
	 */
	@Override
	public long getLast(E column) throws IndexException {
		if(_closed) throw new IllegalStateException();
		try {
			if(_storage.numEntries() == 0) return -1;
			_storage.getLast(_tmp);
			return _indexContentDescriptor.getColumn(column).readLong(_tmp);
		}
		catch(IOException e) {
			throw new CorruptIndexException("Fehler beim Ermitteln des letzten Eintrags", _storage.getFile(), e);
		}
	}


	/**
	 * Gibt den Wert in der ersten Zeile un der angegebenen Spalte zurück
	 * @param column Spalte
	 * @return Wert
	 * @throws IndexException Lesefehler oder es gibt keinen aktuellen Eintrag
	 */
	@Override
	public long getFirst(E column) throws IndexException {
		if(_closed) throw new IllegalStateException();
		try {
			if(_storage.numEntries() == 0) return -1;
			_storage.getFirst(_tmp);
			return _indexContentDescriptor.getColumn(column).readLong(_tmp);
		}
		catch(IOException e) {
			throw new CorruptIndexException("Fehler beim Ermitteln des ersten Eintrags", _storage.getFile(), e);
		}
	}

	/**
	 * Zugriff auf die Low-Level-Speicherungsschicht
	 * @return Siehe {@link de.bsvrz.ars.ars.persistence.index.backend.storage.IndexStorage}
	 */
	public HybridStorage getStorage() {
		return _storage;
	}

	/**
	 * Schließt den Index und speichert alle geänderten Daten auf Platte. Nach dem Aufruf von close() darf die Index-Instanz nicht mehr verwendet werden.
	 *
	 * @throws IndexException Lesefehler der Indexe (z. B. korrupt)
	 */
	@Override
	public void close() throws IndexException {
		_closed = true;
		try {
			_storage.close();
		}
		catch(IOException e) {
			throw new IndexException("Fehler beim Schließen des Indexes", getFile(), e);
		}
	}

	@Override
	public void flush() throws IndexException {
		try {
			_storage.flush();
		}
		catch(IOException e) {
			throw new IndexException("Fehler beim Flushen des Indexes", getFile(), e);
		}
	}

	/**
	 * Anzahl der Einträge, die dieser Index speichert
	 *
	 * @return Anzahl der Einträge
	 */
	@Override
	public long numEntries() {
		return getStorage().numEntries();
	}

	@Override
	public String toString() {
		return getFile().toString();
	}

	/**
	 * Gibt die Datei zurück
	 * @return die Datei, in der der Index gespeichert wird.
	 */
	@Override
	public Path getFile() {
		return _storage.getFile();
	}

	/**
	 * Löscht alle Einträge, die {@link #query(Map)} mit dem entsprechenden Parameter zurückliefern würde. (Ausnahme: Der letzte Container vor dem Anfragebereich,
	 * der ggf. als Startwert zurückgegeben würde, wird nicht gelöscht).
	 * @param query Anfrage (siehe {@link #query(Map)})
	 */
	@Override
	public void removeAll(Map<E, LongRange> query) throws IndexException {
		if(_closed) throw new IllegalStateException();
		if(_storage.numEntries() == 0) return;

		try {
			LongRange range = indexQuery(query);

			if(range == null) return;
			long lowerEndpoint = range.lowerEndpoint();
			long upperEndpoint = range.upperEndpoint();

			byte[] data = new byte[_indexContentDescriptor.getEntryLengthBytes()];
			for(long indexIdx = lowerEndpoint; indexIdx <= upperEndpoint; indexIdx++) {
				_storage.getEntries(indexIdx, 1, data);
				if(isDirectMatch(data, query)) {
					// Eintrag aus Datei bzw. Speicher löschen
					_storage.deleteEntryAtIndex(indexIdx);
					indexIdx--;
					upperEndpoint--; // Oberes Iterationsende anpassen
				}
			}
		}
		catch(IOException e) {
			throw new IndexException("Fehler beim Löschen aus Index", _storage.getFile(), e);
		}
	}

	/**
	 * Gibt einen Bereich von Indexeinträgen zurück, die für eine Anfrage relevant sind
	 * @param query Anfrage (vgl. {@link #query(Map)})
	 * @return Eine {@link LongRange}, die den Bereich der Indexdatei zurückgibt, der die für diese Anfrage relevanten Daten enthält.
	 * Die enthaltenen Long-Werte sind als nullbasierte Zeilennummer (Offset. bzw. Index) innerhalb des Indexes zu verstehen.
	 * Wird {@code null} zurückgegeben, bedeutet dass, dass keine Daten der Anfrage entsprechen.
	 * @throws IOException Fehler beim Lesen des Index
	 */
	@Nullable
	private LongRange indexQuery(final Map<E, LongRange> query) throws IOException {
		LongRange range = new LongRange(0L, _storage.numEntries() - 1);
		for(IndexImplementation indexImplementation : _indexes.values()) {

			LongRange inputRange = query.get(indexImplementation.getColumn().getData());
			if(inputRange == null) continue;

			LongRange queryResult = indexImplementation.query(inputRange);
			if(queryResult == null) {
				// Keine Container im Zeitbereich
				range = null;
				break;
			}
			range = range.intersection(queryResult);
			if(range == null) {
				// Keine Container im Zeitbereich
				break;
			}
		}
		return range;
	}

	private class StrictlyIncreasingIndex implements IndexImplementation {

		private final IndexContentDescriptor.IndexColumn _column;

		public StrictlyIncreasingIndex(final IndexContentDescriptor.IndexColumn column) {
			_column = column;
		}

		@Nullable
		@Override
		public LongRange query(LongRange inputRange) throws IOException {
			Long lowerBound = null;
			Long upperBound = null;
			byte[] tmp = new byte[_storage.entryByteSize()];

			if(inputRange.hasLowerBound()) {
				lowerBound = findEntryIndexOrPreviousLowest(inputRange.lowerEndpoint(), tmp);
			}
			if(inputRange.hasUpperBound()) {
				upperBound = findEntryIndexOrPreviousHighest(inputRange.upperEndpoint(), tmp);
				if(upperBound == -1) {
					// Anfrage liegt vor dem ersten Datensatz
					return null;
				}
			}

			return new LongRange(lowerBound, upperBound);
		}

		public long findEntryIndexOrPreviousLowest(final long searchValue, final byte[] tmp) throws IOException {
			long index = findEntryIndex(searchValue, tmp);
			if(index < 0) {
				index = -(index + 2);
			}
			return index;
		}

		public long findEntryIndexOrPreviousHighest(final long searchValue, final byte[] tmp) throws IOException {
			long index = findEntryIndex(searchValue, tmp);
			if(index < 0) {
				index = -(index + 2);
			}
			return index;
		}

		@Override
		public long findEntryIndex(final long searchValue, final byte[] tmp) throws IOException {
			long low;
			long high = _storage.numEntries() - 1;
			long firstMemoryIndex = _storage.firstMemoryIndex();
			if(firstMemoryIndex <= high && getValue(firstMemoryIndex, tmp) < searchValue) {
				// Nur im Speicher suchen
				low = firstMemoryIndex;
			}
			else {
				// In Datei mitsuchen
				low = 0;
				if(_storage.numEntries() == 0 || searchValue < getValue(low, tmp)) {
					return -1; // Eintrag liegt vor erstem Wert in der Datei, Rückgabe von -(0 + 1)
				}
			}
			long highValue = getValue(high, tmp);
			if(searchValue > highValue) {
				return -(_storage.numEntries() + 1); // Eintrag liegt nach letztem Wert
			}
			return binarySearch(low, high, searchValue, tmp);
		}

		@Override
		public IndexContentDescriptor.IndexColumn getColumn() {
			return _column;
		}

		private long binarySearch(long low, long high, final long searchValue, final byte[] tmp) throws IOException {
			while(low <= high) {
				long mid = low + ((high - low) / 2);
				long midValue = getValue(mid, tmp);

				if(midValue < searchValue) {
					low = mid + 1;
				}
				else if(midValue > searchValue) {
					high = mid - 1;
				}
				else {
					return mid;
				}
			}
			return -(low + 1);
		}

		public long getValue(final long index, final byte[] tmp) throws IOException {
			_storage.getEntries(index, 1, tmp);
			return _column.readLong(tmp);
		}
	}

	private class IncreasingIndex extends StrictlyIncreasingIndex {
		public IncreasingIndex(final IndexContentDescriptor.IndexColumn column) {
			super(column);
		}

		@Override
		public long findEntryIndexOrPreviousLowest(final long searchValue, final byte[] tmp) throws IOException {
			long indexLowest = super.findEntryIndexOrPreviousLowest(searchValue, tmp);
			while(indexLowest > 0 && getValue(indexLowest - 1, tmp) == searchValue) {
				indexLowest--;
			}
			return indexLowest;
		}

		@Override
		public long findEntryIndexOrPreviousHighest(final long searchValue, final byte[] tmp) throws IOException {
			long indexHighest = super.findEntryIndexOrPreviousHighest(searchValue, tmp);
			while(indexHighest < _storage.numEntries() - 1 && getValue(indexHighest + 1, tmp) == searchValue) {
				indexHighest++;
			}
			return indexHighest;
		}
	}
}
