/*
 * Copyright 2019-2020 by Kappich Systemberatung, Aachen
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.ars.ars.persistence.index.backend.management;

import de.bsvrz.ars.ars.persistence.index.IndexException;
import de.bsvrz.ars.ars.persistence.index.result.IndexResult;

import java.nio.file.Path;
import java.util.Map;

/**
 * Basis-Implementierung für Indexe
 *
 * @author Kappich Systemberatung
 * @param <E> Spaltentyp
 */
public interface BaseIndex<E extends Enum<E>> extends AutoCloseable {
	/**
	 * Ermittelt alle Index-Zeilen, die zu der angegebenen Anfrage passen. Das Ergebnis wird performant ermittelt, indem
	 * alle bekannten Einschränkungen ausgewertet werden. Wird nach Spalten eingeschränkt, die nicht im Index vorkommen,
	 * dann werden ggf. überflüssige Zeilen/Container usw. zurückgegeben. Hier muss dann der Anfrager entsprechend selbst filtern.
	 *
	 * @param query Anfrage. Zu jeder Spalte in dem Index kann ein von/bis-Bereich angegeben werden, zu dem Container geliefert werden sollen.
	 *			  Wie für Archivdaten üblich, wird als erster Container ggf. der Container zurückgegeben, der den letzten Datensatz vor dem Anfragezeitraum
	 *			  enthält.
	 * @return Ein IndexResult, der die Ergebnistabelle enthält
	 * @throws IndexException Lesefehler oder korrupter Index
	 */
	IndexResult<E> query(Map<E, LongRange> query) throws IndexException;

	/**
	 * Ermittelt alle Index-Zeilen
	 *
	 * @return Ein IndexResult, der die Ergebnistabelle enthält
	 * @throws IndexException Lesefehler oder korrupter Index
	 */
	IndexResult<E> query() throws IndexException;

	/**
	 * Setzt einen Wert der Einfügezeile, die später mit {@link #insert()} eingefügt werden kann.
	 *
	 * @param column Spalte
	 * @param value  Wert
	 */
	void setInsertValue(E column, long value);

	/**
	 * Setzt einen Wert der Einfügezeile, die später mit {@link #insert()} eingefügt werden kann.
	 *
	 * @param column Spalte
	 * @param value  Wert
	 */
	void setInsertValue(E column, String value);

	/**
	 * Gibt den ersten (der Sortierung nach am kleinsten) Eintrag zurück (der am Dateianfang gespeichert ist)
	 *
	 * @return IndexResult mit einem oder 0 Einträgen (falls der Index leer ist)
	 * @throws IndexException Lesefehler
	 */
	IndexResult<E> first() throws IndexException;

	/**
	 * Fügt die vorher mit {@link #setInsertValue(Enum, long)} gesetzten Werte in den Index ein.
	 *
	 * @throws IndexException Lesefehler der Indexe (z. B. korrupt)
	 */
	void insert() throws IndexException;

	/**
	 * Fügt die vorher mit {@link #setInsertValue(Enum, long)} gesetzten Werte in den Index ein oder überschreibt den bisher gespeicherten Wert. Welcher Wert überschrieben wird, ergibt sich aus Spalten, die eindeutige Werte (Primärschlüssel) enthalten. Ist keine passende Zeile vorhanden, wird eine neue erstellt. 
	 *
	 * @throws IndexException Lesefehler der Indexe (z. B. korrupt)
	 */
	void insertOrReplace() throws IndexException;

	/**
	 * Fügt die vorher mit {@link #setInsertValue(Enum, long)} gesetzten Werte in den Index ein oder prüft, ob das funktionieren würde.
	 *
	 * @param onlyCheck falls true wird nur geprüft, ob die neuen Werte eingefügt werden können, ohne dass die Monotoniebedingungen verletzt werden würden.
	 * @throws IndexException Lesefehler der Indexe (z. B. korrupt)
	 */
	void insert(boolean onlyCheck) throws IndexException;

	/**
	 * Fügt die vorher mit {@link #setInsertValue(Enum, long)} gesetzten Werte in den Index ein oder überschreibt den bisher gespeicherten Wert. Welcher Wert überschrieben wird, ergibt sich aus Spalten, die eindeutige Werte (Primärschlüssel) enthalten. Ist keine passende Zeile vorhanden, wird eine neue erstellt. 
	 *
	 * @param onlyCheck falls true wird nur geprüft, ob die neuen Werte eingefügt werden können, ohne dass die Monotoniebedingungen verletzt werden würden.
	 * @throws IndexException Lesefehler der Indexe (z. B. korrupt)
	 */
	void insertOrReplace(boolean onlyCheck) throws IndexException;

	/**
	 * Fügt die vorher mit {@link #setInsertValue(Enum, long)} gesetzten Werte hinten an den Index an.
	 *
	 * @throws IndexException Lesefehler der Indexe (z. B. korrupt)
	 */
	void append() throws IndexException;

	/**
	 * Fügt die vorher mit {@link #setInsertValue(Enum, long)} gesetzten Werte hinten an den Index an.
	 *
	 * @param onlyCheck falls true wird nur geprüft, ob die neuen Werte eingefügt werden können, ohne dass die Monotoniebedingungen verletzt werden würden und der Index wird nicht verändert.
	 * @throws IndexException Lesefehler der Indexe (z. B. korrupt)
	 */
	void append(boolean onlyCheck) throws IndexException;

	/**
	 * Gibt den Wert in der letzten Zeile un der angegebenen Spalte zurück
	 * @param column Spalte
	 * @return Wert
	 * @throws IndexException Lesefehler oder es gibt keinen aktuellen Eintrag
	 */
	long getLast(E column) throws IndexException;

	/**
	 * Gibt den Wert in der ersten Zeile un der angegebenen Spalte zurück
	 * @param column Spalte
	 * @return Wert
	 * @throws IndexException Lesefehler oder es gibt keinen aktuellen Eintrag
	 */
	long getFirst(E column) throws IndexException;

	/**
	 * Schließt den Index und speichert alle geänderten Daten auf Platte. Nach dem Aufruf von close() darf die Index-Instanz nicht mehr verwendet werden.
	 *
	 * @throws IndexException Lesefehler der Indexe (z. B. korrupt)
	 */
	@Override
	void close() throws IndexException;

	/**
	 * Gibt die Datei zurück
	 *
	 * @return die Datei, in der der Index gespeichert wird.
	 */
	Path getFile();

	/**
	 * Löscht alle Einträge, die {@link #query(Map)} mit dem entsprechenden Parameter zurückliefern würde. (Ausnahme: Der letzte Container vor dem Anfragebereich,
	 * der ggf. als initialer Anfangswert zurückgegeben würde, wird nicht gelöscht).
	 * @param query Anfrage (siehe {@link #query(Map)})
	 * @throws IndexException Lese-/Schreibfehler der Indexe (z. B. korrupt)
	 */
	void removeAll(Map<E, LongRange> query) throws IndexException;

	/**
	 * Anzahl der Einträge, die dieser Index speichert
	 *
	 * @return Anzahl der Einträge
	 */
	long numEntries();

	/**
	 * Schreibt den Index auf die Festplatte
	 * @throws IndexException Lese-/Schreibfehler der Indexe (z. B. korrupt)
	 */
	void flush() throws IndexException;
}
