/*
 *
 * Copyright 2017-2020 by Kappich Systemberatung, Aachen
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.ars.ars.persistence.index.backend.management;

import static java.nio.charset.StandardCharsets.ISO_8859_1;


import de.bsvrz.sys.funclib.kappich.annotations.NotNull;
import de.bsvrz.sys.funclib.kappich.annotations.Nullable;
import java.util.ArrayList;
import java.util.Collections;
import java.util.EnumMap;
import java.util.List;

/**
 * Klasse, die die Inhalte einer Indexdatei beschreibt.
 *
 * @author Kappich Systemberatung
 * @param <E> Enum-Klasse für "Spalten" im Index
 */
public class IndexContentDescriptor<E extends Enum<E>> {

	private final EnumMap<E, IndexColumn> _columnMap;
	private final List<IndexColumn> _columns = new ArrayList<>();
	private int _lengthBytes;

	/**
	 * Erstellt einen neuen IndexContentDescriptor
	 *
	 * @param elementClass Klasse für Spalten
	 */
	public IndexContentDescriptor(Class<E> elementClass) {
		_columnMap = new EnumMap<>(elementClass);
	}

	/**
	 * Gibt alle Spalten zurück
	 *
	 * @return alle Spalten
	 */
	public List<IndexColumn> getColumns() {
		return Collections.unmodifiableList(_columns);
	}

	/**
	 * Gibt den Speicherplatzverbrauch einer Zeile zurück
	 *
	 * @return den Speicherplatzverbrauch einer Zeile
	 */
	public int getEntryLengthBytes() {
		return _lengthBytes;
	}

	/**
	 * Fügt eine Spalte hinzu
	 * @param bytes Anzahl Bytes
	 * @param data  Wert, der gespeichert wird
	 * @param type  Art der Spalte (sortiert, eindeutig, ...)
	 */
	public void addColumn(int bytes, E data, ColumnType type) {
		addColumn(bytes, data, type, false);
	}

	/**
	 * Fügt zwei Spalte, die einen Bereich darstellen hinzu
	 * @param bytes Anzahl Bytes
	 * @param data1  Wert, der für die Min-Spalte gespeichert wird
	 * @param type1  Art der Min-Spalte (sortiert, eindeutig, ...)
	 * @param data2  Wert, der für die Max-Spalte gespeichert wird
	 * @param type2  Art der Max-Spalte (sortiert, eindeutig, ...)
	 */
	public void addRange(int bytes, E data1, ColumnType type1, E data2, ColumnType type2) {
		addColumn(bytes, data1, type1, false);
		addColumn(bytes, data2, type2, true);
	}

	private void addColumn(int bytes, E data, ColumnType type, boolean prevIsMin) {
		IndexColumn min = null;
		if(prevIsMin) {
			min = _columns.get(_columns.size() - 1);
		}
		IndexColumn column = new IndexColumn(_columns.size(), bytes, _lengthBytes, type, data, min);
		_columns.add(column);
		_columnMap.put(data, column);
		_lengthBytes += bytes;
	}

	/**
	 * Gibt die {@link IndexColumn}-Instanz für eine Indexspalte
	 *
	 * @param value Spalte
	 * @return IndexColumn-Objekt mit Dateistruktur-Informationen
	 */
	public IndexColumn getColumn(final E value) {
		return _columnMap.get(value);
	}

	@Override
	public String toString() {
		return "IndexContentDescriptor{" + _columnMap + '}';
	}

	/**
	 * Eine Spalte in einer Indexdatei
	 */
	public static class IndexColumn {
		/** Spaltenindex (position) */
		private final int _index;

		/** Länge in Bytes */
		private final int _lengthBytes;

		/** Offset in bytes je Zeile */
		private final int _offsetBytes;

		/** Typ (String oder Zahl, Sortierung, ...) */
		private final ColumnType _type;

		/**
		 * Enum-Wert, der der Spalte zugeordnet ist
		 */
		private final Enum<?> _data;

		/**
		 * Falls zwei aufeinander folgende Spalten einen Min- und Max-Wert darstellen, wird hier bei der Max-Spalte
		 * auf die Min-Spalte verwiesen. Sonst null.
		 */
		@Nullable
		private final IndexColumn _minColumn;

		private IndexColumn(final int index, final int lengthBytes, final int offsetBytes, final ColumnType type, final Enum<?> data, @Nullable final IndexColumn minColumn) {
			_index = index;
			_lengthBytes = lengthBytes;
			_offsetBytes = offsetBytes;
			_type = type;
			_data = data;
			_minColumn = minColumn;
		}

		private static byte long7(long x) { return (byte) (x >> 56); }
		private static byte long6(long x) { return (byte) (x >> 48); }
		private static byte long5(long x) { return (byte) (x >> 40); }
		private static byte long4(long x) { return (byte) (x >> 32); }
		private static byte long3(long x) { return (byte) (x >> 24); }
		private static byte long2(long x) { return (byte) (x >> 16); }
		private static byte long1(long x) { return (byte) (x >>  8); }
		private static byte long0(long x) { return (byte) x; }
		private static long readLong7(final byte x) { return ((long) x & 0xFFL) << 56; }
		private static long readLong6(final byte x) { return ((long) x & 0xFFL) << 48; }
		private static long readLong5(final byte x) { return ((long) x & 0xFFL) << 40; }
		private static long readLong4(final byte x) { return ((long) x & 0xFFL) << 32; }
		private static long readLong3(final byte x) { return ((long) x & 0xFFL) << 24; }
		private static long readLong2(final byte x) { return ((long) x & 0xFFL) << 16; }
		private static long readLong1(final byte x) { return ((long) x & 0xFFL) << 8; }
		private static long readLong0(final byte x) { return (long) x & 0xFFL; }

		public int getLengthBytes() {
			return _lengthBytes;
		}

		public ColumnType getType() {
			return _type;
		}

		/**
		 * Liest einen Long-Wert aus einem Array
		 * @param data Daten
		 * @return Long-Wert
		 */
		public long readLong(byte[] data) {
			return readLong(data, 0);
		}

		/**
		 * Liest einen Long-Wert aus einem Array
		 * @param data Daten
		 * @param baseOffset zusätzlicher Byte-offset, ab dem gelesen werden soll
		 * @return Long-Wert
		 */
		public long readLong(byte[] data, final int baseOffset) {
			int offset = _offsetBytes + baseOffset;
			long result = 0;
			switch(_lengthBytes) {
				// Fallthrough
				case 8:
					result |= readLong7(data[offset++]);
				case 7:
					result |= readLong6(data[offset++]);
				case 6:
					result |= readLong5(data[offset++]);
				case 5:
					result |= readLong4(data[offset++]);
				case 4:
					result |= readLong3(data[offset++]);
				case 3:
					result |= readLong2(data[offset++]);
				case 2:
					result |= readLong1(data[offset++]);
				case 1:
					result |= readLong0(data[offset]);
			}
			return result;
		}

		/**
		 * Liest einen String aus dem Array
		 * @param data Daten-Array
		 * @return String
		 */
		public String readString(byte[] data) {
			return new String(data, _offsetBytes, _lengthBytes, ISO_8859_1);
		}

		public int getColumnIndex() {
			return _index;
		}

		public Enum<?> getData() {
			return _data;
		}

		/**
		 * Schreibt eine Zahl in ein byte-Array
		 * @param value  Zahl
		 * @param result Ergebnis-Array
		 */
		public void writeBytes(final long value, final byte[] result) {
			int offset = _offsetBytes;
			switch(_lengthBytes) {
				// Fallthrough
				case 8:
					result[offset++] = long7(value);
				case 7:
					result[offset++] = long6(value);
				case 6:
					result[offset++] = long5(value);
				case 5:
					result[offset++] = long4(value);
				case 4:
					result[offset++] = long3(value);
				case 3:
					result[offset++] = long2(value);
				case 2:
					result[offset++] = long1(value);
				case 1:
					result[offset] = long0(value);
					return;
			}
			throw new UnsupportedOperationException(_lengthBytes + " Länge");
		}

		public void writeBytes(final String value, final byte[] result) {
			int offset = _offsetBytes;
			for(int i = 0; i < value.length(); i++) {
				result[offset++] = (byte) value.charAt(i);
			}
			for(int i = value.length(); i < _lengthBytes; i++) {
				result[offset++] = 0;
			}
		}

		@Override
		public String toString() {
			return "IndexColumn{" +
					"type=" + _type +
					", data=" + _data +
					'}';
		}

		@NotNull
		public IndexColumn getMinColumn() {
			if(_minColumn == null) return this;
			return _minColumn;
		}
	}
}
