/*
 *
 * Copyright 2017-2019 by Kappich Systemberatung, Aachen
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.ars.ars.persistence.index.backend.management;

import de.bsvrz.sys.funclib.kappich.annotations.Nullable;

import java.util.NoSuchElementException;

/**
 * Einfache Klasse, die einen Bereich von Long-Werten für die Verwendung in Indexen darstellt.
 *
 * @author Kappich Systemberatung
 */
public class LongRange {

	/**
	 * Untere Begrenzung oder null falls es keine untere Grenze gibt
	 */
	@Nullable
	private final Long _lowerEndPoint;

	/**
	 * Obere Grenze oder null falls es keine obere Grenze gibt
	 */
	@Nullable
	private final Long _upperEndPoint;

	/**
	 * Erstellt eine neue LongRange
	 * @param lowerEndPoint untere Grenze (inklusive)
	 * @param upperEndPoint obere Grenze (inklusive)
	 */
	public LongRange(@Nullable final Long lowerEndPoint, @Nullable final Long upperEndPoint) {
		_lowerEndPoint = lowerEndPoint;
		_upperEndPoint = upperEndPoint;
	}

	/**
	 * Gibt die untere Grenze (inklusive) zurück
	 * @return untere Grenze
	 * @throws NoSuchElementException falls es keine untere Grenze gibt
	 */
	public long lowerEndpoint() {
		if(_lowerEndPoint == null) {
			throw new NoSuchElementException("Keine untere Begrenzung der Anfrage");
		}
		return _lowerEndPoint;
	}

	/**
	 * Gibt die obere Grenze (inklusive) zurück
	 * @return obere Grenze
	 * @throws NoSuchElementException falls es keine obere Grenze gibt
	 */
	public long upperEndpoint() {
		if(_upperEndPoint == null) {
			throw new NoSuchElementException("Keine obere Begrenzung der Anfrage");
		}
		return _upperEndPoint;
	}

	/**
	 * Gibt {@code true} zurück, wenn es eine untere Grenze gibt
	 * @return {@code true}, wenn es eine untere Grenze gibt, sonst {@code false}
	 */
	public boolean hasLowerBound() {
		return _lowerEndPoint != null;
	}

	/**
	 * Gibt {@code true} zurück, wenn es eine obere Grenze gibt
	 * @return {@code true}, wenn es eine obere Grenze gibt, sonst {@code false}
	 */
	public boolean hasUpperBound() {
		return _upperEndPoint != null;
	}

	/**
	 * Gibt {@code true} zurück, wenn der angegebene Wert enthalten ist
	 * @param value Wert
	 * @return {@code true}, wenn der angegebene Wert enthalten ist, sonst {@code false}
	 */
	public boolean contains(long value){
		return (_lowerEndPoint == null || _lowerEndPoint <= value)
				&& (_upperEndPoint == null || _upperEndPoint >= value);
	}

	/**
	 * Berechnet die Schnittmenge von dieser LongRange und einer anderen LongRange. Es wird der Long-Bereich zurückgegeben,
	 * der in beiden Bereichen enthalten ist (und-Verknüpfung).
	 * @param other Andere Long-Range (!= null)
	 * @return Schnittmenge, falls vorhanden, oder {@code null} falls die Bereiche sich nicht überschneiden.
	 */
	@Nullable
	public LongRange intersection(final LongRange other) {
		Long lowerEndPoint = nullsafeMax(_lowerEndPoint, other._lowerEndPoint);
		Long upperEndPoint = nullsafeMin(_upperEndPoint, other._upperEndPoint);
		if(lowerEndPoint != null && upperEndPoint != null && lowerEndPoint > upperEndPoint) return null;
		return new LongRange(lowerEndPoint, upperEndPoint);
	}

	@Nullable
	private static Long nullsafeMin(@Nullable final Long a, @Nullable final Long b) {
		if(a == null) return b;
		if(b == null) return a;
		return Math.min(a, b);
	}

	@Nullable
	private static Long nullsafeMax(@Nullable final Long a, @Nullable final Long b) {
		if(a == null) return b;
		if(b == null) return a;
		return Math.max(a, b);
	}

	@Override
	public String toString() {
		return format(_lowerEndPoint) + "..." + format(_upperEndPoint);
	}

	private static String format(@Nullable final Long point) {
		if(point == null) return "~";
		return point.toString();
	}
}
