/*
 *
 * Copyright 2017-2020 by Kappich Systemberatung, Aachen
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.ars.ars.persistence.index.backend.storage;

import de.bsvrz.sys.funclib.debug.Debug;
import de.bsvrz.sys.funclib.kappich.annotations.Nullable;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;

/**
 * Eine {@link IndexStorage}-Implementierung, die einen Puffer für neu angelegte Objekte im Speicher hält und gleichzeitig auf eine Datei verweist.
 * <p>
 * Läuft der RAM-Puffer über, werden die Einträge an die Datei angehängt.
 * <p>
 * Beim Lesen wird die Datei und der RAM-Puffer wie ein zusammenhängender Speicherbereich angesprochen.
 *
 * @author Kappich Systemberatung
 */
public final class HybridStorage implements IndexStorage, AutoCloseable {

	private final MemoryIndexStorage _memoryIndexStorage;
	private final Path _file;
	private long _fileEntries;
	@Nullable
	private BufferedIndexStorage _fileIndexStorage;
	private static final Debug _debug = Debug.getLogger();
	private final int _memoryLimit;

	/**
	 * Erzeugt eine neue Instanz
	 *
	 * @param entryByteSize Byte-Größe eines Eintrags
	 * @param memoryLimit   Maximale Anzahl Einträge, die im Speicher gehalten werden
	 * @param file          Datei, in die die Einträge geschrieben werden sollen (in der Regel != null, außer für bestimmte Tests)
	 * @throws IOException  IO-Fehler beim Lesen
	 */
	public HybridStorage(final int entryByteSize, final int memoryLimit, @Nullable final Path file) throws IOException {
		_memoryLimit = memoryLimit;
		_memoryIndexStorage = new MemoryIndexStorage(entryByteSize, _memoryLimit);
		_file = file;
		if(_file != null && Files.exists(_file)) {
			long size = Files.size(_file);
			if (size % entryByteSize != 0) {
				throw new IOException("Dateilänge ist nicht durch " + entryByteSize + " teilbar: " + size);
			}
			_fileEntries = size / entryByteSize;
		}
		else if(_file == null) {
			_debug.warning("Initialisierung mit _file == null");
		}
	}

	@Override
	public void getEntries(final long entryIndex, final int numEntries, final byte[] result, final int destPos) throws IOException {
		if(entryIndex >= _fileEntries) {
			_memoryIndexStorage.getEntries(entryIndex - _fileEntries, numEntries, result, destPos);
			return;
		}
		BufferedIndexStorage storage = getFileStorage();
		int numFileEntries = (int) Math.min(_fileEntries - entryIndex, numEntries);
		storage.getEntries(entryIndex, numFileEntries, result, destPos);
		if(numFileEntries < numEntries) {
			_memoryIndexStorage.getEntries(0, numEntries - numFileEntries, result, destPos + numFileEntries * entryByteSize());
		}
	}

	@Override
	public void setEntries(final long entryIndex, final int numEntries, final byte[] data, final int fromPos) throws IOException {
		if(entryIndex >= _fileEntries) {
			_memoryIndexStorage.setEntries(entryIndex - _fileEntries, numEntries, data, fromPos);
			return;
		}
		BufferedIndexStorage storage = getFileStorage();
		int numFileEntries = (int) Math.min(_fileEntries - entryIndex, numEntries);
		storage.setEntries(entryIndex, numFileEntries, data, fromPos);
		if(numFileEntries < numEntries) {
			_memoryIndexStorage.setEntries(0, numEntries - numFileEntries, data, fromPos + numFileEntries * entryByteSize());
		}
	}

	@Override
	public void insertEntries(final long entryIndex, final int numEntries, final byte[] data, final int fromPos) throws IOException {
		if(entryIndex >= _fileEntries && _memoryIndexStorage.numEntries() + numEntries < _memoryIndexStorage.maxNumEntries()) {
			// Kann in Speicher eingefügt werden, kein Problem
			_memoryIndexStorage.insertEntries(entryIndex - _fileEntries, numEntries, data);
			return;
		}
		// Sonst erst den MemoryIndexStorage flushen (in die Datei schreiben) und dann die zusätzlichen Einträge in der Datei einsortieren
		// In der Regel wird hinten angehängt, d.h. kein Performance-Problem
		BufferedIndexStorage storage = getFileStorage();
		flushMemory(storage);
		storage.insertEntries(entryIndex, numEntries, data);
		_fileEntries += numEntries;
	}

	private void flushMemory(final BufferedIndexStorage fileStorage) throws IOException {
		_memoryIndexStorage.flushTo(fileStorage);
		_fileEntries = fileStorage.numEntries();
	}

	private BufferedIndexStorage getFileStorage() throws IOException {
		if(_fileIndexStorage == null) {
			_fileIndexStorage = new BufferedIndexStorage(new FileIndexStorage(_file, entryByteSize()), _memoryLimit);
		}
		return _fileIndexStorage;
	}

	/**
	 * Gibt die Datei zurück
	 * @return die Datei
	 */
	public Path getFile() {
		return _file;
	}

	@Override
	public long numEntries() {
		return _fileEntries + _memoryIndexStorage.numEntries();
	}

	/**
	 * Gibt den Index des ersten Eintrags im Speicher zurück, bzw. die Anzahl der in der datei gespeicherten Einträge
	 * @return den Index des ersten Eintrags im Speicher
	 */
	public long firstMemoryIndex() {
		return _fileEntries;
	}

	/**
	 * Gibt die aktuelle Anzahl der im Speicher gepufferten Einträge zurück
	 * @return aktuelle Anzahl Einträge
	 */
	public int memoryEntries() {
		return (int) _memoryIndexStorage.numEntries();
	}

	@Override
	public int entryByteSize() {
		return _memoryIndexStorage.entryByteSize();
	}

	@Override
	public String toString() {
		BufferedIndexStorage fileIndexStorage = _fileIndexStorage;
		if (fileIndexStorage == null) {
			return getClass().getSimpleName() + " (" + numEntries() + " Einträge, davon 0 im Lesecache und " + memoryEntries() + " im Schreibcache)";
		}
		return getClass().getSimpleName() + " (" + numEntries() + " Einträge, davon " + fileIndexStorage.getBufferedCount() + " im Lesecache und " + memoryEntries() + " im Schreibcache)";
	}

	@Override
	public void close() throws IOException {
		if(_file == null) return;
		if(_fileIndexStorage == null && _memoryIndexStorage.numEntries() == 0) {
			// Datei nicht offen und nichts zu speichern
			if(Files.exists(_file) || !Files.exists(_file.getParent())) {
				// Wenn das Verzeichnis existiert, aber nicht die Datei, Datei trotzdem anlegen (damit nicht nachher eine Indexdatei "fehlt").
				// Wenn das Verzeichnis nicht existiert, brauchen die Indexdateien auch nicht angelegt werden
				// (Teilweise werden im ArchiveTask wegen nachgeforderten Daten Indexe angelegt ohne das Daten existieren)
				return;
			}
		}
		try(BufferedIndexStorage storage = getFileStorage()) {
			flushMemory(storage);
			_fileIndexStorage.close();
		}
		_fileIndexStorage = null;
	}

	@Override
	public void deleteEntryAtIndex(final long entryIndex) throws IOException {
		if(entryIndex >= _fileEntries) {
			// Kann in Speicher gelöscht werden, kein Problem
			_memoryIndexStorage.deleteEntryAtIndex(entryIndex - _fileEntries);
			return;
		}
		BufferedIndexStorage storage = getFileStorage();
		storage.deleteEntryAtIndex(entryIndex);
		_fileEntries--;
	}

	/**
	 * Schreibt alle Änderungen in das Dateisystem
	 *
	 * @throws IOException Fehler beim Schreiben
	 */
	public void flush() throws IOException {
		if(_file == null) return;
		if(_fileIndexStorage == null && _memoryIndexStorage.numEntries() == 0) {
			// Datei nicht offen und nichts zu speichern
			if(Files.exists(_file) || !Files.exists(_file.getParent())) {
				// Wenn das Verzeichnis existiert, aber nicht die Datei, Datei trotzdem anlegen (damit nicht nachher eine Indexdatei "fehlt").
				// Wenn das Verzeichnis nicht existiert, brauchen die Indexdateien auch nicht angelegt werden
				// (Teilweise werden im ArchiveTask wegen nachgeforderten Daten Indexe angelegt ohne das Daten existieren)
				return;
			}
		}
		BufferedIndexStorage storage = getFileStorage();
		flushMemory(storage);
		_fileIndexStorage.flush();
	}
}
