/*
 *
 * Copyright 2017-2020 by Kappich Systemberatung, Aachen
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.ars.ars.persistence.index.backend.storage;

import java.io.IOException;
import java.util.Arrays;

/**
 * Eine {@link IndexStorage}-Implementierung, die die Werte im Speicher hält. Wird als Cache für einfügende Werte benutzt.
 *
 * @author Kappich Systemberatung
 */
public final class MemoryIndexStorage implements IndexStorage {

	private final int _entryByteSize;
	private int _numEntries;
	private final byte[] _data;
	private final int _limit;

	/**
	 * Erzeugt eine Instanz
	 *
	 * @param entryByteSize Bype-Länge einer Zeile
	 * @param limit         Maximale Anzahl Einträge
	 */
	public MemoryIndexStorage(int entryByteSize, final int limit) {
		_entryByteSize = entryByteSize;
		_limit = limit;
		_data = new byte[_limit * entryByteSize];
	}

	@Override
	public void getEntries(final long entryIndex, final int numEntries, final byte[] result, final int destPos) {
		if(numEntries == 0) return;
		if(entryIndex + numEntries > numEntries()) {
			throw new IllegalArgumentException("Enthält nur " + numEntries() + " Elemente, abgefragt wurde aber Element " + entryIndex + " bis " + (entryIndex + numEntries) + ".");
		}
		if(numEntries < 0) {
			throw new IllegalArgumentException("numEntries < 0: " + numEntries);
		}
		int startByte = (int) entryIndex * _entryByteSize;
		System.arraycopy(_data, startByte, result, destPos, numEntries * _entryByteSize);
	}

	@Override
	public void setEntries(final long entryIndex, final int numEntries, final byte[] data, final int fromPos) {
		if(entryIndex + numEntries > numEntries()) {
			throw new IllegalArgumentException("Enthält nur " + numEntries() + " Elemente, abgefragt wurde aber Element " + entryIndex + " bis " + (entryIndex + numEntries) + ".");
		}
		if(numEntries <= 0) {
			throw new IllegalArgumentException("numEntries <= 0: " + numEntries);
		}
		int startByte = (int) entryIndex * _entryByteSize;
		System.arraycopy(data, fromPos, _data, startByte, numEntries * _entryByteSize);
	}

	@Override
	public void insertEntries(final long entryIndex, final int numEntries, final byte[] data, final int fromPos) {
		long currentEntries = numEntries();
		if(numEntries + currentEntries > _limit || numEntries < 0 || entryIndex < 0 || entryIndex > currentEntries) {
			throw new IllegalArgumentException();
		}
		int startShiftByte = (int) entryIndex * _entryByteSize;
		int endShiftByte = _numEntries * _entryByteSize;
		if(endShiftByte > startShiftByte) {
			System.arraycopy(_data, startShiftByte, _data, startShiftByte + numEntries * _entryByteSize, endShiftByte - startShiftByte);
		}
		System.arraycopy(data, fromPos, _data, startShiftByte, numEntries * _entryByteSize);
		_numEntries += numEntries;
	}

	/**
	 * Verschiebt alle Einträge in ein anderes {@link IndexStorage}-Objekt (also z.B. ein dateibasiertes). Danach ist dieses Objekt leer.
	 * @param other Objekt, dass die eigenen Einträge hinzugefügt bekommt
	 * @throws IOException Fehler ein Einfügen
	 * @throws IllegalArgumentException Wenn der Parameter "this" ist.
	 */
	public void flushTo(IndexStorage other) throws IOException {
		if(this == other) {
			throw new IllegalArgumentException();
		}
		other.insertEntries(other.numEntries(), _numEntries, _data, 0);
		_numEntries = 0;
	}

	@Override
	public long numEntries() {
		return _numEntries;
	}

	/**
	 * Gibt die maximale Anzahl Einträge zurück
	 * @return Anzahl
	 */
	public int maxNumEntries() {
		return _limit;
	}

	@Override
	public int entryByteSize() {
		return _entryByteSize;
	}

	@Override
	public void deleteEntryAtIndex(final long entryIndex) {
		int index = (int) entryIndex;
		System.arraycopy(_data, _entryByteSize * (index + 1), _data, _entryByteSize * index, _entryByteSize * (_numEntries - (index + 1)) );
		_numEntries--;
	}

	@Override
	public String toString() {
		byte[] bytes = new byte[(int) (numEntries() * _entryByteSize)];
		getEntries(0, (int) numEntries(), bytes, 0);
		return Arrays.toString(bytes);
	}
}
