/*
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */
package de.bsvrz.ars.ars.persistence.index.result;

import de.bsvrz.ars.ars.persistence.directories.PersistenceDirectory;
import de.bsvrz.ars.ars.persistence.index.backend.management.IndexContentDescriptor;

import java.util.List;
import java.util.Map;
import java.util.NavigableMap;
import java.util.TreeMap;

/**
 * Zusammengefasstes {@link IndexResult} aus Anfragen über mehrere Indexdateien (wenn Persistenzverzeichnis
 * in einzelne Wochenverzeichnisse aufgesplittet wurde).
 *
 * @param <E> Enum-Spaltentyp der Tabelle
 */
public class AggregateIndexResult<E extends Enum<E>> implements LocatedIndexResult<E> {

	private int size;

	/**
	 * Indexergebnisse, aus denen die gesamte Ergebnistabelle zusammengesetzt ist. Der Key gibt dabei jeweils
	 * die Position der Sub-Tabelle in der Gesamttabelle ein, d. h. der erste Eintrag hat den Index und Key 0 und z. B. 47
	 * Indexeinträge, der zweite Eintrag hat dann den Key 47 und 11 Einträge, der dritte Index hat dann als Key 11 + 47
	 * usw.
	 * <p>
	 * Mit dieser Aufteilung kann man die Sub-Tabelle einer Zeilennummer einfach finden indem
	 * {@code _indexesByStartIndex.floorEntry(Zeilennummer)} ausgeführt wird.
	 */
	private final NavigableMap<Integer, LocatedIndexResult<E>> _indexesByStartIndex = new TreeMap<>();

	/**
	 * Fügt ein weiteres Indexergebnis hinzu. Diese Methode muss mit monoton steigenden Archivzeiten
	 * aufgerufen werden.
	 *
	 * @param indexResult Indexabfrageergebnis
	 * @param dir         Zugehöriges Wochenverzeichnis
	 */
	public void add(IndexResult<E> indexResult, PersistenceDirectory dir) {
		if (indexResult.isEmpty() && !_indexesByStartIndex.isEmpty() && _indexesByStartIndex.lastEntry().getValue().hasNext()) {
			// Irrelevante Ergebnisse ignorieren, wenn wir bereits einen "näheren" nächsten Datensatz haben.
			return;
		}
		_indexesByStartIndex.put(size, LocatedIndexResult.adapt(indexResult, dir));
		size += indexResult.size();
	}

	@Override
	public int size() {
		return size;
	}

	@Override
	public long get(int index, E value) {
		Map.Entry<Integer, LocatedIndexResult<E>> entry = _indexesByStartIndex.floorEntry(index);
		return entry.getValue().get(index - entry.getKey(), value);
	}

	@Override
	public int getSignedInt(int index, E value) {
		Map.Entry<Integer, LocatedIndexResult<E>> entry = _indexesByStartIndex.floorEntry(index);
		return entry.getValue().getSignedInt(index - entry.getKey(), value);
	}

	@Override
	public String getString(int index, E value) {
		Map.Entry<Integer, LocatedIndexResult<E>> entry = _indexesByStartIndex.floorEntry(index);
		return entry.getValue().getString(index - entry.getKey(), value);
	}

	@Override
	public boolean getBoolean(int index, E value) {
		Map.Entry<Integer, LocatedIndexResult<E>> entry = _indexesByStartIndex.floorEntry(index);
		return entry.getValue().getBoolean(index - entry.getKey(), value);
	}

	@Override
	public boolean hasNext() {
		return !_indexesByStartIndex.isEmpty() && _indexesByStartIndex.lastEntry().getValue().hasNext();
	}

	@Override
	public long getNext(E value) {
		return _indexesByStartIndex.lastEntry().getValue().getNext(value);
	}

	@Override
	public boolean isEmpty() {
		return size == 0;
	}

	@Override
	public long getMin(E value) {
		return _indexesByStartIndex.values().stream().filter(it -> !it.isEmpty()).mapToLong(it -> it.getMin(value)).min().orElseThrow(IllegalStateException::new);
	}

	@Override
	public long getMax(E value) {
		return _indexesByStartIndex.values().stream().filter(it -> !it.isEmpty()).mapToLong(it -> it.getMax(value)).max().orElseThrow(IllegalStateException::new);
	}

	@Override
	public List<IndexContentDescriptor.IndexColumn> getColumns() {
		if (_indexesByStartIndex.isEmpty()) return List.of();
		return _indexesByStartIndex.firstEntry().getValue().getColumns();
	}

	@Override
	public String toString() {
		return "AggregateIndexResult[" + size() + "]";
	}

	@Override
	public PersistenceDirectory getPersistenceDirectory(int index) {
		Map.Entry<Integer, LocatedIndexResult<E>> entry = _indexesByStartIndex.floorEntry(index);
		return entry.getValue().getPersistenceDirectory(index - entry.getKey());
	}

	@Override
	public PersistenceDirectory getNextPersistenceDirectory() {
		return _indexesByStartIndex.lastEntry().getValue().getNextPersistenceDirectory();
	}
}
