/*
 *
 * Copyright 2017-2020 by Kappich Systemberatung, Aachen
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.ars.ars.persistence.iter;

import de.bsvrz.ars.ars.mgmt.datatree.synchronization.SynchronizationFailedException;
import de.bsvrz.ars.ars.persistence.ContainerDataResult;
import de.bsvrz.ars.ars.persistence.ContainerManagementData;
import de.bsvrz.ars.ars.persistence.IdDataIdentification;
import de.bsvrz.ars.ars.persistence.PersistenceException;
import de.bsvrz.ars.ars.persistence.writer.ArchiveTask;
import de.bsvrz.dav.daf.main.DataState;
import de.bsvrz.dav.daf.main.archive.ArchiveDataKind;
import de.bsvrz.sys.funclib.kappich.annotations.NotNull;
import de.bsvrz.sys.funclib.kappich.annotations.Nullable;
import de.bsvrz.sys.funclib.losb.util.Util;

import java.util.List;

/**
 * Iteration über die Daten von einer Liste von Containern. Die Container müssen bereits nach Datenindex nicht überlappend aufsteigend sortiert sein.
 * <p>
 * Ist das nicht der Fall (z.B. bei nachgeforderten Daten), sollte ein {@link BucketContainerIterator} benutzt werden.
 *
 * @author Kappich Systemberatung
 */
public class SequentialContainerIterator implements DataIterator {

	private final List<QueryDataSequence.IndexedContainer> _containerFiles;
	private DataIterator _dataIterator;
	private int _index;
	private final ArchiveDataKind _adk;
	private final QueryDataSequence.IndexedContainer _nextContainer;
	private final IdDataIdentification _idDataIdentification;
	private ContainerDataResult insertGapDataset = null;
	private final DataGapManager gapManager; 

	/**
	 * Erstellt einen neuen SequentialContainerIterator
	 *
	 * @param containerFiles       Containerdateien, bereits sortiert
	 * @param adk                  Archivdatenart
	 * @param nextContainer        Den Containerdateien folgender Container
	 * @param idDataIdentification Datenidentifikation
	 * @param gapManager           Verwaltung von Datenlücken
	 * @throws PersistenceException Lesefehler im Persistenzverzeichnis
	 */
	public SequentialContainerIterator(final List<QueryDataSequence.IndexedContainer> containerFiles,
	                                   final ArchiveDataKind adk,
	                                   @Nullable final QueryDataSequence.IndexedContainer nextContainer,
	                                   final IdDataIdentification idDataIdentification, DataGapManager gapManager) throws PersistenceException {
		this.gapManager = gapManager;
		if (containerFiles.isEmpty()) throw new IllegalArgumentException("containerFiles");
		
		_adk = adk;
		_nextContainer = nextContainer;
		_containerFiles = containerFiles;
		_idDataIdentification = idDataIdentification;
		
		openContainer(_index);
	}

	private void openContainer(final int index) throws PersistenceException {
		QueryDataSequence.IndexedContainer container = _containerFiles.get(index);
		_dataIterator = getIterator(container);
	}

	@Override
	public void peek(final ContainerDataResult result) throws PersistenceException {
		if (insertGapDataset != null) {
			insertGapDataset.copyTo(result);
			return;
		}
		_dataIterator.peek(result);
	}

	@Override
	public ContainerDataResult peekNext() throws PersistenceException, SynchronizationFailedException {
		if(!isEmpty()) {
			throw new IllegalStateException();
		}
		ContainerDataResult peekNext = _dataIterator.peekNext();
		if(peekNext == null && _nextContainer != null) {
			try(DataIterator iterator = getIterator(_nextContainer)) {
				ContainerDataResult result = new ContainerDataResult();
				iterator.poll(result);
				return result;
			}
		}
		return peekNext;
	}

	private DataIterator getIterator(final QueryDataSequence.IndexedContainer nextContainer) throws PersistenceException {
		return nextContainer.getDirectory()
				.iterator(_adk, nextContainer.getContainerId(), _idDataIdentification);
	}

	@Override
	public long peekDataIndex() {
		if (insertGapDataset != null) {
			return insertGapDataset.getDataIndex();
		}
		return _dataIterator.peekDataIndex();
	}

	@Override
	public long peekDataTime() {
		if (insertGapDataset != null) {
			return insertGapDataset.getDataTime();
		}
		return _dataIterator.peekDataTime();
	}

	@Override
	public long peekArchiveTime() {
		if (insertGapDataset != null) {
			return insertGapDataset.getArchiveTime();
		}
		return _dataIterator.peekArchiveTime();
	}

	@Override
	public void remove() throws PersistenceException, SynchronizationFailedException {
		if (insertGapDataset != null) {
			insertGapDataset = null;
			return;
		}
		long oldArchiveTime = peekArchiveTime();
		long oldDataTime = peekDataTime();
		long oldDataIndex = peekDataIndex();
		_dataIterator.remove();
		while(_dataIterator.isEmpty() && _index < _containerFiles.size() - 1) {
			_dataIterator.close();
			_index++;
			openContainer(_index);

			// Hier wird eine künstliche Lücke eingefügt, wenn ein Wochenverzeichnis fehlt
			long newDataIndex = peekDataIndex();
			long newArchiveTime = peekArchiveTime();
			if (ArchiveTask.isIndexGap(oldDataIndex, newDataIndex) && gapManager.isRangeUnavailable(oldArchiveTime, newArchiveTime)) {
				long gapIndex = Util.dIdxSetArSBit(oldDataIndex);
				if (gapIndex > oldDataIndex) {
					// Wenn die Datenindexe gleich wären, wäre das eine normale Datenlücke, die schon im Archivsystem
					// registriert ist und keine Wochenverzeichnis-Lücke, Sicherstellen, dass hier nicht der Datenindex
					// doppelt versendet wird
					insertGapDataset = new ContainerDataResult();
					insertGapDataset.setArchiveTime(oldArchiveTime);
					insertGapDataset.setDataIndex(gapIndex);
					insertGapDataset.setDataTime(oldDataTime);
					insertGapDataset.setDataState(DataState.UNAVAILABLE_BLOCK);
					insertGapDataset.setDataKind(_adk);
				}
			}
		}
	}

	@Override
	public boolean isEmpty() {
		return _dataIterator.isEmpty();
	}

	@Override
	public void close() {
		_dataIterator.close();
	}

	@NotNull
	@Override
	public ContainerManagementData getContainerManagementData() throws PersistenceException {
		return _dataIterator.getContainerManagementData();
	}
}
