/*
 *
 * Copyright 2017-2020 by Kappich Systemberatung, Aachen
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.ars.ars.persistence.iter;

import de.bsvrz.ars.ars.mgmt.datatree.synchronization.SynchronizationFailedException;
import de.bsvrz.ars.ars.persistence.ContainerDataResult;
import de.bsvrz.ars.ars.persistence.ContainerManagementData;
import de.bsvrz.ars.ars.persistence.PersistenceException;
import de.bsvrz.ars.ars.persistence.SimpleContainerManagementData;
import de.bsvrz.dav.daf.main.archive.ArchiveOrder;
import de.bsvrz.dav.daf.main.archive.ArchiveTimeSpecification;
import de.bsvrz.dav.daf.main.archive.TimingType;
import de.bsvrz.sys.funclib.kappich.annotations.NotNull;

import java.util.*;

/**
 * Diese Klasse entspricht einem {@link CombineDataIterator} und bekommt im Konstruktor als zusätzliches Argument einen Zeitbereich.
 * Diese Klasse filtert alle Datensätze heraus, die nicht im Zeitbereich liegen (außer ggf. den ersten Datensatz vor dem Zeitbereich, der noch im Zeitbereich gültig ist).
 *
 * @author Kappich Systemberatung
 */
public class TimeSpecificationCombineDataIterator extends CombineDataIterator{

	private final ArchiveTimeSpecification _timeSpecification;

	private ContainerDataResult _last;

	private FirstDataSet _firstData;

	/**
	 * Erstellt eine neue Instanz
	 * @param containerSequences Die untergeordneten Sequenzen, eine pro Datenart
	 * @param order Sortierung
	 * @param timeSpecification Zeitbereich
	 * @throws PersistenceException Lesefehler im Persistenzverzeichnis
	 * @throws SynchronizationFailedException Synchronisierung fehlgeschlagen
	 */
	public TimeSpecificationCombineDataIterator(final Collection<DataSequence> containerSequences, ArchiveOrder order, ArchiveTimeSpecification timeSpecification) throws PersistenceException, SynchronizationFailedException {
		super(containerSequences, order);
		_timeSpecification = timeSpecification;
		final List<FirstDataSet> firstDataCandidates = new ArrayList<>();
		for(DataIterator iterator : _iterators) {
			initIterator(iterator, firstDataCandidates);
		}
		super.updateDelegate();

		if(firstDataCandidates.contains(null)) {
			// Wir haben (mindestens) einen Datensatz am Intervallanfang, weitere Initiale datensätze können ignoriert werden.
			_firstData = null;
			return;
		}

		eliminateIllegalCandidates(firstDataCandidates);

		if(firstDataCandidates.isEmpty()) {
			// Es gibt keine gültigen Kandidaten
			_firstData = null;
			return;
		}

		if(timeSpecification.getTimingType() == TimingType.DATA_INDEX) {
			// Alle ersten Datensätze eliminieren, außer den größten
			_firstData = Collections.max(firstDataCandidates, Comparator.comparing(it -> it.getData().getDataIndex()));
		}
		else if(timeSpecification.getTimingType() == TimingType.DATA_TIME) {
			_firstData = Collections.max(firstDataCandidates, Comparator.comparing(it -> it.getData().getDataTime()));
		}
		else if(timeSpecification.getTimingType() == TimingType.ARCHIVE_TIME) {
			_firstData = Collections.max(firstDataCandidates, Comparator.comparing(it -> it.getData().getArchiveTime()));
		}
	}

	/**
	 * Eliminiert alle Kandidaten für den ersten Datensatz, die durch die Sortierung nicht als ersten Datensatz in Frage kommen. Beispiel: Es gibt einen
	 * ON-Datensatz der vom Datenzeitstempel vor dem Anfrageintervall liegt aber einen Datenindex hat der größer ist als der erste OA-Datensatz im Intervall.
	 * <p>
	 * Der ON-Datensatz darf bei einer Sortierung nach Datenindex nicht als initialer Datensatz verschickt werden, da sonst der nächste OA-Datensatz einen
	 * kleineren Datenindex hätte und die Sortierung nicht eingehalten würde.
	 *
	 * @param firstDataCandidates Zu filternde Kandidaten
	 */
	private void eliminateIllegalCandidates(final List<FirstDataSet> firstDataCandidates) {
		if(isEmpty()) {
			// Wenn es keine Datensätze nach dem initialen Datensatz gibt, ist die Sortierung in jeden Fall Monoton, d.h. es muss nichts eliminiert werden.
			return;
		}
		if(getOrder() == ArchiveOrder.BY_INDEX) {
			firstDataCandidates.removeIf(it -> it.getData().getDataIndex() > peekDataIndex());
		}
		else {
			// getOrder() == ArchiveOrder.BY_DATA_TIME
			firstDataCandidates.removeIf(it -> it.getData().getDataTime() > peekDataTime());
		}
	}

	/**
	 * Entfernt vom Iterator alle Datensätze, die vor dem Startzeitpunkt liegen. Der letzte Datensatz vor dem Startzeitpunkt ist der Kandidat für den
	 * initialen Datensatz und wird in firstDataCandidates abgelegt.
	 * Wird kein erster Datensatz vor dem Anfrageintervall benötigt, weil der erste Datensatz genau auf dem Intervallanfang liegt, wird dort null abgelegt.
	 *
	 * @param iterator            Iterator, von dem Datensätze entfernt werden sollen
	 * @param firstDataCandidates In diese Liste werden Kandidaten für den ersten Datensatz eingefügt, oder ein {@code null},
	 *                            wenn die Methode sicher ist, dass kein initialer Datensatz benötigt wird, z. B.
	 *                            falls der erste Datensatz gleich auf dem Anfragestart liegt.
	 */
	private void initIterator(final DataIterator iterator, final List<FirstDataSet> firstDataCandidates) throws PersistenceException, SynchronizationFailedException {
		if(iterator.isEmpty()) {
			// Leere Anfrage
			return;
		}
		if(exceedsUpperBound(iterator)) {
			// Der Datensatz befindet sich hinter dem Anfrageintervall
			removeIfInvalid(iterator); // Alle ungültigen Datensätze entfernen
			return;
		}
		if(exceedsLowerBound(iterator)) {
			// Wir befinden uns vor dem Anfrageintervall
			final ContainerDataResult firstData = new ContainerDataResult();
			while(true) {
				iterator.peek(firstData);
				// Kopie erstellen, falls der Container wieder geschlossen wird.
				final SimpleContainerManagementData firstContainerManagementData = new SimpleContainerManagementData(iterator.getContainerManagementData());
				iterator.remove();
				if(iterator.isEmpty()) {
					// Alle Datensätze sind vor dem Anfrageanfang, den letzten in die Liste einfügen
					firstDataCandidates.add(new FirstDataSet(firstData, firstContainerManagementData));
					return;
				}
				if(!exceedsLowerBound(iterator)) {
					// Wir haben einen Datensatz gefunden, der im Anfrageintervall liegt.
					if(isAtStart(iterator)) {
						// Wir brauchen den vorigen Datensatz nicht, weil wir uns direkt auf dem Intervall-Anfang befinden
						firstDataCandidates.add(null);
					}
					else {
						// Normalfall: Den letzen Datensatz vor dem Anfrageintervall in Liste einfügen.
						firstDataCandidates.add(new FirstDataSet(firstData, firstContainerManagementData));
					}
					return;
				}
			}
		}
		else if(isAtStart(iterator)) {
			// Wir brauchen den initialen Datensatz nicht, weil wir uns direkt auf dem Intervall-Anfang befinden
			firstDataCandidates.add(null);
		}
	}


	/**
	 * Entfernt den aktuellen Datensatz, solange er über dem Anfrageende liegt.
	 * @throws PersistenceException Lesefehler im Persistenzverzeichnis
	 * @param iterator Iterator, aus dem entfernt werden soll
	 */
	private void removeIfInvalid(final DataIterator iterator) throws PersistenceException, SynchronizationFailedException {
		if(iterator.isEmpty()) return;
		if(exceedsUpperBound(iterator)) {
			_last = new ContainerDataResult();
			iterator.poll(_last);
		}
		if(iterator.isEmpty()) return;
		while(exceedsUpperBound(iterator)) {
			iterator.remove();
			if(iterator.isEmpty()) return;
		}
	}

	/**
	 * Entfernt den aktuellen Datensatz, solange er über dem Anfrageende liegt.
	 * @throws PersistenceException Lesefehler im Persistenzverzeichnis
	 */
	private void removeIfInvalid() throws PersistenceException, SynchronizationFailedException {
		if(super.isEmpty()) return;
		if(exceedsUpperBound(this)) {
			_last = new ContainerDataResult();
			super.peek(_last);
			super.remove();  // Nicht this.remove um StackOverFlow zu vermeiden

			if(super.isEmpty()) return;
			while(exceedsUpperBound(this)) {
				super.remove(); // Nicht this.remove um StackOverFlow zu vermeiden
				if(super.isEmpty()) return;
			}
		}
	}

	@Override
	public ContainerDataResult peekNext() throws PersistenceException, SynchronizationFailedException {
		if(!isEmpty()) throw new IllegalStateException();
		return _last != null ? _last : super.peekNext();
	}

	/**
	 * Gibt {@code true} zurück, wenn sich der aktuelle Datensatz des Iterators vor dem definierten Zeitbereich befindet
	 * @return {@code true}, wenn sich der aktuelle Datensatz vor dem Zeitbereich befindet, sonst {@code false}
	 * @param iterator Iterator, der geprüft wird
	 */
	private boolean exceedsLowerBound(final DataIterator iterator) {
		if(_timeSpecification.getTimingType() == TimingType.DATA_INDEX) {
			long dataIndex = iterator.peekDataIndex();
			return dataIndex < _timeSpecification.getIntervalStart();
		}
		if(_timeSpecification.getTimingType() == TimingType.DATA_TIME) {
			long dataTime = iterator.peekDataTime();
			return dataTime < _timeSpecification.getIntervalStart();
		}
		if(_timeSpecification.getTimingType() == TimingType.ARCHIVE_TIME) {
			long archiveTime = iterator.peekArchiveTime();
			return archiveTime < _timeSpecification.getIntervalStart();
		}
		throw new AssertionError("Unbekannter TimingType: " + _timeSpecification.getTimingType());
	}

	/**
	 * Gibt {@code true} zurück, wenn sich der aktuelle Datensatz hinter dem Zeitbereich befindet
	 * @return {@code true}, wenn sich der aktuelle Datensatz hinter dem Zeitbereich befindet, sonst {@code false}
	 * @param iterator Iterator, der geprüft wird
	 */
	private boolean exceedsUpperBound(final DataIterator iterator) {
		if(_timeSpecification.getTimingType() == TimingType.DATA_INDEX) {
			long dataIndex = iterator.peekDataIndex();
			return  dataIndex > _timeSpecification.getIntervalEnd();
		}
		if(_timeSpecification.getTimingType() == TimingType.DATA_TIME) {
			long dataTime = iterator.peekDataTime();
			if(dataTime == 0) return false;
			return dataTime > _timeSpecification.getIntervalEnd();
		}
		if(_timeSpecification.getTimingType() == TimingType.ARCHIVE_TIME) {
			long archiveTime = iterator.peekArchiveTime();
			if(archiveTime == 0) return false;
			return archiveTime > _timeSpecification.getIntervalEnd();
		}
		throw new AssertionError("Unbekannter TimingType: " + _timeSpecification.getTimingType());
	}

	/**
	 * Gibt {@code true} zurück, wenn der aktuelle Datensatz genau auf dem Intervallanfang liegt
	 * @return {@code true}, wenn der aktuelle Datensatz genau auf dem Intervallanfang liegt, sonst {@code false}
	 * @param iterator Iterator
	 */
	private boolean isAtStart(final DataIterator iterator) {
		if(_timeSpecification.getTimingType() == TimingType.DATA_INDEX) {
			long dataIndex = iterator.peekDataIndex();
			return dataIndex == _timeSpecification.getIntervalStart();
		}
		if(_timeSpecification.getTimingType() == TimingType.DATA_TIME) {
			long dataTime = iterator.peekDataTime();
			return dataTime == _timeSpecification.getIntervalStart();
		}
		if(_timeSpecification.getTimingType() == TimingType.ARCHIVE_TIME) {
			long archiveTime = iterator.peekArchiveTime();
			return archiveTime == _timeSpecification.getIntervalStart();
		}
		throw new AssertionError("Unbekannter TimingType: " + _timeSpecification.getTimingType());
	}

	@Override
	public void peek(final ContainerDataResult result) throws PersistenceException {
		if(_firstData != null) {
			_firstData.getData().copyTo(result);
			return;
		}
		super.peek(result);
	}

	@Override
	public long peekDataIndex() {
		if(_firstData != null) {
			return _firstData.getData().getDataIndex();
		}
		return super.peekDataIndex();
	}

	@Override
	public long peekDataTime() {
		if(_firstData != null) {
			return _firstData.getData().getDataTime();
		}
		return super.peekDataTime();
	}

	@Override
	public long peekArchiveTime() {
		if(_firstData != null) {
			return _firstData.getData().getArchiveTime();
		}
		return super.peekArchiveTime();
	}

	@Override
	public void remove() throws PersistenceException, SynchronizationFailedException {
		if(_firstData != null) {
			_firstData = null;
			if(!super.isEmpty()) {
				// Der Datensatz hinter dem ersten könnte gleich hinter dem Intervallende sein
				removeIfInvalid();
			}
			return;
		}
		super.remove();
		removeIfInvalid();
	}

	@Override
	public boolean isEmpty() {
		return _firstData == null && super.isEmpty();
	}

	@NotNull
	@Override
	public ContainerManagementData getContainerManagementData() throws PersistenceException {
		if(isEmpty()) throw new NoSuchElementException();
		if(_firstData != null) {
			return _firstData.getFirstContainerManagementData();
		}
		return super.getContainerManagementData();
	}

	private static class FirstDataSet {
		private final ContainerDataResult _data;
		private final SimpleContainerManagementData _firstContainerManagementData;

		public FirstDataSet(final ContainerDataResult data, final SimpleContainerManagementData firstContainerManagementData) {
			_data = data;
			_firstContainerManagementData = firstContainerManagementData;
		}

		public ContainerDataResult getData() {
			return _data;
		}

		public SimpleContainerManagementData getFirstContainerManagementData() {
			return _firstContainerManagementData;
		}
	}
}
