/*
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */
package de.bsvrz.ars.ars.persistence.layout;

import de.bsvrz.ars.ars.persistence.IdDataIdentification;

import java.nio.file.Path;
import java.util.ArrayDeque;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * Klasse, die ein altes Archivsystem-Layout (Version 5 und früher) abbildet.
 * <p>
 * Das Layout besteht aus tief verschachtelten Verzeichnis-Strukturen,
 * wobei die dezimale ID von Objekt, Attributgruppe und Aspekt jeweils in
 * Dreierpäckchen aufgeteilt wird, z. B. wäre
 * obj123/obj456/obj789/atg246/atg357/asp17 die Ordnerstruktur für
 * das Objekt 123456789, die Attributgruppe 246357 und den Aspekt 17.
 * </p>
 *
 * @author Kappich Systemberatung
 */
public class ClassicPersistenceDirectoryLayout implements PersistenceDirectoryLayout {
	private static final String OBJ_ID_PREFIX = "obj";
	private static final String ATG_ID_PREFIX = "atg";
	private static final String ASP_ID_PREFIX = "asp";
	private static final String SV_PREFIX = "sv";
	/**
	 * RegExp-Pattern für Objekt-Verzeichnisse
	 */
	public static final Pattern OBJ_DIRNAME_PAT = Pattern.compile("^" + OBJ_ID_PREFIX + "(\\d+)$");
	/**
	 * RegExp-Pattern für Attributgruppen-Verzeichnisse
	 */
	private static final Pattern ATG_DIRNAME_PAT = Pattern.compile("^" + ATG_ID_PREFIX + "(\\d+)$");
	/**
	 * RegExp-Pattern für Aspekt-Verzeichnisse
	 */
	private static final Pattern ASP_DIRNAME_PAT = Pattern.compile("^" + ASP_ID_PREFIX + "(\\d+)$");
	/**
	 * RegExp-Pattern für Simulationsvarianten-Verzeichnisse
	 */
	private static final Pattern SV_DIRNAME_PAT = Pattern.compile("^" + SV_PREFIX + "(\\d+)$");
	/**
	 * RegExp-Pattern für beliebige Verzeichnisse (extrahiert die Nummer)
	 */
	private static final Pattern NUM_DIR_PAT = Pattern.compile("^\\w+?(\\d+)$");

	/**
	 * Singleton-Instanz
	 */
	public static final ClassicPersistenceDirectoryLayout Instance = new ClassicPersistenceDirectoryLayout();

	private ClassicPersistenceDirectoryLayout() {
	}

	/**
	 * Liefert den Archiv-Pfad für die angegebene Datenidentifikation und den Wurzelpfad des Archivsystems. Der Pfad wird gebildet, indem die IDs und
	 * die SV in Dreiergruppen zerlegt und mit einem Praefix versehen werden. Pro Dreiergruppe und pro SimVarwird ein Verzeichnis verwendet.
	 *
	 * @param basePath Wurzelpfad des Archivsystems
	 * @param objId    Objekt-ID
	 * @param atgId    Attributgruppen-ID
	 * @param aspId    Aspekt-ID
	 * @return Archiv-Pfad
	 */
	@Override
	public Path getPath(Path basePath, long objId, long atgId, long aspId) {
		Path tmp = basePath;
		tmp = appendIdElements(tmp, OBJ_ID_PREFIX, objId);
		tmp = appendIdElements(tmp, ATG_ID_PREFIX, atgId);
		tmp = appendIdElements(tmp, ASP_ID_PREFIX, aspId);
		return tmp.resolve("sv0");
	}

	/**
	 * Zerlegt die numerische ID in Gruppen von jeweils 3 Zeichen in Dezimaldarstellung und ergänzt das übergebene Path-Objekt, sodass die übergebene
	 * Separator-/Prefixsequenz hinzugefügt wird. Beispiel: mit "obj" im Parameter prefix führt die ID 12345678 dazu, dass der Pfad
	 * "/obj123/obj456/obj78" ergänzt wird.
	 *
	 * @param basePath Basispfad
	 * @param prefix   Prefix, das vor jeder Zeichengruppe eingefügt werden soll.
	 * @param id       Numerische ID.
	 * @return Dateipfad
	 */
	public Path appendIdElements(Path basePath, String prefix, long id) {
		Path tmp = basePath;
		final String idString = Long.toString(id);
		final int end = idString.length();
		int start = 0;
		StringBuilder substringBuilder = new StringBuilder(prefix.length() + 3);
		substringBuilder.append(prefix);
		while (start < end) {
			int newStart = start + 3;
			if (newStart > end) {
				newStart = end;
			}
			substringBuilder.setLength(prefix.length());
			substringBuilder.append(idString, start, newStart);
			tmp = tmp.resolve(substringBuilder.toString());
			start = newStart;
		}
		return tmp;
	}

	/**
	 * Prüft, ob der gegebene Verzeichnisname ein gueltiger Name innerhalb eines Pfades ist, der eine Objekt-ID repraesentiert.
	 *
	 * @param dirName Verzeichnisname
	 * @return Wahr, wenn der Verzeichnisname korrekt ist, falsch sonst.
	 * @see #OBJ_DIRNAME_PAT
	 */
	private boolean isValidObjIdFolderName(String dirName) {
		return OBJ_DIRNAME_PAT.matcher(dirName).matches();
	}

	/**
	 * Prüft, ob der gegebene Verzeichnisname ein gueltiger Name innerhalb eines Pfades ist, der eine Attributgruppen-ID repraesentiert.
	 *
	 * @param dirName Verzeichnisname
	 * @return Wahr, wenn der Verzeichnisname korrekt ist, falsch sonst.
	 * @see #ATG_DIRNAME_PAT
	 */
	private boolean isValidAtgIdFolderName(String dirName) {
		return ATG_DIRNAME_PAT.matcher(dirName).matches();
	}

	/**
	 * Prüft, ob der gegebene Verzeichnisname ein gueltiger Name innerhalb eines Pfades ist, der eine Aspekt-ID repraesentiert.
	 *
	 * @param dirName Verzeichnisname
	 * @return Wahr, wenn der Verzeichnisname korrekt ist, falsch sonst.
	 * @see #ASP_DIRNAME_PAT
	 */
	private boolean isValidAspIdFolderName(String dirName) {
		return ASP_DIRNAME_PAT.matcher(dirName).matches();
	}

	/**
	 * Prüft, ob der gegebene Verzeichnisname ein gueltiger Name für ein Verzeichnis ist, das eine Simulationsvariante repraesentiert.
	 *
	 * @param dirName Verzeichnisname
	 * @return Wahr, wenn der Verzeichnisname korrekt ist, falsch sonst.
	 * @see #SV_DIRNAME_PAT
	 */
	public boolean isValidSvFolderName(String dirName) {
		return SV_DIRNAME_PAT.matcher(dirName).matches();
	}

	/**
	 * Die Methode getIDPart gibt aus einem Ordnernamen den zahlen_Teil zurück (z. B. 123 aus "obj123")
	 *
	 * @param dirName von Typ String
	 * @return String
	 */
	private String getIDPart(String dirName) {
		Matcher m = NUM_DIR_PAT.matcher(dirName);
		if (m.find()) {
			return m.group(1);
		} else {
			throw new IllegalArgumentException(dirName);
		}
	}

	@Override
	public int getMaxDepth() {
		return 22;
	}

	@Override
	public PathType identifyPath(List<String> pathStack) {
		int state = 0;
		int numObj = 0;
		int numAtg = 0;
		int numAsp = 0;
		int numSv = 0;
		for (String s : pathStack) {
			if (state == 0) {
				if (isValidObjIdFolderName(s)) {
					numObj++;
					continue;
				}
				state++;
			}
			if (state == 1) {
				if (isValidAtgIdFolderName(s)) {
					numAtg++;
					continue;
				}
				state++;
			}
			if (state == 2) {
				if (isValidAspIdFolderName(s)) {
					numAsp++;
					continue;
				}
				state++;
			}
			if (numSv == 0 && isValidSvFolderName(s)) {
				numSv++;
				continue;
			}
			return PathType.ForeignPath;
		}
		if (numSv == 0) {
			return PathType.PartialIdPath;
		}
		if (numObj == 0 || numAtg == 0 || numAsp == 0) {
			return PathType.ForeignPath;
		}
		return getDataIdentificationFromPath(pathStack, 0) != null ? PathType.CompleteIdPath : PathType.ForeignPath;
	}

	@Override
	public IdDataIdentification getDataIdentificationFromPath(List<String> pathStack, int simulationVariant) {
		try {
			ArrayDeque<String> tmp = new ArrayDeque<>(pathStack);
			String fileName = tmp.pollLast();
			int sv;
			if (isValidSvFolderName(fileName)) {
				sv = Integer.parseInt(getIDPart(fileName));
			} else {
				return null;
			}

			fileName = tmp.pollLast();
			StringBuilder aspString = new StringBuilder();
			StringBuilder atgString = new StringBuilder();
			StringBuilder objString = new StringBuilder();
			do {
				if (fileName == null) {
					return null;
				}
				if (isValidAspIdFolderName(fileName)) {
					aspString.insert(0, getIDPart(fileName));
				} else {
					break;
				}
			}
			while ((fileName = tmp.pollLast()) != null);
			do {
				if (fileName == null) {
					return null;
				}
				if (isValidAtgIdFolderName(fileName)) {
					atgString.insert(0, getIDPart(fileName));
				} else {
					break;
				}
			}
			while ((fileName = tmp.pollLast()) != null);
			do {
				if (fileName == null) {
					return null;
				}
				if (isValidObjIdFolderName(fileName)) {
					objString.insert(0, getIDPart(fileName));
				} else {
					break;
				}
			}
			while ((fileName = tmp.pollLast()) != null);

			return new IdDataIdentification(Long.parseLong(objString.toString()), Long.parseLong(atgString.toString()),
					Long.parseLong(aspString.toString()), sv);
		} catch (NumberFormatException ignored) {
		}
		return null;
	}

	/**
	 * Erzeugt eine {@link PersistenceDirectoryLayoutInstance} basierend auf diesem Layout
	 *
	 * @param basePath          Basisverzeichnis, in dem das Layout Unterverzeichnisse erstellen soll.
	 * @param simulationVariant Simulationsvariante
	 * @return Eine PersistenceDirectoryLayoutInstance
	 */
	public static PersistenceDirectoryLayoutInstance instance(Path basePath, int simulationVariant) {
		return Instance.createInstance(basePath, simulationVariant);
	}
}
