/*
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */
package de.bsvrz.ars.ars.persistence.layout;

import de.bsvrz.ars.ars.persistence.IdDataIdentification;

import java.nio.file.Path;
import java.util.List;

/**
 * Dieses Interface wird benutzt von Implementierungen von Persistenzverzeichnislayouts, also Klassen, die die Struktur
 * eines Persistenzverzeichnisses definieren.
 *
 * @author Kappich Systemberatung
 */
public interface PersistenceDirectoryLayout {

	/**
	 * Liefert den Archiv-Pfad für die angegebene Datenidentifikation und den Wurzelpfad des Archivsystems.
	 *
	 * @param basePath Wurzelverzeichnis
	 * @param objId    Objekt-ID
	 * @param atgId    Attributgruppen-ID
	 * @param aspId    Aspekt-ID
	 * @return Archiv-Pfad
	 */
	Path getPath(Path basePath, long objId, long atgId, long aspId);

	/**
	 * Liefert den Archiv-Pfad für die angegebene Datenidentifikation.
	 *
	 * @param basePath           Wurzelverzeichnis
	 * @param dataIdentification objID, atgID, aspID; sv wird ignoriert
	 * @return Archiv-Pfad
	 */
	default Path getPath(Path basePath, IdDataIdentification dataIdentification) {
		return getPath(basePath, dataIdentification.getObjectId(), dataIdentification.getAtgId(), dataIdentification.getAspectId());
	}

	/**
	 * Gibt die maximale Tiefe zurück, in der datenidentifikationsverzeichnisse zu erwarten sind
	 *
	 * @return maximale Tiefe im Dateisystem-Baum
	 */
	int getMaxDepth();

	/**
	 * Identifiziert den Typ des angegebenen Verzeichnisses
	 * @param pathStack Stack wo man sich im Verzeichnisbaum befindet
	 * @return Typ des angegebenen Pfads   
	 */
	PathType identifyPath(List<String> pathStack);

	/**
	 * Identifiziert die Datenidentifiaktion des angegebenen Verzeichnisses
	 * @param pathStack Stack wo man sich im Verzeichnisbaum befindet
	 * @param simulationVariant Simulatiosnvariante
	 * @return Typ des angegebenen Pfads   
	 */
	IdDataIdentification getDataIdentificationFromPath(List<String> pathStack, int simulationVariant);

	/**
	 * Erzeugt eine {@link PersistenceDirectoryLayoutInstance} basierend auf diesem Layout
	 *
	 * @param basePath          Basisverzeichnis, in dem das Layout Unterverzeichnisse erstellen soll.
	 * @param simulationVariant Simulationsvariante
	 * @return Eine PersistenceDirectoryLayoutInstance
	 */
	default PersistenceDirectoryLayoutInstance createInstance(Path basePath, int simulationVariant) {
		return new PersistenceDirectoryLayoutInstance(this, basePath, simulationVariant);
	}
}
