/*
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */
package de.bsvrz.ars.ars.persistence.util;

import de.bsvrz.sys.funclib.kappich.annotations.NotNull;

import java.time.Duration;
import java.util.ArrayDeque;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.ToLongFunction;

/**
 * Eine Klasse ähnlich einer {@link java.util.concurrent.BlockingQueue} ohne Kapazitätsgrenze,
 * die es ermöglicht, darauf zu warten, dass die Queue leer wird.
 * <p>
 * Außerdem ermöglicht die Queue mit {@link #terminate()} bzw. {@link #terminateNow()} das Terminieren
 * dieser Queue. Alle Daten, die nach dem
 * Terminierungsignal eingefügt werden, werden ignoriert und der Versuch, weitere Daten mit {@link #take()}
 * abzuholen, führt dann zu einer {@link TerminatedException}. Dies entspricht dem Einfügen eines speziellen Objekts
 * in die Queue, allerdings ist es nicht möglich, eine Queue mehrfach zu terminieren.
 *
 * @param <E>
 */
public class SignalingQueue<E> {

	private final ArrayDeque<E> delegate;

	private final ReentrantLock lock = new ReentrantLock();

	private final Condition canTake = lock.newCondition();

	private final Condition empty = lock.newCondition();

	private boolean terminated = false;

	private final AtomicInteger capacityEstimate = new AtomicInteger();

	/**
	 * Erstellt eine neue SignalingQueue.
	 */
	public SignalingQueue() {
		delegate = new ArrayDeque<>();
	}

	/**
	 * Fügt ein Element zur Queue hinzu. Da die Größe der Queue unbegrenzt ist, ist dies immer erfolgreich,
	 * außer die Queue wurde bereits terminiert, in dem Fall wird das Element ignoriert.
	 *
	 * @param element Element
	 */
	public void add(@NotNull E element) {
		lock.lock();
		try {
			if (terminated) {
				return;
			}
			boolean wasEmpty = delegate.isEmpty();
			delegate.add(element);
			updateCapacityEstimate(delegate.size());
			if (wasEmpty) {
				canTake.signalAll();
			}
		} finally {
			lock.unlock();
		}
	}

	/**
	 * Aktualisiert die Abschätzung der internen ArrayDeque-Kapazität
	 *
	 * @param size Anzahl Elemente
	 */
	private void updateCapacityEstimate(int size) {
		int capacity = size + size / 2;
		if (capacity < size) {
			// Overflow sanity check
			capacity = size;
		}
		capacityEstimate.accumulateAndGet(capacity, Math::max);
	}

	/**
	 * Holt ein Element aus der Queue und wartet dafür, falls nötig, solange, bis ein Element mit {@link #add(Object)}
	 * hinzugefügt wurde, oder bis die Queue mit {@link #terminate()} terminiert wurde.
	 *
	 * @return Element
	 * @throws InterruptedException Beim Warten auf Elemente unterbrochen
	 * @throws TerminatedException  Es gibt kein Element mehr und die Queue wurde terminiert, d. h. das letzte Element wurde
	 *                              bereits im vorherigen Schritt geholt
	 */
	public E take() throws InterruptedException, TerminatedException {
		lock.lock();
		try {
			while (delegate.isEmpty() && !terminated) {
				canTake.await();
			}
			if (delegate.isEmpty()) {
				throw new TerminatedException();
			}
			E result = delegate.pop();
			if (delegate.isEmpty()) {
				empty.signalAll();
			}
			return result;
		} finally {
			lock.unlock();
		}
	}

	/**
	 * Holt ein Element aus der Queue und wartet ggf. solange bis eines mit {@link #add(Object)}
	 * hinzugefügt wurde, oder bis die Queue mit {@link #terminate()} terminiert wurde oder bis der angegebene
	 * Timeout abgelaufen ist.
	 *
	 * @param timeout Timeout
	 * @return Element oder null (nur bei Timeout-Ablauf)
	 * @throws InterruptedException Beim Warten auf Elemente unterbrochen
	 * @throws TerminatedException  Es gibt kein Element mehr und die Queue wurde terminiert, d. h. das letzte Element wurde
	 *                              bereits im vorherigen Schritt geholt
	 */
	public E take(Duration timeout) throws InterruptedException, TerminatedException {
		long nanos = timeout.toNanos();
		lock.lockInterruptibly();
		try {
			while (delegate.isEmpty() && !terminated) {
				if (nanos <= 0L)
					return null;
				nanos = canTake.awaitNanos(nanos);
			}
			if (delegate.isEmpty()) {
				throw new TerminatedException();
			}
			E result = delegate.pop();
			if (delegate.isEmpty()) {
				empty.signalAll();
			}
			return result;
		} finally {
			lock.unlock();
		}
	}

	/**
	 * Wartet, bis die Queue leer ist
	 *
	 * @param timeout Timeout
	 * @return true: Queue hat sich innerhalb des timeouts geleert, false: timout ist abgelaufen, bevor Queue leer ist
	 * @throws InterruptedException Unterbrochen beim Warten
	 */
	public boolean waitUntilEmpty(Duration timeout) throws InterruptedException {
		long nanos = timeout.toNanos();
		lock.lock();
		try {
			while (!delegate.isEmpty()) {
				if (nanos <= 0L)
					return false;
				nanos = empty.awaitNanos(nanos);
			}
			return true;
		} finally {
			lock.unlock();
		}
	}

	/**
	 * Gibt die Anzahl der Elemente zurück. Das Signal zur Terminierung wird nicht mitgezählt.
	 *
	 * @return die Anzahl der Elemente
	 */
	public int size() {
		lock.lock();
		try {
			return delegate.size();
		} finally {
			lock.unlock();
		}
	}

	/**
	 * Gibt zurück, ob die Queue leer ist. Das Signal zur Terminierung wird nicht mitgezählt.
	 *
	 * @return die Anzahl der Elemente
	 */
	public boolean isEmpty() {
		lock.lock();
		try {
			return delegate.isEmpty();
		} finally {
			lock.unlock();
		}
	}

	/**
	 * Löscht alle Elemente
	 */
	public void clear() {
		lock.lock();
		try {
			delegate.clear();
		} finally {
			lock.unlock();
		}
	}

	/**
	 * Löscht alle Elemente und signalisiert, dass die Queue terminiert wurde,
	 * d. h. der nächste take()-Aufruf wirft eine {@link TerminatedException}.
	 */
	public void terminateNow() {
		lock.lock();
		try {
			terminated = true;
			delegate.clear();
			canTake.signalAll();
		} finally {
			lock.unlock();
		}
	}

	/**
	 * Signalisiert, dass die Queue terminiert wurde, d. h. das Einfügen von weiteren Elementen
	 * ist nicht mehr möglich und nach dem Holen des letzten Elements wird bei weiteren Aufrufen von
	 * take()-Aufruf eine {@link TerminatedException} geworfen.
	 */
	public void terminate() {
		lock.lock();
		try {
			terminated = true;
			canTake.signalAll();
		} finally {
			lock.unlock();
		}
	}

	/**
	 * Berechnet eine Summe über alle Elemente, z. B. zur Abschätzung des Gesamt-Speicherverbrauchs
	 *
	 * @param extractor Extraktionsfunktion, die jedes Element in eine Zahl umwandelt, welche dann aufaddiert wird
	 * @return Summe über alle Elemente
	 */
	public long sumElements(ToLongFunction<? super E> extractor) {
		lock.lock();
		try {
			return delegate.stream().mapToLong(extractor).sum();
		} finally {
			lock.unlock();
		}
	}

	/**
	 * Gibt eine Abschätzung der Arraygröße der intern verwendeten Queue zurück
	 *
	 * @return Geschätzte Gesamtanzahl der Elemente im Array
	 */
	public long capacityEstimate() {
		return capacityEstimate.get();
	}
}
