/*
 *
 * Copyright 2017-2020 by Kappich Systemberatung, Aachen
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.ars.ars.persistence.walk;

import de.bsvrz.ars.ars.mgmt.tasks.base.TaskProgressInterface;
import de.bsvrz.ars.ars.persistence.IdDataIdentification;
import de.bsvrz.ars.ars.persistence.PersistenceException;
import de.bsvrz.ars.ars.persistence.directories.PersistenceDirectory;
import de.bsvrz.ars.ars.persistence.walk.internal.DataIdentificationDirGetters;
import de.bsvrz.ars.ars.persistence.walk.internal.StandardDataidentificationDirWalk;
import de.bsvrz.ars.ars.persistence.walk.internal.StatusPrinter;
import de.bsvrz.dav.daf.main.archive.ArchiveDataSpecification;
import de.bsvrz.sys.funclib.kappich.annotations.NotNull;

import java.time.Duration;
import java.util.Collection;

/**
 * Interface für die Durchführung für Aktionen, die Containerdatei-Verzeichnisse in beliebiger Reihenfolge besuchen.
 * <p>
 * Die Arbeit kann dabei auf mehrere Threads verteilt werden.
 *
 * @author Kappich Systemberatung
 */
public interface DataIdentificationDirWalk {

	/**
	 * Factory-Methode, die einen DataIdentificationDirWalk erzeugt, der alle im Dateisystem vorhandenen Datenidentifikationsverzeichnisse durchsucht.
	 * @param persistenceDirectory PersistenceManager
	 * @return Ein Objekt, das die Aktion starten kann
	 * @see #execute(String, int, DataIdentificationDirAction)
	 */
	static DataIdentificationDirWalk allDirectories(@NotNull PersistenceDirectory persistenceDirectory) {
		return new StandardDataidentificationDirWalk(DataIdentificationDirGetters.all(persistenceDirectory));
	}

	/**
	 * Factory-Methode, die einen DataIdentificationDirWalk erzeugt, der die in der angegebenen Anfrage enthaltenen Archivdaten-Spezifikationen durchsucht.
	 * @param persistenceDirectory PersistenceManager
	 * @param archiveDataSpecifications Anfrage
	 * @return Ein Objekt, das die Aktion starten kann
	 * @see #execute(String, int, DataIdentificationDirAction)
	 */
	static DataIdentificationDirWalk create(PersistenceDirectory persistenceDirectory, Collection<? extends ArchiveDataSpecification> archiveDataSpecifications) {
		return new StandardDataidentificationDirWalk(DataIdentificationDirGetters.selectedContainerDirs(persistenceDirectory, archiveDataSpecifications));
	}

	/**
	 * Factory-Methode, die einen DataIdentificationDirWalk erzeugt, der die in der angegebenen Anfrage enthaltenen Archivdaten-Spezifikationen durchsucht.
	 * @param persistenceDirectory PersistenceManager
	 * @param archiveDataSpecifications Anfrage
	 * @return Ein Objekt, das die Aktion starten kann
	 * @see #execute(String, int, DataIdentificationDirAction)
	 */
	static DataIdentificationDirWalk dataIdentifications(PersistenceDirectory persistenceDirectory, Collection<? extends IdDataIdentification> archiveDataSpecifications) {
		return new StandardDataidentificationDirWalk(DataIdentificationDirGetters.selectedDataIdentifications(persistenceDirectory, archiveDataSpecifications));
	}

	/**
	 * Gibt das Intervall zurück, in dem Statusmeldungen ausgegeben werden (positiv)
	 * @return das Intervall
	 */
	Duration getStatusInterval();

	/**
	 * Setzt das Statusintervall
	 * @param statusInterval Intervall
	 */
	void setStatusInterval(Duration statusInterval);

	/**
	 * Gibt die Klasse zurück, die Statusmeldungen ausgibt.
	 * @return Klasse
	 */
	StatusPrinter getStatusPrinter();

	/**
	 * Setzt die Klasse, die Statusmeldungen schreibt.
	 * @param statusPrinter Beliebige Subklasse/Instanz von {@link StatusPrinter}
	 */
	void setStatusPrinter(StatusPrinter statusPrinter);

	TaskProgressInterface getPublisherTask();

	void setPublisherTask(TaskProgressInterface publisherTask);

	/**
	 * Führt den Verzeichnisdurchlauf aus. Jedes ContainerDirWalk-Objekt kann nur einmal benutzt werden.
	 *
	 * @param actionName Name (zur Benennung der Thread-Objekte, Ausgabe in Debug-Meldungen usw.)
	 * @param numThreads Anzahl Threads zur Bearbeitung
	 * @param action Durchzuführende Aktion
	 * @throws PersistenceException Falls ein Fehler Auftritt
	 */
	void execute(String actionName, int numThreads, DataIdentificationDirAction action) throws PersistenceException;

	/**
	 * Sorgt für ein sofortiges, aber kontrolliertes Beenden. Der aktuelle Verzeichnisdurchlauf je Thread wird abgeschlossen.
	 */
	void terminate();

	/**
	 * Inkrementiert die Anzahl besuchter Container (interne Methode, die von {@link ContainerWalk} aufgerufen wird, damit diese Klasse die Anzahl Container für die Ausgabe mit-zählen kann)
	 * @param increment Anzahl, um die inkrementiert werden soll
	 */
	void increaseContCount(int increment);
}
