/*
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */
package de.bsvrz.ars.ars.persistence.walk.internal;

import java.util.function.Consumer;

/**
 * Instanzen dieser Klasse dienen dem {@link StandardDataidentificationDirWalk} per Dependency-Injection als Quelle
 * von Datenidentifikationsverzeichnissen.
 * <p>
 * Durch die Dependency-Injection sind die Typen in dieser Klasse etwas verwirrend.
 * Eine genaue Beschreibung findet sich in der {@link #create(Consumer, CountEstimator)}-Factory-Methode.
 * <p>
 * Hintergrund dieser Klasse und der ähnlichen Struktur ist, dass das Auflisten von datenidentifikationsverzeichnissen
 * am effizientesten "aktiv" passiert, d. h. der {@link de.bsvrz.ars.ars.persistence.walk.DataIdentificationDirWalk}
 * hat keine {@link java.util.Iterator}-mäßigen Abfragemethoden, sondern liefert laufend neue Verzeichnisse
 * und übergibt diese als Ereignis an einen {@link de.bsvrz.ars.ars.persistence.layout.DataIdentificationListener}.
 * <p>
 * Diese Klasse erlaubt über Dependency Injection eine Entkoppelung von dieser aktiven Struktur und zudem eine
 * Abschätzung des Datenumfangs mit einem {@link CountEstimator}.
 */
public final class DataIdentificationDirGetter {

	private final Consumer<DataIdentificationDirListener> directoryGetter;
	
	private final CountEstimator countEstimator;

	/**
	 * Privater Konstruktor. Beschreibung der Parameter siehe {@link #create(Consumer, CountEstimator)}.
	 *
	 * @param directoryGetter Klasse die einen {@link de.bsvrz.ars.ars.persistence.layout.DataIdentificationListener}
	 *                        akzeptiert und diesem die gefundenen Containerverzeichnisse übergibt.
	 * @param countEstimator  Klasse, die die Anzahl der Datenidentifikationsverzeichnisse schätzt.
	 */
	private DataIdentificationDirGetter(Consumer<DataIdentificationDirListener> directoryGetter, CountEstimator countEstimator) {
		this.directoryGetter = directoryGetter;
		this.countEstimator = countEstimator;
	}

	/**
	 * Erstellt einen DataIdentificationDirGetter.
	 *
	 * @param directoryGetter Klasse die einen {@link de.bsvrz.ars.ars.persistence.layout.DataIdentificationListener}
	 *                        akzeptiert und diesem die gefundenen Containerverzeichnisse übergibt. Das heißt, dieser
	 *                        Parameter ist eine Implementierung, die Datenidentifikationsverzeichnisse liefert,
	 *                        z. B. durch Iteration der Verzeichnisstruktur.
	 * @param countEstimator  Klasse, die die gesamte Anzahl der Datenidentifikationsverzeichnisse schätzt.
	 * @return Eine neue Instanz
	 */
	public static DataIdentificationDirGetter create(Consumer<DataIdentificationDirListener> directoryGetter, CountEstimator countEstimator) {
		return new DataIdentificationDirGetter(directoryGetter, countEstimator);
	}

	/**
	 * Führt den Verzeichnisdurchlauf aus. Diese Methode startet den Verzeichnisdurchlauf, indem
	 * der beim Erstellen der Instanz angegebene directoryGetter gestartet wird und an ihn der Listener übergeben wird.
	 * Der Listener, also der Parameter dieser Methode, bekommt also dann alle gefundenen 
	 * Datenidentifikationsverzeichnisse nacheinander übergeben.
	 * @param listener Listener, der Datenidentifikationsverzeichnisse empfängt
	 */
	public void run(DataIdentificationDirListener listener) {
		directoryGetter.accept(listener);
	}

	/**
	 * Schätz mithilfe des übergebenen {@link CountEstimator}-Objekts die Gesamtzahl aller Datenidentifikationen
	 * @return u. U. nicht verlässlicher Schätzwert
	 */
	public long estimateCount() {
		return countEstimator.estimate();
	}

	/**
	 * Setzt die tatsächliche Anzahl an Datenidentifikationsverzeichnissen nach einem erfolgreichen Durchlauf.
	 * Der {@link CountEstimator} kann damit für die Zukunft die Schätzung aktualisieren,
	 * ist aber nicht verpflichtet den Wert zu nutzen,
	 * wenn er bereits eine bessere Quelle für die Schätzung hat.
	 * <p>
	 * Diese Methode wird nur intern nach einem erfolgreichen Durchlauf aufgerufen.
	 *
	 * @param actualCount Tatsächliche Anzahl an Datenidentifikationsverzeichnissen
	 */
	void setActualCount(long actualCount) {
		countEstimator.setCount(actualCount);
	}
}
