/*
 *
 * Copyright 2017-2019 by Kappich Systemberatung, Aachen
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.ars.ars.persistence.walk.internal;

import com.google.common.collect.LinkedHashMultimap;
import com.google.common.collect.Multimap;
import de.bsvrz.ars.ars.persistence.DataIdentificationDir;
import de.bsvrz.ars.ars.persistence.IdDataIdentification;
import de.bsvrz.ars.ars.persistence.directories.PersistenceDirectory;
import de.bsvrz.ars.ars.persistence.layout.AbortWalkException;
import de.bsvrz.ars.ars.persistence.layout.DataIdentificationListener;
import de.bsvrz.ars.ars.persistence.layout.DirectoryLayoutWalk;
import de.bsvrz.dav.daf.main.archive.ArchiveDataSpecification;

import java.io.IOException;
import java.nio.file.Path;
import java.util.Collection;
import java.util.Map;

/**
 * Interface für Klassen, die eine Menge von Datenidentifikations-Verzeichnissen nacheinander zurückgeben.
 *
 * @author Kappich Systemberatung
 */
public class DataIdentificationDirGetters {

	private DataIdentificationDirGetters() {
	}


	/**
	 * Erstellt eine Instanz, die über von {@link ArchiveDataSpecification}-Objekten spezifizierte
	 * Datenidentifikationen iteriert
	 *
	 * @param persistenceDirectory Persistenz
	 * @param ads                  Archivdatenspezifikationen, über die iteriert werden soll
	 * @return DataIdentificationDirGetter
	 */
	public static DataIdentificationDirGetter selectedContainerDirs(final PersistenceDirectory persistenceDirectory, final Collection<? extends ArchiveDataSpecification> ads) {
		Multimap<IdDataIdentification, ArchiveDataSpecification> tmpMap = LinkedHashMultimap.create();
		for (ArchiveDataSpecification archiveDataSpecification : ads) {
			tmpMap.put(new IdDataIdentification(archiveDataSpecification), archiveDataSpecification);
		}
		return DataIdentificationDirGetter.create(listener -> {
					try {
						for (Map.Entry<IdDataIdentification, Collection<ArchiveDataSpecification>> entry : tmpMap.asMap().entrySet()) {
							listener.foundDataIdentification(new DataIdentificationDir(persistenceDirectory, entry.getKey(), entry.getValue()));
						}
					} catch (AbortWalkException ignored) {
					}
				}, CountEstimator.fixed(tmpMap.size())
		);
	}

	/**
	 * Erstellt eine Instanz, die über ausgewählte Datenidentifikationen in einem Persistenzverzeichnis iteriert
	 *
	 * @param persistenceDirectory Persistenzverzeichnis
	 * @param dataIdentifications  Collection mit Containerverzeichnissen, über die iteriert werden soll
	 * @return DataIdentificationDirGetter
	 */
	public static DataIdentificationDirGetter selectedDataIdentifications(final PersistenceDirectory persistenceDirectory, final Collection<? extends IdDataIdentification> dataIdentifications) {
		return DataIdentificationDirGetter.create(listener -> {
			try {
				for (IdDataIdentification dataIdentification : dataIdentifications) {
					listener.foundDataIdentification(new DataIdentificationDir(persistenceDirectory, dataIdentification));
				}
			} catch (AbortWalkException ignored) {
			}
		}, CountEstimator.fixed(dataIdentifications.size()));
	}

	/**
	 * Erstellt eine neue DataIdentificationDirGetter-Instanz, die über alle Datenidentifikationen eines
	 * Persistenzverzeichnisses iteriert.
	 *
	 * @param persistenceDirectory Persistenzverzeichnis
	 * @return DataIdentificationDirGetter
	 */
	public static DataIdentificationDirGetter all(final PersistenceDirectory persistenceDirectory) {
		return DataIdentificationDirGetter.create(listener ->
				DirectoryLayoutWalk.walk(persistenceDirectory.getLayoutInstance(), new DataIdentificationListener() {
					@Override
					public void foundDataIdentification(IdDataIdentification dataIdentification, Path dir) throws AbortWalkException {
						listener.foundDataIdentification(new DataIdentificationDir(persistenceDirectory, dataIdentification, dir));
					}

					@Override
					public void foundForeignDirectory(Path dir) {

					}

					@Override
					public void foundForeignFile(Path file) {

					}

					@Override
					public void ioException(Path file, IOException exception) {

					}
				}), persistenceDirectory);
	}

	/**
	 * Erstellt eine neue DataIdentificationDirGetter-Instanz, die alle Verzeichnisse durchsucht und gefundene
	 * Verzeichnisse (inkl. fremde Dateien und Verzeichnisse) vor der normalen Iteration
	 * an den delegate weiterleitet, z. B. zum Logging oder bei einem Analyselauf.
	 *
	 * @param persistenceDirectory Persistenzverzeichnis
	 * @param delegate             Delegate-Listener
	 * @return DataIdentificationDirGetter
	 */
	public static DataIdentificationDirGetter allWithDelegate(final PersistenceDirectory persistenceDirectory, DataIdentificationListener delegate) {
		return DataIdentificationDirGetter.create(listener ->
				DirectoryLayoutWalk.walk(persistenceDirectory.getLayoutInstance(), new DataIdentificationListener() {
					@Override
					public void foundDataIdentification(IdDataIdentification dataIdentification, Path dir) throws AbortWalkException {
						delegate.foundDataIdentification(dataIdentification, dir);
						DataIdentificationDir dataIdentificationDir = new DataIdentificationDir(persistenceDirectory, dataIdentification, dir);
						listener.foundDataIdentification(dataIdentificationDir);
					}

					@Override
					public void foundForeignDirectory(Path dir) throws AbortWalkException {
						delegate.foundForeignDirectory(dir);
					}

					@Override
					public void foundForeignFile(Path file) throws AbortWalkException {
						delegate.foundForeignFile(file);
					}

					@Override
					public void ioException(Path file, IOException exception) throws AbortWalkException {
						delegate.ioException(file, exception);
					}
				}), persistenceDirectory);
	}

}
