/*
 *
 * Copyright 2017-2020 by Kappich Systemberatung, Aachen
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.ars.ars.persistence.walk.internal;

import java.text.NumberFormat;
import java.time.Duration;


/**
 * Klasse, die den Status einer ContainerDirWalk-Aktion ausgibt. Kann überschrieben werden, um einen angepassten/erweiterten Status zu veröffentlichen.
 *
 * @author Kappich Systemberatung
 */
public class StatusPrinter {

	public enum ApproximationType {
		Minimum,
		Approximation,
		Exact
	}
	
	/**
	 * Gibt die auszugebende Statusmeldung zurück. Kann überschrieben werden, um weitere Informationen hinzuzufügen.
	 * @param actionName Name der Aktion / des Tasks
	 * @param runtime Bisherige Laufzeit
	 * @param approximationType Sind bereits alle relevanten Dateien indiziert?
	 * @param allContainerDirs (Geschätzte) Anzahl aller Containerdateien-Ordner
	 * @param visitedContainerDirs Anzahl der bereits besuchten Containerdateien-Ordner
	 * @param visitedContainerFiles Anzahl der bereits besuchten Containerdateien
	 * @return Nachricht
	 */
	public String getStatusMessage(final String actionName, final Duration runtime, final ApproximationType approximationType, final long allContainerDirs, final long visitedContainerDirs, final long visitedContainerFiles) {
		if (approximationType == ApproximationType.Minimum) {
			return actionName + " läuft.\nBisherige Laufzeit: " + formatDuration(runtime) + "...\n"
					+ visitedContainerDirs + " von mindestens " + allContainerDirs + " Container-Verzeichnissen besucht.\n"
					+ estimateMinRuntime(runtime, visitedContainerDirs, allContainerDirs) + "..."
					+ (visitedContainerFiles == 0 ? "" : "\n" + visitedContainerFiles + " Container-Dateien besucht.");
		} else if (approximationType == ApproximationType.Approximation) {
			return actionName + " läuft.\nBisherige Laufzeit: " + formatDuration(runtime) + "...\n"
					+ visitedContainerDirs + " von ungefähr " + allContainerDirs + " Container-Verzeichnissen besucht.\n"
					+ estimateRuntime(runtime, visitedContainerDirs, allContainerDirs) + "..."
					+ (visitedContainerFiles == 0 ? "" :"\n" + visitedContainerFiles + " Container-Dateien besucht.");
		}
		else {
			return actionName + " läuft.\nBisherige Laufzeit: " + formatDuration(runtime) + "...\n"
					+ visitedContainerDirs + " von " + allContainerDirs + " Container-Verzeichnissen besucht.\n"
					+ estimateRuntime(runtime, visitedContainerDirs, allContainerDirs) + "..."
					+ (visitedContainerFiles == 0 ? "" :"\n" + visitedContainerFiles + " Container-Dateien besucht.");

		}
	}

	/**
	 * Formatiert die Ausgabe für die Fertigmeldung.
	 *
	 * @param actionName            Name der Aktion
	 * @param runtime               Bisherige Laufzeit
	 * @param visitedContainerDirs  Anzahl besuchter Containerdateien-Verzeichnisse
	 * @param visitedContainerFiles Anzahl besuchte Containerdateien
	 * @return Textuelle Ausgabe
	 */
	public String getSuccessMessage(final String actionName, final Duration runtime, final long visitedContainerDirs, final long visitedContainerFiles) {
		return actionName + " beendet.\nLaufzeit: " + formatDuration(runtime) + ".\n" + visitedContainerDirs + " Container-Verzeichnisse besucht." +
				(visitedContainerFiles == 0 ? "" : "\n" + visitedContainerFiles + " Container-Dateien besucht.");

	}

	/**
	 * Formatiert die Ausgabe für die Fertigmeldung nach Fehler.
	 *
	 * @param actionName            Name der Aktion
	 * @param runtime               Bisherige Laufzeit
	 * @param visitedContainerDirs  Anzahl besuchter Containerdateien-Verzeichnisse
	 * @param visitedContainerFiles Anzahl besuchte Containerdateien
	 * @return Textuelle Ausgabe
	 */
	public String getErrorMessage(final String actionName, final Duration runtime, final long visitedContainerDirs, final long visitedContainerFiles) {
		return actionName + " nach Fehler abgebrochen.\nLaufzeit: " + formatDuration(runtime) + ".\n" + visitedContainerDirs + " Container-Verzeichnisse besucht." +
				(visitedContainerFiles == 0 ? "" : "\n" + visitedContainerFiles + " Container-Dateien besucht.");

	}

	/**
	 * Berechnet den Fortschritt und formatiert eine Ausgabe.
	 * @param runtime Bisherige Laufzeit
	 * @param visited Anzahl besuchter Objekte
	 * @param all Anzahl aller Objekte
	 * @return Textuelle Ausgabe
	 */
	public static String estimateRuntime(final Duration runtime, final long visited, final long all) {
		double finishedRatio = visited / (double)all;
		long currentRuntimeSeconds = runtime.getSeconds();
		double estimatedCompleteTime = currentRuntimeSeconds / finishedRatio;
		if (visited == 0 || all == 0 || estimatedCompleteTime > Duration.ofDays(1000).getSeconds()) {
			// Überlauf vermeiden
			return "Fortschritt: Unbekannt";
		}
		long remainingSeconds = (long) (estimatedCompleteTime - currentRuntimeSeconds);
		Duration remainingDuration = Duration.ofSeconds(remainingSeconds);
		String duration = formatDuration(remainingDuration);
		return "Fortschritt: " + NumberFormat.getPercentInstance().format(finishedRatio) + ", noch ca. " + duration;
	}

	/**
	 * Berechnet den Fortschritt und formatiert eine Ausgabe, wenn die Gesamtzahl der Objekte unbekannt ist.
	 * @param runtime Bisherige Laufzeit
	 * @param visited Anzahl besuchter Objekte
	 * @param indexed Anzahl bisher gefundener Objekte
	 * @return Textuelle Ausgabe
	 */
	public static String estimateMinRuntime(final Duration runtime, final long visited, final long indexed) {
		double maxFinishedRatio = visited / (double)indexed;
		long currentRuntimeSeconds = runtime.getSeconds();
		double estimatedMinCompleteTime = currentRuntimeSeconds / maxFinishedRatio;
		long remainingMinSeconds = (long) (estimatedMinCompleteTime - currentRuntimeSeconds);
		Duration remainingDuration = Duration.ofSeconds(remainingMinSeconds);
		if(remainingDuration.toMinutes() < 1) {
			return "Fortschritt: unbekannt";
		}
		String minDuration = formatDuration(remainingDuration);
		return "Fortschritt: unbekannt, geschätzte Laufzeit: noch mindestens " + minDuration;
	}

	/**
	 * Formatiert eine Zeitdauer.
	 * @param runtime Dauer
	 * @return Formatierte Ausgabe
	 */
	public static String formatDuration(final Duration runtime) {
		long days = runtime.toDays();
		long hours = runtime.toHours() % 24;
		long minutes = runtime.toMinutes() % 60;
		long seconds = runtime.getSeconds() % 60;
		if(days == 0) return String.format("%02d:%02d:%02d", hours, minutes, seconds);
		if(days == 1) return String.format("1 Tag, %02d:%02d:%02d", hours, minutes, seconds);

		return String.format("%d Tage, %02d:%02d:%02d", days, hours, minutes, seconds);
	}
}
