/*
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */
package de.bsvrz.ars.ars.persistence.writer;

import de.bsvrz.ars.ars.mgmt.datatree.DataIdentNode;
import de.bsvrz.ars.ars.persistence.*;
import de.bsvrz.ars.ars.persistence.directories.ActivePersistenceDirectory;
import de.bsvrz.dav.daf.main.DataState;
import de.bsvrz.dav.daf.main.archive.ArchiveDataKind;
import de.bsvrz.dav.daf.main.config.DataModel;
import de.bsvrz.sys.funclib.debug.Debug;
import de.bsvrz.sys.funclib.kappich.annotations.NotNull;
import de.bsvrz.sys.funclib.losb.util.Util;
import de.bsvrz.sys.funclib.operatingMessage.MessageGrade;
import de.bsvrz.sys.funclib.operatingMessage.MessageSender;
import de.bsvrz.sys.funclib.operatingMessage.MessageState;
import de.bsvrz.sys.funclib.operatingMessage.MessageType;

import java.util.Objects;

/**
 * Ein Auftrag, einen Onlinedatensatz zu archivieren
 *
 * @param dataset            Online-Datensatz
 * @param dataIdentification Datenidentifikation
 * @param delayed            Nachgeliefert?
 */
public record ArchiveOnlineData(SerializableDataset dataset, IdDataIdentification dataIdentification,
                                boolean delayed) implements ArchiveJob {

	/**
	 * Kennzeichen der Betriebsmeldung
	 */
	public static final String MSG_PID_DATAINDEX_ANOMALY = "Datenindex nicht monoton";

	private static final Debug _debug = Debug.getLogger();

	public void accept(ArchiveTask archiveTask) {
		archiveData(archiveTask, true);
	}

	private void archiveData(ArchiveTask archiveTask, boolean allowDataGap) {
		try {
			var archTime = dataset.archiveTime();
			var dataTime = dataset.dataTime();
			var dataIdx = dataset.dataIndex();
			var adk = delayed ? ArchiveDataKind.ONLINE_DELAYED : ArchiveDataKind.ONLINE;
			var din = archiveTask.getDidTree().get(dataIdentification);

			IdContainerFileDir idContainerFileDirOnline = dataIdentification.resolve(ArchiveDataKind.ONLINE);
			IdContainerFileDir idContainerFileDirDelayed = dataIdentification.resolve(ArchiveDataKind.ONLINE_DELAYED);

			PersistenceManager persistenceManager = archiveTask.getPersistenceManager();
			ActivePersistenceDirectory persistenceDirectory = persistenceManager.updateAndGetActivePersistenceDirectory(archTime, dataIdentification.getSimVariant());

			// Letzte Datenindexe, Datenzeiten nachsehen
			StandardOpenContainerData acd = persistenceDirectory.getOpenContainerData(idContainerFileDirOnline);
			StandardOpenContainerData ncd = persistenceDirectory.getOpenContainerData(idContainerFileDirDelayed);

			// Daten des letzten Containers lesen.
			// Es kann sein, dass es keinen offenen Container gibt, aber um Lücken zu bestimmen, müssen
			// dann die Werte des letzten geschlossenen Containers ermittelt werden.
			// Das geht schnell über den Verwaltungsdatenindex.
			ArchiveTask.LastContainerData lastContainerData = archiveTask.initLastContainerData(acd, ncd, dataIdentification);

			if (handleIndexBackstep(persistenceManager, lastContainerData, dataIdx, din, adk, dataset, archTime)) return;

			// Lücke? (Wenn der letze OA-Datensatz "keine Quelle" war, keine Lücke einfuegen).
			// Nicht extra prüfen, wenn beim letzten Aufruf PotGap eingefuegt wurde und diesmal der echte DS.
			// Aus Performance-Gruenden wird im DIN vermerkt, ob der letzte DS ein "keine Quelle" war. Damit
			// spart man den Dateizugriff.
			boolean insertGapDS = allowDataGap && shouldInsertGap(archiveTask, din, lastContainerData, dataIdx, idContainerFileDirOnline);

			boolean append;
			if (insertGapDS) {
				// Lücken werden eingefuegt, indem zuerst der Datensatz auf einen Lücken-Datensatz umgesetzt
				// wird und danach work() mit dem urspr. Datensatz nochmal aufgerufen wird. Alles andere bleibt gleich.
				// Lücken werden immer im OA-Cont. eingetragen. Also ggf. Datensatzart wechseln.
				if (!adk.equals(ArchiveDataKind.ONLINE)) {
					adk = ArchiveDataKind.ONLINE;
				}

				// DIdx der Lücke gemäß TAnfArS 5.1.2.4.3.5 setzen
				dataIdx = Util.dIdxSetArSBit(lastContainerData.lastDataIdx());
				if (din.isFirstDataAfterSubscription(adk)) {
					// Datenzeit nach einem Neustart gemäß Nerz-Ä-78
					long unsubscriptionTime = din.getUnsubscriptionTime();
					dataTime = unsubscriptionTime == -1 ? lastContainerData.lastDataTime() : unsubscriptionTime;
				} else {
					dataTime = ArchiveTask.getLastArchiveTime();
				}
				SerializableDataset gap = SerializableDataset.createGap(archTime, dataIdx, dataTime);
				append = archiveTask.getSerializationHelper().writeData(gap, persistenceDirectory, idContainerFileDirOnline);
			} else {
				append = archiveTask.getSerializationHelper().writeData(dataset, persistenceDirectory, dataIdentification.resolve(adk));
			}
			if (!append) {
				archiveTask.getFailCounterOnline().incrementAndGet();
				return;
			}

			persistenceDirectory.markAsDirty(dataIdentification.resolve(adk)); // Markieren, dass etwas verändert wurde

			din.setFirstDataAfterSubscription(adk, false); // ab jetzt duerfen Datenindexe nicht mehr gleich sein

			// Für nächsten DS merken, damit man nicht in den Cont. schauen muss
			boolean isNoSource = dataset.dataState() == DataState.NO_SOURCE;
			din.setLastOAWasNoSource(adk.equals(ArchiveDataKind.ONLINE) && isNoSource);
			if (isNoSource) {
				if (din.getUnsubscriptionTime() == -1) {
					din.setUnsubscriptionTime(dataTime);
				}
				din.setValidData(false);
			} else {
				din.setUnsubscriptionTime(-1);
				din.setValidData(true);
			}
			if (!insertGapDS) {
				// Datenlückensaetze zählen nicht
				archiveTask.getSuccessCounterOnline().incrementAndGet();

				// nicht quittieren, falls kQ/kD oder nur eine Datenlücke eingefuegt wurde
				if (dataset.hasData()) {
					DataModel dataModel = archiveTask.getArchMgr().getDataModel();
					if (dataModel != null) {
						try {
							archiveTask.sendAck(dataset.asResultData(dataModel, dataIdentification, delayed), din);
						} catch (Exception e) {
							_debug.error("Fehler beim Quittieren eines Datensatzes: " + e.getMessage() + Debug.NEWLINE + dataset, e);
						}
					}
				}
			} else {
				// Nochmal ausführen, da jetzt nicht mehr die Lücke archiviert wird
				archiveData(archiveTask, false);
			}
		} catch (Exception e) {
			archiveTask.getFailCounterOnline().incrementAndGet();
			_debug.error("Fehler beim Archivieren eines Datensatzes: " + e.getMessage() + Debug.NEWLINE + dataset, e);
		}
	}

	/**
	 * Prüft, ob der Datenindex plausibel ist und sendet ggf. die entsprechenden Betriebsmeldungen. Prüft, ob der Datenindex kleiner, gleich oder Größer als
	 * der zuletzt archivierte DI ist. Wenn der DI kleiner ist, wird eine BetrMeld verschickt und nichts archiviert. Wenn der DI gleich ist und dies der erste DS
	 * nach einer Neuanmeldung ist, wird der DS ignoriert (weil der DAV möglicherweise einen gepufferten DS geschickt hat). Wenn der DI Größer ist, wird normal
	 * archiviert.
	 *
	 * @return Wahr, falls der Index nicht plausibel ist und der Datensatz nicht archiviert werden darf, falsch sonst.
	 */
	boolean handleIndexBackstep(PersistenceManager persistenceManager, @NotNull final ArchiveTask.LastContainerData lastContainerData, long dataIdx, DataIdentNode din, ArchiveDataKind adk, SerializableDataset rd, long arsTime) {

		if (dataIdx == lastContainerData.lastDataIdx() && din.isFirstDataAfterSubscription(adk)) {
			_debug.fine("Datenindex nach Neuanmeldung identisch zum letzten archivierten Datenindex. Datensatz ignoriert." + Debug.NEWLINE, rd);
			return true;
		}

		if (dataIdx <= lastContainerData.lastADataIdx()) {
			StringBuilder sb = new StringBuilder(8192);
			sb.append("Es ist ein fehlerhafter Datenindex aufgetreten. Der soeben erhaltene Datenindex ist nicht streng monoton steigend.");
			sb.append("\n\nSoeben erhalten\n===============");
			sb.append("\nDatenidentifikation: ");
			sb.append(persistenceManager.formatContainerDirectory(din.getDataIdentification().resolve(adk)));
			sb.append("\nDatenindex:          ");
			sb.append(dataIdx);
			sb.append(" [").append(Util.dIdx2Str(dataIdx)).append("] [").append(Util.dIdx2StrExt(dataIdx)).append("]");
			sb.append("\nDatenzeit:           ");
			sb.append(Util.timestrMillisFormatted(rd.dataTime()));
			sb.append("\nerhalten:            ");
			sb.append(Util.timestrMillisFormatted(arsTime));
			sb.append("\nnachgeliefert:       ");
			sb.append(Util.bTF(delayed));
			sb.append("\nZustand:             ");
			sb.append(rd.dataState());
			sb.append("\n\n**Beginn Daten**\n");
			sb.append(rd);
			sb.append("\n**Ende Daten**\n");
			sb.append("\nZuletzt archiviert\n==================");
			sb.append("\nArt:                 ");
			sb.append(ArchiveDataKind.ONLINE);
			if (lastContainerData.lastADataIdx() < 0) {
				sb.append("\nDatenindex:          -");
				sb.append("\nDatenzeit:           -");
				sb.append("\nArchivzeit:          -");
			} else {
				sb.append("\nDatenindex:          ");
				sb.append(lastContainerData.lastADataIdx());
				sb.append(" [").append(Util.dIdx2Str(lastContainerData.lastADataIdx())).append("]");
				sb.append(" [").append(Util.dIdx2StrExt(lastContainerData.lastADataIdx())).append("]");
				sb.append("\nDatenzeit:           ").append(Util.timestrMillisFormatted(lastContainerData.lastADTime()));
				sb.append("\nArchivzeit:          ").append(Util.timestrMillisFormatted(lastContainerData.lastAATime()));
			}
			sb.append("\nArt:                 ");
			sb.append(ArchiveDataKind.ONLINE_DELAYED);
			if (lastContainerData.lastNDataIdx() < 0) {
				sb.append("\nDatenindex:          -");
				sb.append("\nDatenzeit:           -");
				sb.append("\nArchivzeit:          -");
			} else {
				sb.append("\nDatenindex:          ");
				sb.append(lastContainerData.lastNDataIdx());
				sb.append(" [").append(Util.dIdx2Str(lastContainerData.lastNDataIdx())).append("]");
				sb.append(" [").append(Util.dIdx2StrExt(lastContainerData.lastNDataIdx())).append("]");
				sb.append("\nDatenzeit:           ").append(Util.timestrMillisFormatted(lastContainerData.lastNDTime()));
				sb.append("\nArchivzeit:          ").append(Util.timestrMillisFormatted(lastContainerData.lastNATime()));
			}
			sb.append("\n");
			_debug.warning("Betriebsmeldung: Datenindex nicht monoton", sb.toString());
			MessageSender.getInstance().sendMessage(
					MSG_PID_DATAINDEX_ANOMALY, MessageType.APPLICATION_DOMAIN, "", MessageGrade.FATAL, MessageState.MESSAGE, sb.toString()
			);
			return true;
		}
		return false;
	}


	private static boolean shouldInsertGap(ArchiveTask archiveTask, DataIdentNode din, ArchiveTask.LastContainerData lastContainerData, long dataIdx, IdContainerFileDir containerFileDir) {
		if (archiveTask.isDataGap(dataIdx, lastContainerData)) {
			if (din.lastDataNoSourceAssigned()) {
				return !din.lastDataWasNoSource();
			} else {
				// Hier muss man leider auf den Index synchronisieren, um Lesezugriff auf den Container zu haben
				return !scanOAContForLastNoSrc(lastContainerData.lastAContID(), archiveTask.getPersistenceManager(), containerFileDir);
			}
		}
		return false;
	}

	/**
	 * Durchläuft den aktuellen Container und prüft, ob der letzte Datensatz die Kennung "keine Quelle" hat (langsam!). Dies ist nur notwendig, wenn das erste
	 * mal auf die DID zugegriffen wird. Sonst genuegt {@link DataIdentNode#lastDataWasNoSource()}. Der Aufrufer dieser Methode muss die Synchronisierung über den
	 * DataIdentNode durchfuehren.
	 *
	 * @return Wahr, falls der letzte Datensatz die Kennung "keine Quelle" hat, falsch sonst
	 */
	static boolean scanOAContForLastNoSrc(long lastAContID, PersistenceManager persMgr, IdContainerFileDir containerFileDir) {
		if (lastAContID == -1) return false;
		try (LockedContainerDirectory directory = persMgr.lockIndex(containerFileDir)) {
			ContainerDataResult data = persMgr.getLastDataSet(directory);
			return data != null && data.isNoSource();
		} catch (Exception e) {
			_debug.warning("Fehler beim Scan über den letzten Container wird ignoriert", e);
		}
		return false;
	}


	@Override
	public boolean equals(Object obj) {
		if (obj == this) return true;
		if (obj == null || obj.getClass() != this.getClass()) return false;
		var that = (ArchiveOnlineData) obj;
		return Objects.equals(this.dataset, that.dataset);
	}

	@Override
	public String toString() {
		return "ArchiveOnlineData[" +
				"dataset=" + dataset + ']';
	}


	@Override
	public long estimateMemoryUsage() {
		return 28 + dataset.estimateMemoryUsage();
	}
}
