/*
 *
 * Copyright 2005-2008 by beck et al. projects GmbH, Munich
 * Copyright 2009-2020 by Kappich Systemberatung, Aachen
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.ars.ars.persistence.writer;

import de.bsvrz.ars.ars.mgmt.ArchiveManager;
import de.bsvrz.ars.ars.mgmt.datatree.DataIdentNode;
import de.bsvrz.ars.ars.mgmt.datatree.synchronization.SyncKey;
import de.bsvrz.ars.ars.mgmt.datatree.synchronization.SynchronizationFailedException;
import de.bsvrz.ars.ars.mgmt.tasks.RepeatingTask;
import de.bsvrz.ars.ars.persistence.*;
import de.bsvrz.ars.ars.persistence.index.IndexException;
import de.bsvrz.ars.ars.persistence.util.SignalingQueue;
import de.bsvrz.ars.ars.persistence.util.TerminatedException;
import de.bsvrz.dav.daf.main.DataNotSubscribedException;
import de.bsvrz.dav.daf.main.ResultData;
import de.bsvrz.dav.daf.main.SendSubscriptionNotConfirmed;
import de.bsvrz.dav.daf.main.archive.ArchiveDataKind;
import de.bsvrz.dav.daf.main.config.AttributeGroup;
import de.bsvrz.dav.daf.main.config.DataModel;
import de.bsvrz.sys.funclib.debug.Debug;
import de.bsvrz.sys.funclib.kappich.annotations.NotNull;
import de.bsvrz.sys.funclib.losb.datk.ContainerSettings;
import de.bsvrz.sys.funclib.losb.util.Util;

import java.util.concurrent.atomic.AtomicLong;

/**
 * Task zur Archivierung von Datensätzen. Ist nicht von SingleTask abgeleitet, weil ein spezieller ArchiveRingBuffer verwendet wird, der zwei Objekte
 * (ResultData und Archivzeit) speichern kann. Die work()-Methode hat deswegen zwei Parameter.
 *
 * @author beck et al. projects GmbH
 * @author Alexander Schmidt
 * @version $Revision$ / $Date$ / ($Author$)
 */
public class ArchiveTask extends RepeatingTask {
	private final AtomicLong _successCountOnline = new AtomicLong(0);

	private final AtomicLong _successCountRequested = new AtomicLong(0);

	private final AtomicLong _failCountOnline = new AtomicLong(0);

	private final AtomicLong _failCountRequested = new AtomicLong(0);

	private final AtomicLong _closeCount = new AtomicLong(0);

	private final SignalingQueue<ArchiveJob> inputDataQueue;

	private static long lastArchiveTime;


	//////////////////////////////////////////////////////////////////////////////////////////////////////////
	// ContainerSettings:
	//
	/**
	 * Container-Einstellungen des Archivsystems (aus "atg.archivContainer").
	 */
	private static ContainerSettings containerSettings;


	static {
		// Standardwerte (keine Default-Werte im Datenkatalog vorgesehen):
		final ContainerSettings settings = new ContainerSettings();
		settings.stdCloseConditions.maxContAnzDS = 5000;
		settings.stdCloseConditions.maxContSize = 20 * 1000 * 1000;    // 20 MB
		settings.stdCloseConditions.maxContTime = 60 * 60 * 24 * 7;    // 1 Woche
		setContainerSettings(settings);
	}

	private final SerializationHelper serializationHelper = new SerializationHelper(new CloseConditions() {
		@Override
		public ContainerSettings.CloseCondition getCloseCondition(long atg) {
			ContainerSettings.CloseCondition exceptionalConditions = getException(atg);
			if (exceptionalConditions != null) {
				return exceptionalConditions;
			}
			return containerSettings.stdCloseConditions;
		}

		private ContainerSettings.CloseCondition getException(long atg) {
			try {
				DataModel dataModel = getArchMgr().getDataModel();
				if (dataModel == null) {
					return null;
				}
				AttributeGroup group = dataModel.getObject(atg, AttributeGroup.class);
				if (group == null) {
					return null;
				}
				return containerSettings.getExceptionSettings(group);
			} catch (Exception e) {
				_debug.warning("Attributgruppe nicht ermittelbar: " + atg, e);
				return null;
			}
		}
	}, this.getPersistenceManager(), true);

	public SerializationHelper getSerializationHelper() {
		return serializationHelper;
	}

	/**
	 * Setzt die Container-Einstellungen des Archivsystems.
	 *
	 * @param cs Container-Einstellungen
	 */
	public static void setContainerSettings(ContainerSettings cs) {
		containerSettings = cs;
	}

	/**
	 * Liefert die Container-Einstellungen des Archivsystems.
	 *
	 * @return Container-Einstellungen
	 */
	public static ContainerSettings getContainerSettings() {
		return containerSettings;
	}
	//
	//////////////////////////////////////////////////////////////////////////////////////////////////////////

	/**
	 * Konstruktor des Archiv-Tasks.
	 *
	 * @param archiveMgr Archiv-Manager
	 * @param inQueue    Input-Queue
	 */
	public ArchiveTask(ArchiveManager archiveMgr, SignalingQueue<ArchiveJob> inQueue) {
		super(archiveMgr);
		inputDataQueue = inQueue;
	}

	@Override
	public void step() throws InterruptedException {
		try {
			inputDataQueue.take().accept(this);
		} catch (TerminatedException e) {
			return;
		}

		taskStepDone();
	}

	@Override
	public void terminateTask() {
		super.terminateTask();
		inputDataQueue.terminateNow();
	}

	/**
	 * @return Anzahl der seit Systemstart erfolgreich archivierten Datensätze
	 */
	public long getSuccessCount() {
		return getSuccessCountOnline() + getSuccessCountRequested();
	}

	/**
	 * @return Anzahl der seit Systemstart nicht erfolgreich archivierten Datensätze
	 */
	public long getFailedCount() {
		return getFailedCountOnline() + getFailedCountRequested();
	}

	/**
	 * @return Anzahl der seit Systemstart erfolgreich archivierten Datensätze
	 */
	public long getSuccessCountRequested() {
		return _successCountRequested.longValue();
	}

	/**
	 * @return Anzahl der seit Systemstart nicht erfolgreich archivierten Datensätze
	 */
	public long getFailedCountRequested() {
		return _failCountRequested.longValue();
	}

	/**
	 * @return Anzahl der seit Systemstart erfolgreich archivierten Datensätze
	 */
	public long getSuccessCountOnline() {
		return _successCountOnline.longValue();
	}

	/**
	 * @return Anzahl der seit Systemstart nicht erfolgreich archivierten Datensätze
	 */
	public long getFailedCountOnline() {
		return _failCountOnline.longValue();
	}

	/**
	 * @return Zahl aller seit dem Start oder letzten {@link #resetDSCounter()} erfolgreich verarbeiteten Close-Container-Datensätze. Kann zur
	 * Test-Synchronisation verwendet werden.
	 */
	public long getCloseContainerSuccess() {
		return _closeCount.longValue();
	}


	/**
	 * Setzt alle Zähler auf 0.
	 */
	public void resetDSCounter() {
		_successCountOnline.set(0);
		_failCountOnline.set(0);
		_successCountRequested.set(0);
		_failCountRequested.set(0);
		_closeCount.set(0);
	}

	public static void setLastArchiveTime(long atime) {
		lastArchiveTime = atime;
	}

	/**
	 * Letzte verwendete Archivzeit. Wird für die Archivzeitüberwachung benutzt.
	 */
	public static long getLastArchiveTime() {
		return lastArchiveTime;
	}

	@NotNull
	LastContainerData initLastContainerData(StandardOpenContainerData acd, StandardOpenContainerData ncd, IdDataIdentification dataIdentification) throws SynchronizationFailedException, IndexException {

		if (acd != null || ncd != null) {
			return new LastContainerData(
					acd == null ? -1 : acd.getMaxDataIdx(),
					ncd == null ? -1 : ncd.getMaxDataIdx(),
					acd == null ? -1 : acd.getMaxDataTime(),
					ncd == null ? -1 : ncd.getMaxDataTime(),
					acd == null ? -1 : acd.getMaxArcTime(),
					ncd == null ? -1 : ncd.getMaxArcTime(),
					acd == null ? -1 : acd.getContainerId()
			);
		}
		// nicht in OpenContainerData, aus Verwaltungsdatenindex lesen.
		ContainerHeaders irA;
		ContainerHeaders irN;
		try (SyncKey<IdDataIdentification> lock = getPersistenceManager().lockIndex(dataIdentification)) {
			irA = getPersistenceManager().getLastContainerHeaders(new LockedContainerDirectory(lock, ArchiveDataKind.ONLINE));
			irN = getPersistenceManager().getLastContainerHeaders(new LockedContainerDirectory(lock, ArchiveDataKind.ONLINE_DELAYED));
		}
		long lastADataIdx = irA != null ? irA.getContainerHeaderParamAsLong(ContainerManagementInformation.CHP_DATA_IDX_MAX) : -1;
		long lastNDataIdx = irN != null ? irN.getContainerHeaderParamAsLong(ContainerManagementInformation.CHP_DATA_IDX_MAX) : -1;
		long lastADTime = irA != null ? irA.getContainerHeaderParamAsLong(ContainerManagementInformation.CHP_DATA_TIME_MAX) : -1;
		long lastNDTime = irN != null ? irN.getContainerHeaderParamAsLong(ContainerManagementInformation.CHP_DATA_TIME_MAX) : -1;
		long lastAATime = irA != null ? irA.getContainerHeaderParamAsLong(ContainerManagementInformation.CHP_ARC_TIME_MAX) : -1;
		long lastNATime = irN != null ? irN.getContainerHeaderParamAsLong(ContainerManagementInformation.CHP_ARC_TIME_MAX) : -1;
		long lastAContID = irA != null ? irA.getContainerHeaderParamAsLong(ContainerManagementInformation.CHP_CONT_ID) : -1;

		return new LastContainerData(
				lastADataIdx,
				lastNDataIdx,
				lastADTime,
				lastNDTime,
				lastAATime,
				lastNATime,
				lastAContID
		);
	}

	/**
	 * Prüft, ob eine potenzielle Datenlücke vorliegt. Der Aufrufer dieser Methode muss die Synchronisierung über den DataIdentNode durchführen.
	 *
	 * @param dataIndex         Datenindex des aktuell zu archivierenden Datensatzes
	 * @param lastContainerData Daten des zuvor archivierten Datensatzes oder null, falls noch nichts archiviert
	 * @return Wahr, falls eine potenzielle Datenlücke vorliegt, falsch sonst.
	 */
	boolean isDataGap(long dataIndex, LastContainerData lastContainerData) {
		if (getArchMgr().getInQueuesMgr().getRuntimeControl().archiveOnlyData()) return false;    // nur für Testzwecke

		if (lastContainerData == null) {
			return false;    // noch nie vorher was archiviert
		}

		// "63/93-Fehler": Wenn der letzte DS ein Gap-Marker ist, der nächste DS aber nicht geschrieben
		// wurde, würde das Setzen des ArSBits von lastDataIdx nichts ändern und einen Indexfehler
		// verursachen. Der neue DS wird nicht geschrieben und das Problem wiederholt sich.
		// Daher kann es durch die folgende Abfrage nie eine Lücke geben, wenn der letzte
		// DS eine Lückenmarkierung ist.
		else if (Util.dIdxArSBit(lastContainerData.lastDataIdx()) == 1) {
			return false;
		} else if (isIndexGap(lastContainerData.lastDataIdx(), dataIndex)) {
			_debug.fine(
					"Potentielle Datenluecke erkannt: letzter archivierter Index (O" + (lastContainerData.lastIsOnline() ? "A" : "N") + "): " + lastContainerData.lastDataIdx() + " ["
							+ Util.dIdx2Str(lastContainerData.lastDataIdx()) + "] [" + Util.dIdx2StrExt(lastContainerData.lastDataIdx()) + "]" + ", erhaltener Datenindex" + dataIndex + " ["
							+ Util.dIdx2Str(dataIndex) + "] [" + Util.dIdx2StrExt(dataIndex) + "]"
			);
			return true;
		} else {
			return false;
		}
	}

	/**
	 * Gibt zurück, ob zwischen 2 Datensätze eine Indexlücke vorliegt
	 *
	 * @param lastDataIdx erster Datensatz
	 * @param nextDataIdx nächster Datensatz
	 * @return true: Indexlücke, sonst false
	 */
	public static boolean isIndexGap(long lastDataIdx, long nextDataIdx) {
		return Util.dIdxNoModBits(nextDataIdx) > Util.dIdxNoModBits(lastDataIdx) + 1;
	}

	/**
	 * Sendet eine Quittung für den angegebenen Datensatz.
	 *
	 * @param rd  Datensatz
	 * @param din Datenknoten mit Parametrierung
	 */
	protected void sendAck(ResultData rd, DataIdentNode din) {
		// Der Quittierungsaspekt wird von ArchivConfig sofort abgemeldet, sobald eine DID nicht mehr
		// archiviert/parametriert/quittiert wird.
		// Daher kann für solche bereits in der Queue vorhandenen DS keine Quittung gesendet werden.

		if (din.isArSParameterized() && din.arSParamIsArchivieren() && din.arSParamIsQuittierenValid()) {
			long qAspId = din.arSParamGetQuittieren();
			try {
				getArchMgr().getInQueuesMgr().getDataAckSender().sendAck(rd, qAspId);
				_debug.finest("Quittung unter Aspekt '" + qAspId + "' gesendet fuer", Util.rd2Str(rd));
			} catch (DataNotSubscribedException | SendSubscriptionNotConfirmed e) {
				_debug.warning("Senden der Quittung unter Aspekt '" + qAspId + "' fehlgeschlagen: " + e.getMessage() + Debug.NEWLINE + Util.rd2Str(rd));
			}
		}
	}

	public AtomicLong getFailCounterOnline() {
		return _failCountOnline;
	}

	public AtomicLong getFailCounterRequested() {
		return _failCountRequested;
	}

	public AtomicLong getCloseCounter() {
		return _closeCount;
	}

	public AtomicLong getSuccessCounterOnline() {
		return _successCountOnline;
	}

	public AtomicLong getSuccessCounterRequested() {
		return _successCountRequested;
	}

	/**
	 * Sammlung von Wertebereichen der letzten Container
	 *
	 * @param lastADataIdx letzter aktueller Datenindex
	 * @param lastNDataIdx letzter nachgelieferter Datenindex
	 * @param lastADTime   letzte aktuelle Datenzeit
	 * @param lastNDTime   letzte nachgelieferte Datenzeit
	 * @param lastAATime   letzte aktuelle Archivzeit
	 * @param lastNATime   letzte nachgelieferte Archivzeit
	 * @param lastAContID  letzte aktuelle Container-ID
	 */
	public record LastContainerData(
			long lastADataIdx,
			long lastNDataIdx,
			long lastADTime,
			long lastNDTime,
			long lastAATime,
			long lastNATime,
			long lastAContID) {

		/**
		 * Gibt den letzten Datenindex (allgemein) zurück.
		 *
		 * @return den letzten Datenindex (allgemein)
		 */
		public long lastDataIdx() {
			return Math.max(lastADataIdx, lastNDataIdx);
		}

		/**
		 * Gibt die letzte Datenzeit (allgemein) zurück.
		 *
		 * @return die letzte Datenzeit (allgemein)
		 */
		public long lastDataTime() {
			return Math.max(lastADTime, lastNDTime);
		}

		/**
		 * Gibt {@code true} zurück, wenn der letzte Datensatz online empfangen wurde.
		 *
		 * @return {@code true}, wenn der letzte Datensatz online empfangen wurde, sonst {@code false}
		 */
		public boolean lastIsOnline() {
			return lastADataIdx > lastNDataIdx;
		}
	}
}
