/*
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */
package de.bsvrz.ars.ars.persistence.writer;

import de.bsvrz.ars.ars.mgmt.tasks.SerializeTaskHelper;
import de.bsvrz.ars.ars.persistence.ContainerDataResult;
import de.bsvrz.ars.ars.persistence.ContainerFile;
import de.bsvrz.ars.ars.persistence.IdDataIdentification;
import de.bsvrz.ars.ars.persistence.util.SignalingQueue;
import de.bsvrz.dav.daf.main.*;
import de.bsvrz.dav.daf.main.config.Aspect;
import de.bsvrz.dav.daf.main.config.AttributeGroup;
import de.bsvrz.dav.daf.main.config.DataModel;
import de.bsvrz.sys.funclib.kappich.annotations.NotNull;
import de.bsvrz.sys.funclib.kappich.annotations.Nullable;
import org.jetbrains.annotations.Contract;

/**
 * Interface, für Datensätze, die archiviert werden können. Um einen {@link SerializableDataset} zu erhalten,
 * sollte einer der statischen Factory-Methoden dieses Interfaces benutzt werden.
 */
public interface SerializableDataset {

	/**
	 * Erstellt einen {@link SerializableDataset}, der eine (potenzielle) Datenlücke repräsentiert.
	 *
	 * @param archTime Archivzeit
	 * @param dataIdx  Datenindex
	 * @param dataTime Datenzeit
	 * @return neuer Datensatz
	 */
	@NotNull
	@Contract("_, _, _ -> new")
	static SerializableDataset createGap(long archTime, long dataIdx, long dataTime) {
		return new RegularSerializableDataset(archTime, dataTime, dataIdx, ContainerFile.POT_GAP);
	}

	/**
	 * Erstellt einen neuen {@link SerializableDataset} basierend auf bereits archivierten Daten.
	 * Das ist z. B. für die Migration sinnvoll.
	 *
	 * @param result Bereits archivierte Daten
	 * @return neuer Datensatz
	 */
	@NotNull
	@Contract("_ -> new")
	static SerializableDataset create(ContainerDataResult result) {
		byte[] dataBytes = result.getData();
		if (dataBytes == null) {
			dataBytes = switch (result.getDataState().getCode()) {
				case 2 -> ContainerFile.NO_DATA;
				case 3 -> ContainerFile.NO_SOURCE;
				case 4 -> ContainerFile.NO_RIGHTS;
				default -> ContainerFile.POT_GAP;
			};
		}

		return new RegularSerializableDataset(result.getArchiveTime(), result.getDataTime(), result.getDataIndex(),
				dataBytes, result.getDataUncompressedSize(), result.isCompressed());
	}

	/**
	 * Erstellt einen neuen {@link SerializableDataset} für leere Datensätze.
	 * Das ist z. B. für die Migration sinnvoll.
	 *
	 * @param dataset     Leerer Datensatz
	 * @param archiveTime Archivzeit
	 * @return neuer Datensatz
	 */
	@NotNull
	private static SerializableDataset createEmpty(Dataset dataset, long archiveTime) {
		byte[] bytes = switch (dataset.getDataType().getCode()) {
			case 2 -> ContainerFile.NO_DATA;
			case 3 -> ContainerFile.NO_SOURCE;
			case 4 -> ContainerFile.NO_RIGHTS;
			default -> throw new IllegalArgumentException("Unerwarteter DataState: " + dataset.getDataType());
		};

		return new RegularSerializableDataset(archiveTime, dataset.getDataTime(), dataset.getDataIndex(), bytes);
	}

	/**
	 * Erstellt einen neuen {@link SerializableDataset} basierend auf einem Datenverteiler-Datensatz.
	 *
	 * @param dataset     Datenverteiler-Datensatz (z. B. ein {@link ResultData})
	 * @param archiveTime Archivzeit
	 * @param helper      Hilfsklasse für Kompression mit Puffer. Diese Klasse ist nicht threadsafe und darf
	 *                    nur von einem create-Aufruf gleichzeitig benutzt werden.   
	 * @return neuer Datensatz
	 */
	@NotNull
	static SerializableDataset create(Dataset dataset, long archiveTime, SerializeTaskHelper helper) {

		Data data = dataset.getData();
		if (data == null) {
			return createEmpty(dataset, archiveTime);
		}

		AsyncSerializableDataset asyncSerializableDataset = new AsyncSerializableDataset(
				archiveTime,
				dataset.getDataTime(),
				dataset.getDataIndex(),
				(byte) dataset.getDataType().getCode(),
				data
		);

		helper.complete(asyncSerializableDataset);

		return asyncSerializableDataset;
	}

	/**
	 * Erstellt einen neuen {@link SerializableDataset} basierend auf einem Datenverteiler-Datensatz.
	 * Die Serialisierung des Datenverteiler-Datensatzes erfolgt asynchron, indem ein
	 * {@link AsyncSerializableDataset} in die übergebene Queue eingefügt wird. Diese Queue muss extern von einem
	 * Thread oder Threadpool abgearbeitet werden, was unabhängig von dieser Methode implementiert werden muss.
	 * <p>
	 * Im Archivsystem wird die Queue vom {@link de.bsvrz.ars.ars.mgmt.tasks.SerializeTask} abgearbeitet.
	 *
	 * @param dataset        Datenverteiler-Datensatz (z. B. ein {@link ResultData})
	 * @param archiveTime    Archivzeit
	 * @param serializeQueue Queue, in der der Serialisierungsjob eingefügt wird.
	 * @return neuer Datensatz
	 */
	@NotNull
	@Contract("_, _, _ -> new")
	static SerializableDataset createAsync(Dataset dataset, long archiveTime, SignalingQueue<AsyncSerializableDataset> serializeQueue) {

		Data data = dataset.getData();
		if (data == null) {
			return createEmpty(dataset, archiveTime);
		}

		AsyncSerializableDataset asyncSerializableDataset = new AsyncSerializableDataset(
				archiveTime,
				dataset.getDataTime(),
				dataset.getDataIndex(),
				(byte) dataset.getDataType().getCode(),
				data
		);

		serializeQueue.add(asyncSerializableDataset);

		return asyncSerializableDataset;
	}
	
	/**
	 * Gibt die Archivzeit in Millisekunden seit Epoch zurück.
	 *
	 * @return die Archivzeit
	 */
	long archiveTime();

	/**
	 * Gibt die Datenzeit in Millisekunden seit Epoch zurück.
	 *
	 * @return die Datenzeit
	 */
	long dataTime();

	/**
	 * Gibt den Datenindex zurück.
	 *
	 * @return den Datenindex
	 */
	long dataIndex();

	/**
	 * Gibt den Zustand des Datensatzes zurück.
	 *
	 * @return den Zustand des Datensatzes
	 */
	@NotNull
	DataState dataState();

	/**
	 * Gibt ein den serialisierten Data-Inhalt zurück. Die Daten sind evtl. komprimiert (siehe {@link #uncompressedSize()}).
	 * Datenlücken werden als 6-Byte Folge wie {@link de.bsvrz.ars.ars.persistence.ContainerFile#NO_SOURCE} usw.
	 * zurückgegeben. Dieses Byte-Array enthält noch nicht den Header mit Zeitstempeln usw., wie er in die
	 * Containerdatei geschrieben wird. Dieser kann mit 
	 * {@link de.bsvrz.ars.ars.mgmt.tasks.ArchiveDataSerializer#serializeData(SerializableDataset)}
	 * erzeugt werden.
	 *
	 * @return serialisierte und ggf. komprimierte Daten
	 */
	@NotNull
	byte[] serializedData();

	/**
	 * Schätz den Speicherverbrauch dieses Objekts
	 *
	 * @return Speicherverbrauch in Bytes
	 */
	long estimateMemoryUsage();

	/**
	 * Gibt zurück, ob Nutzdaten vorhanden sind. Dies ist der Fall, wenn {@link #dataState()} den Wert
	 * {@link DataState#DATA} zurückliefert.
	 *
	 * @return Sind Nutzdaten vorhanden?
	 */
	default boolean hasData() {
		return dataState().equals(DataState.DATA);
	}

	/**
	 * Gibt den ursprünglichen empfangenen Dateninhalt zurück. Der Wert kann null sein und ist nur vorhanden,
	 * wenn die Daten vorher vom Datenverteiler empfangen wurden. Dies wird nur für die Quittierung benötigt.
	 *
	 * @return Originaler Datensatz
	 */
	@Nullable
	Data originalData();

	/**
	 * Gibt für die Quittierung diesen Datensatz als ResultData zurück.
	 * <p>
	 * Dies ist nur möglich, wenn in originalData() der ursprüngliche Datensatz übergeben wurde.
	 *
	 * @param dataModel Datenmodell
	 * @param did       Datenidentifikation
	 * @param isDelayed Nachgeliefert?
	 * @return Datenverteiler-Datensatz
	 */
	default ResultData asResultData(DataModel dataModel, IdDataIdentification did, boolean isDelayed) {
		AttributeGroup attributeGroup = dataModel.getObject(did.getAtgId(), AttributeGroup.class);
		Aspect asp = dataModel.getObject(did.getAspectId(), Aspect.class);
		return new ResultData(
				dataModel.getObject(did.getObjectId()),
				new DataDescription(attributeGroup,
						asp,
						(short) did.getSimVariant()
				),
				isDelayed,
				dataIndex(),
				dataTime(),
				(byte) (dataState().getCode() - 1),
				originalData()
		);
	}

	/**
	 * Gibt die Größe des originalen, unkomprimierten Daten-Arrays zurück. Enthält 0
	 * (={@link ContainerFile#NOT_COMPRESSED}) falls die Daten unkomprimiert sind, dann zählt die Größe
	 * des serializedData-Bytearrays.
	 *
	 * @return die Größe des unkomprimierten Daten-Arrays
	 */
	int uncompressedSize();

}
