/*
 *
 * Copyright 2009-2020 by Kappich Systemberatung, Aachen
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.ars.ars.repair;

import de.bsvrz.ars.ars.persistence.*;
import de.bsvrz.ars.ars.persistence.directories.ReadonlyPersistenceDirectory;
import de.bsvrz.ars.ars.persistence.directories.PersistenceDirectory;
import de.bsvrz.ars.ars.persistence.layout.PersistenceDirectoryLayoutInstance;
import de.bsvrz.sys.funclib.commandLineArgs.ArgumentList;
import de.bsvrz.sys.funclib.debug.Debug;

import java.nio.file.Path;

/**
 * Hilfsprogramm, mit dem es möglich ist, Container-Dateien des Archivsystems, in denen bestimmte Defekte vorliegen zu identifizieren und zu korrigieren. Das
 * Programm durchsucht in einem vorgegebenen Verzeichnis und auf Wunsch auch rekursiv in allen Unterverzeichnisse nach Container-Dateien und führt Prüfungen auf
 * bestimmte Defekte durch, die durch Fehler in älteren Versionen des Archivsystems ausgelöst wurden. Defekte Containerdateien werden durch eine entsprechende
 * Ausgabe identifiziert und auf Wunsch (wenn möglich) auch korrigiert.
 * <p>
 * Das Programm unterstützt folgende Aufrufparameter: -verzeichnis=... -rekursiv=ja|nein -reparieren=nein|ja -index=nein|ja -container=nein|ja
 * <p>
 * Mit {@code -verzeichnis=...} kann ein beliebiges Verzeichnis angegeben werden, in dem Containerdateien gesucht werden. Defaultwert ist das aktuelle
 * Arbeitsverzeichnis, in dem das Programm gestartet wurde.
 * <p>
 * Mit {@code -rekursiv=ja|nein} kann spezifiziert werden, ob auch Unterverzeichnisse des Ausgangsverzeichnisses rekursiv durchsucht werden sollen oder
 * nicht. Ohne Angabe dieses Arguments werden auch Unterverzeichnisse rekursiv durchsucht.
 * <p>
 * Mit {@code -reparieren=nein|ja} kann spezifiziert werden, ob fehlerhafte Dateien nur identifiziert werden sollen oder ob sie auch repariert werden
 * sollen.
 * <p>
 * Mit {@code -index=ja|nein} kann spezifiziert werden, ob Indexdateien überprüft werden
 * sollen.
 * <p>
 * Mit {@code -container=ja|nein} kann spezifiziert werden, ob Containerdateien überprüft werden
 * sollen.
 *
 * @author Kappich Systemberatung
*/
@SuppressWarnings("UseOfSystemOutOrSystemErr")
public class ContainerRescue {

	private final Path _startDirectory;

	private final boolean _recursiveScan;

	private final boolean _fixDefects;

	private final boolean _checkContainerFiles;

	private final boolean _checkIndexFiles;

	private final boolean _extended;

	public static void main(String[] args) {
		try {
			final ArgumentList arguments = new ArgumentList(args);
			Debug.init("ContainerRescue", arguments);
			final ContainerRescue containerRescue = new ContainerRescue(arguments);
			containerRescue.start();
		}
		catch(Exception e) {
			System.err.println("Ein unerwarteter Fehler ist während der Programmausführung aufgetreten: " + e);
			e.printStackTrace(System.err);
		}
	}

	public ContainerRescue(final ArgumentList arguments) {
		_startDirectory = arguments.fetchArgument("-verzeichnis=.").asPath();
		_recursiveScan = arguments.fetchArgument("-rekursiv=ja").booleanValue();
		_checkContainerFiles = arguments.fetchArgument("-container=ja").booleanValue();
		_checkIndexFiles = arguments.fetchArgument("-index=ja").booleanValue();
		_extended = arguments.fetchArgument("-erweitert=nein").booleanValue();
		_fixDefects = arguments.fetchArgument("-reparieren=nein").booleanValue();
		arguments.ensureAllArgumentsUsed();
	}

	private void start() throws PersistenceException {
		System.out.println(
				"Aufrufparameter: -verzeichnis=" + _startDirectory + " -rekursiv=" + (_recursiveScan ? "ja" : "nein") + " -container="
						+ (_checkContainerFiles ? "ja" : "nein") + " -index=" + (_checkIndexFiles ? "ja" : "nein") + " -erweitert=" + (_extended ? "ja" : "nein") + " -reparieren=" + (_fixDefects ? "ja" : "nein")
		);

		System.out.println(
				"########## Das Verzeichnis " + _startDirectory.toAbsolutePath()
						+ (_recursiveScan ? " und rekursiv alle enthaltenen Unterverzeichnisse werden" : " wird") + " nach defekten Dateien untersucht. " + "Defekte Dateien werden ausgegeben " + (_fixDefects ? "und" : "aber nicht") + " repariert."
		);
		long t0 = System.currentTimeMillis();

		// Dummy-Implementierung. Der Rescue-Lauf erzeugt werder Container-IDs noch unterstützt er Synchronisierung!
		DataIdentificationManager containerCreator = dataIdentification -> {
			throw new AssertionError();
		};
		PersistenceDirectoryLayoutInstance layoutInstance = PersistenceDirectoryLayoutInstance.detectFromPath(_startDirectory);

		RescueResult rescueResult;
		ScanMode scanMode = new ScanMode(_checkIndexFiles, _checkContainerFiles, _checkIndexFiles && _extended, _checkContainerFiles && _extended, true, false);
		if (!_recursiveScan) {
			rescueResult = new RescueResult();
			PersistenceCheckRunner.scanDirectory(_startDirectory, layoutInstance, scanMode, rescueResult);
		} else {
			PersistenceDirectory persistenceDirectory = new ReadonlyPersistenceDirectory(containerCreator, layoutInstance);
			rescueResult = PersistenceCheckRunner.scan(persistenceDirectory, scanMode);
		}
		long t1 = System.currentTimeMillis();
		long scanTime = t1 - t0;
		System.out.println("########## Zusammenfassung der Prüfung:");
		System.out.println("Dauer der Prüfung:                      " + String.format("%.3f Sekunden", scanTime / 1000.0));
		System.out.println("Anzahl geprüfter Container:             " + rescueResult.getCheckedContainerFileCount());
		System.out.println("Anzahl Container ohne Defekte:          " + rescueResult.getValidContainerFileCount());
		System.out.println("Anzahl Container mit Defekten:          " + rescueResult.getDefectContainerFileCount());
		System.out.println("Anzahl geprüfter Indexdateien:          " + rescueResult.getCheckedIndexFileCount());
		System.out.println("Anzahl Indexdateien ohne Defekte:       " + rescueResult.getValidIndexFileCount());
		System.out.println("Anzahl Indexdateien mit Defekten:       " + rescueResult.getDefectIndexFileCount());
		System.out.println("Anzahl korrigierbare Defekte:           " + rescueResult.getFixableDefectsCount());
		System.out.println("Anzahl gelöschte Backup-Index-Dateien : " + rescueResult.getDeletedFiles());
		if (rescueResult.getFixableDefectsCount() > 0) {
			if (_fixDefects) {
				PersistenceCheckRunner.fixDefects(rescueResult, layoutInstance);
			}
			else {
				System.out.println("########## Zur Korrektur der korrigierbaren Defekte bitte das Aufrufargument -reparieren=ja verwenden.");
			}
		}
	}

}
