/*
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.migration.
 *
 * de.bsvrz.ars.migration is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.migration is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.migration.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */
package de.bsvrz.ars.migration.kappich;

import com.google.common.collect.Range;
import de.bsvrz.ars.ars.persistence.PersistenceException;
import de.bsvrz.ars.persistence.FileSystemPersistenceModule;
import de.bsvrz.dav.daf.main.impl.archive.PersistentContainerStreamSupplier;
import de.bsvrz.sys.funclib.commandLineArgs.ArgumentList;
import de.bsvrz.sys.funclib.debug.Debug;

import java.io.IOException;
import java.nio.file.Path;
import java.time.Instant;

/**
 * Hauptklasse für das Migrationstool. Diese Klasse liest Aufrufargumente und startet dann die eigentliche Migration, die im {@link MigrateWorker}
 * implementiert ist.
 */
public class Migrate {

    /**
     * Main-methode, die beim Start der Migration ausgeführt wird.
     *
     * @param args Aufrufargumente
     */
    public static void main(String[] args) {
        try {
            runWithArgs(args);
        } catch (IllegalArgumentException e) {
            System.err.println();
            e.printStackTrace(System.err);
            System.exit(1);
        } catch (PersistenceException e) {
            System.err.println("Bei der Migration trat ein Fehler in der Persistenzschicht auf:");
            e.printStackTrace(System.err);
            System.exit(1);
        } catch (Exception e) {
            System.err.println("Bei der Migration trat ein Fehler auf:");
            e.printStackTrace(System.err);
            System.exit(1);
        }
    }

    /**
     * Diese methode entspricht der Main-methode, jedoch ohne Fehlerbehandlung.
     *
     * @param args Argumente
     * @throws PersistenceException Fehler in Persistenzschicht
     */
    public static void runWithArgs(String[] args) throws PersistenceException {
        ArgumentList argumentList = new ArgumentList(args);
        Path src = argumentList.fetchArgument("-src").asPath();
        Path dest = argumentList.fetchArgument("-dst").asPath();
        Instant from = argumentList.hasArgument("-from") ? argumentList.fetchArgument("-from").asInstant() : Instant.MIN;
        Instant to = argumentList.hasArgument("-to") ? argumentList.fetchArgument("-to").asInstant() : Instant.MAX;
        int threads = argumentList.fetchArgument("-threads=4").intValue();
        boolean closeIndexes = argumentList.fetchArgument("-closeIndexes=false").booleanValue();

        Debug.init("Migration", argumentList);

        FileSystemPersistenceModule containerLoader = new FileSystemPersistenceModule(src);
        argumentList.ensureAllArgumentsUsed();

        PersistentContainerStreamSupplier allContainers = containerLoader.getAllContainers();

        MigrateWorker worker = new MigrateWorker(allContainers, dest, threads, Range.closedOpen(from, to), closeIndexes);
        worker.start();
    }

}
