/*
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.migration.
 *
 * de.bsvrz.ars.migration is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.migration is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.migration.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */
package de.bsvrz.ars.migration.kappich;

import com.google.common.collect.Range;
import de.bsvrz.ars.ars.mgmt.datatree.DataIdentTree;
import de.bsvrz.ars.ars.mgmt.datatree.synchronization.SyncKey;
import de.bsvrz.ars.ars.mgmt.datatree.synchronization.SynchronizationFailedException;
import de.bsvrz.ars.ars.persistence.*;
import de.bsvrz.ars.ars.persistence.directories.ActivePersistenceDirectory;
import de.bsvrz.ars.ars.persistence.directories.mgmt.lock.LockFileManager;
import de.bsvrz.ars.ars.persistence.directories.mgmt.range.Week;
import de.bsvrz.ars.ars.persistence.directories.mgmt.range.WeekDomain;
import de.bsvrz.ars.ars.persistence.index.IndexException;
import de.bsvrz.ars.ars.persistence.layout.ShortPersistenceDirectoryLayout;
import de.bsvrz.ars.ars.persistence.walk.internal.StatusPrinter;
import de.bsvrz.ars.ars.persistence.writer.SerializableDataset;
import de.bsvrz.ars.ars.persistence.writer.SerializationHelper;
import de.bsvrz.ars.persistence.Container;
import de.bsvrz.dav.daf.main.DataState;
import de.bsvrz.dav.daf.main.archive.ArchiveDataKind;
import de.bsvrz.dav.daf.main.archive.ArchiveDataKindCombination;
import de.bsvrz.dav.daf.main.impl.archive.*;
import de.bsvrz.sys.funclib.debug.Debug;
import de.bsvrz.sys.funclib.kappich.annotations.NotNull;
import de.bsvrz.sys.funclib.losb.datk.ContainerSettings;

import java.io.*;
import java.nio.file.DirectoryStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.time.Duration;
import java.time.Instant;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicLong;
import java.util.stream.Stream;
import java.util.zip.DataFormatException;
import java.util.zip.Deflater;
import java.util.zip.Inflater;

import static de.bsvrz.ars.ars.persistence.StartupProperties.STUP_LAST_ATIME;
import static de.bsvrz.ars.ars.persistence.StartupProperties.STUP_MAX_CONT_ID;

/**
 * Diese Klasse führt die eigentliche Migration durch
 */
public class MigrateWorker implements ContainerCreator {

    private static final Debug _debug = Debug.getLogger();

    private final PersistentContainerStreamSupplier src;
    private final Path target;

    private final AtomicLong containerIdCounter = new AtomicLong();
    private final AtomicLong lastATime = new AtomicLong();
    private final WeekDomain domain;
    private final ConcurrentHashMap<Path, ActivePersistenceDirectory> pathCache = new ConcurrentHashMap<>();
    private final ContainerSettings.CloseCondition closeConditions;
    private final ThreadLocal<SerializationHelper> serializationHelper;
    private final int numThreads;
    private final Range<Instant> archiveTimeRange;
    private final LockFileManager lockFileManager;
    private final Map<IdContainerFileDir, DataIdentificationMigration> didWorkers = new HashMap<>();
    private final boolean closeIndexes;

    /**
     * Erstellt einen neuen MigrateWorker.
     *
     * @param streamSupplier   Alte Containerdateien
     * @param target           Zielverzeichnis
     * @param numThreads       Anzahl Threads für Migration
     * @param archiveTimeRange Archivzeitbereich, der Migriert werden soll
     */
    public MigrateWorker(PersistentContainerStreamSupplier streamSupplier, Path target, int numThreads, Range<Instant> archiveTimeRange, boolean closeIndexes) {
        this.src = streamSupplier;
        this.target = target;
        this.numThreads = numThreads;
        this.archiveTimeRange = archiveTimeRange;
        this.closeIndexes = closeIndexes;
        this.domain = new WeekDomain();
        lockFileManager = new LockFileManager();

        CacheManager.getInstance().setCacheEnabled(false);

        if (Files.exists(target) && !isEmpty(target)) {
            throw new IllegalArgumentException("Zielverzeichnis existiert bereits: " + target);
        }

        closeConditions = new ContainerSettings.CloseCondition();
        closeConditions.maxContTime = 1_000_000;
        closeConditions.maxContSize = 50_000_000;
        closeConditions.maxContAnzDS = 100_000;

        serializationHelper = ThreadLocal.withInitial(() -> new SerializationHelper(atg -> closeConditions, MigrateWorker.this, false));
    }

    /**
     * Prüft, ob ein Verzeichnis leer ist
     *
     * @param path Verzeichnis
     * @return true: leer, false: kein Verzeichnis oder nicht leer oder anderer Fehler
     */
    public boolean isEmpty(Path path) {
        if (Files.isDirectory(path)) {
            try (DirectoryStream<Path> stream = Files.newDirectoryStream(path)) {
                return !stream.iterator().hasNext();
            } catch (IOException e) {
                return false;
            }
        }
        return false;
    }

    /**
     * Führt die Migration durch.
     *
     * @throws PersistenceException Fehler beim Lesen oder Schreiben
     */
    public void start() throws PersistenceException {
        migrateData();
        closeDirectories();
        writeStartupProperties();
    }

    private void migrateData() {
        while (true) {
            try {
                Container container = (Container) src.fetchNextContainer();
                if (container == null) {
                    break;
                }
                IdContainerFileDir containerFileDir = container.getContainerIdentification();
                didWorkers.computeIfAbsent(containerFileDir, DataIdentificationMigration::new).handleContainer(container);
            } catch (Exception e) {
                _debug.error("Ein Container konnte nicht gelesen werden und wird ignoriert", e);
            }
        }
    }

    /**
     * Schreibt die Startup-Properties.info
     *
     * @throws PersistenceException
     */
    private void writeStartupProperties() throws PersistenceException {
        StartupProperties startupProperties = new StartupProperties(target);
        startupProperties.setVal(STUP_MAX_CONT_ID, containerIdCounter.get());
        startupProperties.setVal(STUP_LAST_ATIME, lastATime.get());
    }

    /**
     * Schließt alle Wochenverzeichnisse ab (d.h. die letzte Containerdatei wird geschlossen und die Indexe werden generiert)
     *
     * @throws PersistenceException
     */
    private void closeDirectories() throws PersistenceException {
        List<ActivePersistenceDirectory> directories = new ArrayList<>(pathCache.values());
        directories.sort(Comparator.comparing(it -> domain.ofPath(target.relativize(it.getBasePath()))));
        int count = directories.size();
        _debug.info("Schließe " + count + " Verzeichnisse ab.");
        Instant start = Instant.now();
        int finished = 0;
        StatusPrinter statusPrinter = new StatusPrinter();
        for (ActivePersistenceDirectory value : directories) {
            _debug.info(statusPrinter.getStatusMessage("Verzeichnisse abschließen", Duration.between(start, Instant.now()), StatusPrinter.ApproximationType.Exact, count, finished, 0) + "\nAktuelles Verzeichnis: " + value);
            try {
                if (this.closeIndexes) {
                    RestorePersDirTsk.RestoreWorker worker = new RestorePersDirTsk.RestoreWorker("Indexe erzeugen", RebuildMode.Full, this, numThreads);
                    worker.doRestore(value);
                }
                value.closePermanently();
                Path basePath = value.getBasePath();
                lockFileManager.close(basePath);
                deleteDirectoryIfEmpty(basePath);
            } catch (InterruptedException e) {
                _debug.warning("Unterbrochen", e);
                return;
            } catch (IOException e) {
                _debug.warning("Lock-Datei konnte nicht gelöscht werden", e);
            } finally {
                value.getIndexTree().closeIndexes();
            }
            finished++;
        }
    }

    /**
     * Löscht ein Verzeichnis, wenn es leer ist und gelöscht werden kann.
     * Fehler beim Löschen werden ignoriert.
     *
     * @param path Verzeichnis
     */
    private static void deleteDirectoryIfEmpty(Path path) {
        try {
            try (Stream<Path> stream = Files.list(path)) {
                if (stream.iterator().hasNext()) {
                    return;
                }
            }
            Files.deleteIfExists(path);
        } catch (IOException ignored) {
        }
    }

    /**
     * Erzeugt ein Dummy-Lock-Objekt für den synchronisierten Zugriff auf Datenidentifikationen.
     * Da kein gleichzeitiger Zugriff in der Migration stattfindet, braucht hier auch nichts gelockt zu werden.
     * <p>
     * Das Schließen der Indexdateien des aktuelle nThreads nach Zugriff ist jedoch erforderlich.
     *
     * @param dataIdentification Datenidentifikation
     * @return Lock-Objekt
     */
    private SyncKey<IdDataIdentification> getLock(IdDataIdentification dataIdentification) {
        return new MigrationKey(dataIdentification, key -> {
            for (ActivePersistenceDirectory value : pathCache.values()) {
                value.getIndexTree().closeIndexes();
            }
        });
    }

    /**
     * Gibt die nächste freie Container-ID zurück
     *
     * @return Container-ID
     */
    @Override
    public long nextContainerID() {
        return containerIdCounter.incrementAndGet();
    }

    @Override
    public SyncKey<IdDataIdentification> lockIndex(IdDataIdentification dataIdentification) {
        return getLock(dataIdentification);
    }

    @Override
    public DataIdentTree getDataIdentTree() {
        return null;
    }

    /**
     * Worker-Klasse für die Migration einer einzelnen Datenidentifikation
     */
    private class DataIdentificationMigration {
        private final IdContainerFileDir containerFileDir;
        private ActivePersistenceDirectory prevDirectory;
        private Week prevTimeRange;


        final Deflater deflater = new Deflater();
        final Inflater inflater = new Inflater();

        /**
         * Puffer zum Schreiben von komprimierten Daten
         */
        final byte[] writeBuf = new byte[ContainerFile.BUFFER_SIZE];

        /**
         * Liefert den übergebenen Puffer, falls dessen Größe ausreichend ist, andernfalls einen neu angelegten.
         * Zweck: sowenig Arbeit für die Speicherverwaltung wie möglich.
         *
         * @param defaultBuffer Vorhandener Puffer
         * @param desiredSize   Gewünschte PufferGröße
         * @return Byte-Array-Puffer
         */
        static byte[] getBuf(byte[] defaultBuffer, int desiredSize) {
            return (desiredSize <= defaultBuffer.length) ? defaultBuffer : new byte[desiredSize];
        }

        /**
         * Konstruktor.
         *
         * @param containerFileDir Datenidentifikation, die migriert werden soll.
         */
        public DataIdentificationMigration(IdContainerFileDir containerFileDir) {
            this.containerFileDir = containerFileDir;
        }

        /**
         * Führt die Migration aus
         */
        public void handleContainer(Container container) throws Exception {
            if (containerFileDir.getSimVariant() != 0) {
                // Simulationen werden nicht migriert
                return;
            }

            SyncKey<IdDataIdentification> lock = getLock(containerFileDir.dataIdentification());

            ArchiveDataKind adk = containerFileDir.archiveDataKind();
            LockedContainerDirectory containerDirectory = new LockedContainerDirectory(lock, adk);

            // Container migrieren
            handleContainer(containerDirectory, container);
        }

        private void handleContainer(LockedContainerDirectory directory, Container container) throws Exception {
            ContainerDataResult tmp = new ContainerDataResult();

            try (var supplier = container.getSupplier()) {
                while (true) {
                    PersistentData data = supplier.fetchNextData();
                    if (data == null) {
                        break;
                    }
                    migrateSingleData(data, tmp);
                    if (!archiveTimeRange.contains(Instant.ofEpochMilli(tmp.getArchiveTime()))) {
                        // Datensatz wird ignoriert, nicht im Zeitbereich
                        return;
                    }
                    writeToOutput(tmp, directory, updateOutDir(tmp.getArchiveTime()));
                }
            }
        }

        private void migrateSingleData(PersistentData data, ContainerDataResult result) throws DataFormatException {
            PersistentContainerData containerData = data.getData();
            byte[] byteData = containerData.getDataBytes();
            result.setDataState(containerData.getDataType());
            result.setDataKind(containerFileDir.archiveDataKind());
            DataTiming timing = containerData.getTiming();
            result.setDataIndex(timing.getDataIndex());
            result.setDataTime(timing.getDataTime());
            result.setArchiveTime(timing.getArchiveTime());
            result.setData(null);
            if (byteData != null) {
                byte[] dest = getBuf(writeBuf, byteData.length);

                if (containerData.getCompression() == ArchiveDataCompression.ZIP) {
                    inflater.reset();
                    inflater.setInput(byteData);

                    int uncompressedSize = inflater.inflate(dest);

                    if (inflater.finished()) {
                        // Data fits into the 'dest' buffer
                        byteData = new byte[uncompressedSize]; // Trim padding bytes
                        System.arraycopy(dest, 0, byteData, 0, uncompressedSize);
                    } else {
                        // Data didn't fit, switch to dynamically allocated buffer
                        ByteArrayOutputStream unzippedData = new ByteArrayOutputStream(byteData.length * 2);

                        // Write the already uncompressed portion of the data
                        unzippedData.write(dest, 0, uncompressedSize);

                        byte[] buffer = new byte[1024];
                        while (!inflater.finished()) {
                            int count = inflater.inflate(buffer);
                            unzippedData.write(buffer, 0, count);
                        }
                        byteData = unzippedData.toByteArray();
                    }
                } else if (containerData.getCompression() != ArchiveDataCompression.NONE) {
                    // Der Datensatz wurde mit einer unbekannte Version gepackt
                    throw new RuntimeException("Entpacken von Datensätzen nicht möglich, da die Version des Packers nicht unterstützt wird");
                }

                if (byteData.length > ContainerFile.MAX_UNCOMPRESSED) {
                    deflater.reset();
                    deflater.setInput(byteData);
                    deflater.finish();
                    int compressedDataLength = deflater.deflate(dest);

                    if (compressedDataLength < byteData.length && deflater.finished()) {
                        // Komprimieren hat was gebracht 
                        result.setData(Arrays.copyOf(dest, compressedDataLength));
                        result.setDataUncompressedSize(byteData.length);
                        result.setDataSize(compressedDataLength);
                        result.setCompressed(true);
                        return;
                    }
                }

                result.setData(byteData);
                result.setDataSize(byteData.length);
                result.setDataUncompressedSize(ContainerFile.NOT_COMPRESSED);
                result.setCompressed(false);
            }
        }

        private void writeToOutput(ContainerDataResult result, ContainerDirectory location, ActivePersistenceDirectory persistenceDirectory) throws IndexException, SynchronizationFailedException, PersistenceException {

            lastATime.accumulateAndGet(result.getArchiveTime(), Math::max);

            SerializableDataset dataset = SerializableDataset.create(result);
            serializationHelper.get().writeData(dataset, persistenceDirectory, location);
        }

        private ActivePersistenceDirectory updateOutDir(long archiveTime) throws PersistenceException {
            Week timeRange = domain.ofEpochMillis(archiveTime);
            if (timeRange.equals(prevTimeRange)) {
                return prevDirectory;
            }
            if (prevDirectory != null) {
                for (ArchiveDataKind archiveDataKind : ArchiveDataKindCombination.all()) {
                    LockedContainerDirectory containerDirectory = new LockedContainerDirectory(getLock(this.containerFileDir.dataIdentification()), archiveDataKind);
                    closeOpenContainer(containerDirectory);
                }
            }
            prevTimeRange = timeRange;
            Path targetDir = target.resolve(domain.getPath(timeRange));
            prevDirectory = pathCache.computeIfAbsent(targetDir, path -> createPersistenceDirectory(targetDir));
            return prevDirectory;
        }

        private void closeOpenContainer(LockedContainerDirectory containerDirectory) throws PersistenceException {
            // Hier nicht ActivePersistenceDirectory#closeOpenContainer benutzen, da die Indexdateien noch nicht erstellt werden sollen
            OpenContainerData openContainerData = prevDirectory.getLoadedContainerData(containerDirectory);

            if (openContainerData instanceof StandardOpenContainerData containerData) {
                try (var containerFileHandle = prevDirectory.accessOpenContainer(containerDirectory, containerData.getContainerId())) {
                    ContainerFile cf = containerFileHandle.getContainerFile();
                    cf.closeContainer(containerData);
                    prevDirectory.removeOpenContainerData(containerDirectory);
                }
            }
        }
    }


    @NotNull
    private ActivePersistenceDirectory createPersistenceDirectory(Path targetDir) throws UncheckedIOException {
        try {
            lockFileManager.createWritable(targetDir);
            return new ActivePersistenceDirectory(MigrateWorker.this, ShortPersistenceDirectoryLayout.Instance.createInstance(targetDir, 0));
        } catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }


}
