/*
 * Copyright 2005 by Kappich+Kniß Systemberatung Aachen (K2S)
 * Copyright 2020-2023 by DTV-Verkehrsconsult, Aachen
 *
 * ALL RIGHTS RESERVED.
 *
 * THIS  SOFTWARE  IS  PROVIDED  "AS IS"  AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES,  INCLUDING,  BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF  MERCHANTABILITY  AND  FITNESS  FOR  A  PARTICULAR  PURPOSE  ARE
 * DISCLAIMED. IN NO EVENT SHALL DTV-VERKEHRSCONSULT GmbH, OR ITS CON-
 * TRIBUTORS BE LIABLE FOR ANY  DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY,  OR CONSEQUENTIAL  DAMAGES  (INCL.,  BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;  LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCL.
 * NEGLIGENCE  OR OTHERWISE)  ARISING  IN  ANY  WAY OUT  OF THE USE OF
 * THIS  SOFTWARE,  EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */
package de.bsvrz.ars.persistence;

import de.bsvrz.dav.daf.main.impl.archive.*;
import de.bsvrz.dav.daf.main.impl.archive.filesaver.ArchiveFileSaver;
import de.bsvrz.sys.funclib.commandLineArgs.ArgumentList;
import de.bsvrz.sys.funclib.debug.Debug;

import java.io.*;
import java.nio.file.Path;

/**
 * Implementierung eines Persistenzmoduls des Archivsystems, das die Container in entsprechenden Dateien im Dateisystem
 * speichert. Dieses Persistenzmodul kann beim Start des Archivsystems mit dem Aufrufparameter <code>
 * -persistenzModul=de.kappich.ars.archive.persistence.FileSystemPersistenceModule </code> in das Archivsystem eingebunden werden.
 * Als zusätzliche Aufrufparameter ist für dieses Persistenzmodul ist die Angabe des Wurzelverzeichnis im lokalen
 * Dateisystem mittels <code> -persistenzVerzeichnis=wurzelverzeichnis </code> erforderlich. Unter diesem Verzeichnis
 * werden die Container-Dateien gespeichert bzw. gesucht. Ein weiterer Aufrufparameter ist die zu nutzende Sicherung
 * {@link ArchiveFileSaver}, diese wird mit <code>-sicherungsModul=Klassenname</code> angegeben.
 * Der letzte Aufrufparameter gibt an, ob die zu archivierenden Datensätze gepackt archiviert werden sollen. Der
 * Aufrufparameter lautet <code>-packer=...</code> und ist per default auf "keiner" eingestellt, weitere Parameter sind
 * <code>zip</code> und <code>alle</code>. Der Parameter <code>-packer=keiner</code> bedeutet, dass die Datensätze nicht
 * gepackt werden, der Parameter <code>-packer=zip</code> benutzt den Packer ZIP, der Parameter
 * <code>-packer=alle</code> benutzt alle Packer, die zur Verfügung stehen und sucht denn besten Packer heraus. Der
 * Aufrufparameter <code>-packer=alle</code> ist nicht implementiert und darf nicht benutzt werden, ist aber für die
 * Zukunft gedacht.
 *
 * @author Kappich+Kniß Systemberatung Aachen (K2S)
 * @author Roland Schmitz (rs)
 * @version $Revision$ / $Date$ / ($Author$)
 */
public class FileSystemPersistenceModule {
    /**
     * Wurzelverzeichnis zur Ablage von Containerdateien
     */
    private final Path _rootFolder;

    /**
     * Konstruktor des Persistenzmoduls, der das Wurzelverzeichnis für die Containerdateien festlegt.
     *
     * @param rootFolder Wurzelverzeichnis für die Containerdateien
     */
    public FileSystemPersistenceModule(Path rootFolder) {
        _rootFolder = rootFolder;
    }

    /**
     * Diese Methode lädt einen Container von einem Speichermedium Typ A und stellt danach das Objekt zur Verfügung.
     *
     * @param containerId Eindeutige Containeridentifikation des benötigten Containers
     * @return Archivdatensatzcontainer
     * @throws IOException              Fehler beim Zugriff auf die Datei, die den Container speichert
     * @throws IllegalStateException    Der Container wurde mit einem K2S Modul erzeugt, konnte aber nicht zurückgelesen
     *                                  werden, da die Version des Containers unbekannt ist
     * @throws IllegalArgumentException Der angegebene Container kann nicht gelesen werden, da er nicht mit einem K2S Modul
     *                                  erzeugt wurde
     */
    public Container getContainer(long containerId) throws IOException, IllegalStateException, IllegalArgumentException {
        return new Container(containerId, _rootFolder);
    }

    public PersistentContainerStreamSupplier getAllContainers() {
        return new PersistentContainerSupplier(_rootFolder, this);
    }

}
